"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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