import logging
import requests

from retry import retry

from ..exceptions import (
    APICaptchaNotReady,
    APICaptchaUnsolvable,
    ParamsException,
    LowBalanceException,
)

API = {"captcha.guru": "http://api.captcha.guru", "2captcha": "http://2captcha.com"}


class CaptchaGuru2Captcha(object):
    """
    API Interface for captcha.guru and 2captcha.com captcha solving providers

    http://learn.captcha.guru/#/
    https://2captcha.com/2captcha-api
    """

    def __init__(self, service_provider_name, service_provider_key):
        """
        Check account money balance before initialization

        :param str service_provider_name: 'captcha.guru' or '2captcha'
        :param str service_provider_key: Valid captcha.guru or 2captcha api key
        """
        self.url = API[service_provider_name]
        self.api_key = service_provider_key
        self.api_request_id = None

        self.check_balance()

    def check_balance(self):
        """
        Check account money balance

        :raises ParamsException: if incorrect 'api_key' provided
        :raises LowBalanceException: if money account balance is low
        """
        response = requests.get("{}/res.php?action=getbalance&key={}&json=1".format(self.url, self.api_key))
        json = response.json()
        logging.info(json)

        if json["request"] == "ERROR_WRONG_USER_KEY" or json["request"] == "ERROR_KEY_DOES_NOT_EXIST":
            raise ParamsException("Incorrect api_key provided. Cannot get data!")
        if float(json["request"]) <= 10.00:
            raise LowBalanceException(
                "Account money balance is very low: {}! Put something there".format(json["request"])
            )

    def get_in(self, payload):
        """
        Send GET to in.php endpoint. Obtain request id that should be used in get_res

        :param str payload: valid params for GET in.php request
        """
        logging.info(payload)
        response = requests.get("{}/in.php?key={}&{}&json=1".format(self.url, self.api_key, payload))
        json = response.json()
        logging.info(json)
        self.api_request_id = json["request"]

    def post_in(self, post_body):
        """
        Send POST to in.php endpoint. Obtain request id that should be used in get_res

        :param str post_body: base64 encoded image
        """
        data = {
            "key": self.api_key,
            "method": "base64",
            "body": post_body,
            "json": 1,
        }
        response = requests.post("{}/in.php".format(self.url), data)
        json = response.json()
        logging.info(json)
        self.api_request_id = json["request"]

    @retry(APICaptchaNotReady, delay=3, tries=30)
    def get_res(self):
        """
        Send GET to res.php endpoint. Return token for request id

        :return: captcha token
        :raises APICaptchaNotReady: if CAPCHA_NOT_READY response
        :raises APICaptchaUnsolvable: if ERROR_CAPTCHA_UNSOLVABLE response
        """
        response = requests.get(
            "{}/res.php?key={}&action=get&id={}&json=1".format(self.url, self.api_key, self.api_request_id)
        )
        json = response.json()
        logging.info(json)
        if json["request"] == "CAPCHA_NOT_READY":
            raise APICaptchaNotReady()

        if json["request"] == "ERROR_CAPTCHA_UNSOLVABLE":
            raise APICaptchaUnsolvable()

        if json["status"] == 1:
            return json["request"]
