# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['serdataclasses']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'serdataclasses',
    'version': '0.8.0',
    'description': 'Serialize/deserialize Python objects from/to typed structures.',
    'long_description': '# serdataclasses\n\n[![image-version](https://img.shields.io/pypi/v/serdataclasses.svg)](https://python.org/pypi/serdataclasses)\n[![image-license](https://img.shields.io/pypi/l/serdataclasses.svg)](https://python.org/pypi/serdataclasses)\n[![image](https://img.shields.io/pypi/pyversions/serdataclasses.svg)](https://python.org/pypi/serdataclasses)\n[![image-ci](https://github.com/pappasam/serdataclasses/workflows/serdataclasses%20ci/badge.svg)](https://github.com/pappasam/serdataclasses/actions?query=workflow%3A%22serdataclasses+ci%22)\n\nSerialize (`serdataclasses.dump`) from and deserialize (`serdataclasses.load`) to strongly-typed, native Python data structures.\n\n## Motivation\n\nNo typing-focused [serde](https://en.wikipedia.org/wiki/Serialization) library in Python satisfies me. Call me needy, but when I translate between loosely-typed data structures like `list` and `dict` into strongly-typed data structures like `NamedTuple` and `dataclasses.dataclass`, I want the following capabilities:\n\n1. Effortlessl deserialization of unstructured Python types into structured, type-hinted Python types (`dataclasses.dataclass`, `typing.NamedTuple`)\n2. Effortless serialization of structured, type-hinted Python objects into unstructured Python types (eg, the reverse)\n3. Clear error messages when serde fails at runtime\n4. No inherited, non-standard types. dataclasses, NamedTuples, and other standard Python types are bread and butter\n5. Editor support: I like my autocompletion, so I jump through lots of hoops to make this library compatible with Jedi\n6. Handle [optional properties](https://www.typescriptlang.org/docs/handbook/interfaces.html#optional-properties) with a domain-specific `serdataclasses.OptionalProperty`\n7. Provide option to automatically convert primitive types, but avoid converting ambiguous types (`Union`, `TypeVar`, etc). Handle of `Optional` and `serdataclasses.OptionalProperty`\n8. Require no 3rd party dependencies; Python 3.8+\n\n## Installation\n\n```bash\n# With pip\npip install serdataclasses\n\n# With poetry\npoetry add serdataclasses\n```\n\n## Usage\n\nSee [examples folder](example) if you\'d like to get your hands dirty. Otherwise, keep reading for a complete, real-world example.\n\n### Example: Libraries and Books\n\nAssume that, from an external API, you receive a `JSON` list of libraries containing the library name and a list of books that each library currently has.\n\n```json\n[\n  {\n    "name": "Clark County Library",\n    "books": [\n      {\n        "title": "Hello, World!",\n        "author": "Susy Smith",\n        "year": 1929,\n        "tags": ["boring"]\n      },\n      {\n        "title": "The great showman",\n        "author": "Beth John"\n      },\n      {\n        "title": "My favorite pony",\n        "author": null\n      }\n    ]\n  },\n  {\n    "name": "Only 1 book here",\n    "books": [\n      {\n        "title": "The great fun time",\n        "author": "Smitty",\n        "year": 1950,\n        "tags": ["swell"]\n      }\n    ]\n  }\n]\n```\n\nNow you want to ingest this document into Python. Your first step is probably to deserialize the JSON string (or file) into Python data structures. Assuming the JSON is read from a file called `libraries.py`, the following script will print the following:\n\n```python\nimport json\nfrom pprint import pprint\n\nwith open("libraries.json", "r") as infile:\n    libraries_raw = json.load(infile)\n\npprint(libraries_raw)\nprint(type(libraries_raw))\nprint(type(libraries_raw[0]))\n```\n\n```text\n[{\'books\': [{\'author\': \'Susy Smith\',\n             \'tags\': [\'boring\'],\n             \'title\': \'Hello, World!\',\n             \'year\': 1929},\n            {\'author\': \'Beth John\', \'title\': \'The great showman\'},\n            {\'author\': None, \'title\': \'My favorite pony\'}],\n  \'name\': \'Clark County Library\'},\n {\'books\': [{\'author\': \'Smitty\',\n             \'tags\': [\'swell\'],\n             \'title\': \'The great fun time\',\n             \'year\': 1950}],\n  \'name\': \'Only 1 book here\'}]\n<class \'list\'>\n<class \'dict\'>\n```\n\nSome observations:\n\n1. Python\'s native `json` module deserializes the JSON string / document into Python\'s primitive (or primitive-like) types\n2. `null` is translated to Python\'s `None`\n3. The first list element is a dictionary. So Python appears to have translated the JSON into a list of dictionaries.\n4. There is little inherent structure to the Python objects deserialized by the JSON module. By this, I mean that we have no way of knowing whether the dictionaries contain keys that we expect or are structured improperly. Should books also have an `"isbn"` field? Does code we write that uses `libraries_raw` expect an `"isbn"` field? What happens if there are missing tags?\n\nThe first 3 items are merely facts; `serdataclasses` accepts these facts and builds on them. The 4th item in this list is THE problem that `serdataclasses` is designed to solve. If we take the above Python dictionary associate it with a Python variable `LIBRARIES`, we can define a container for use with `serdataclasses` to ingest `LIBRARIES` into a strongly-typed Python container.\n\n```python\nfrom dataclasses import dataclass, field\nfrom pprint import pprint\nfrom typing import List, Optional\n\nimport serdataclasses\nfrom serdataclasses import OptionalProperty\n\n[\n    {\n        "books": [\n            {\n                "author": "Susy Smith",\n                "tags": ["boring"],\n                "title": "Hello, World!",\n                "year": 1929,\n            },\n            {"author": "Beth John", "title": "The great showman"},\n            {"author": None, "title": "My favorite pony"},\n        ],\n        "name": "Clark County Library",\n    },\n    {\n        "books": [\n            {\n                "author": "Smitty",\n                "tags": ["swell"],\n                "title": "The great fun time",\n                "year": 1950,\n            }\n        ],\n        "name": "Only 1 book here",\n    },\n]\n\n@dataclass\nclass Book:\n    author: Optional[str]\n    title: str\n    year: OptionalProperty[int]\n    tags: List[str] = field(default_factory=list)\n\n@dataclass\nclass Library:\n    books: List[Book]\n    name: str\n\nLIBRARIES_LOADED = serdataclasses.load(LIBRARIES, List[Library])\nprint(LIBRARIES_LOADED[0].name)\nprint(LIBRARIES_LOADED[0].books[1].author)\npprint(serdataclasses.dump(LIBRARIES_LOADED))\n```\n\nRunning the above script, we get the following output to the terminal:\n\n```text\n[{\'books\': [{\'author\': \'Susy Smith\',\n             \'tags\': [\'boring\'],\n             \'title\': \'Hello, World!\',\n             \'year\': 1929},\n            {\'author\': \'Beth John\', \'tags\': [], \'title\': \'The great showman\'},\n            {\'author\': None, \'tags\': [], \'title\': \'My favorite pony\'}],\n  \'name\': \'Clark County Library\'},\n {\'books\': [{\'author\': \'Smitty\',\n             \'tags\': [\'swell\'],\n             \'title\': \'The great fun time\',\n             \'year\': 1950}],\n  \'name\': \'Only 1 book here\'}]\n```\n\nNotice how we have the following features:\n\n1. Data structures are loaded, recursively, without you needing to write anything more than a couple standard Python classes.\n2. For missing properties / dictionary keys (for example, `Book.tags`), we can set a default value in our dataclass using standard Python and `serdataclasses` adds the default value to our structure\n3. For missing properties without default values, serdataclasses intelligently omits them when re-serializing the result. There is also an option to `serdataclasses.load` that allows you to convert missing values to `None` and keep the keys in the output. For all other desired default values, just use `dataclasses.field`; no need to re-invent the wheel!\n\nWhat about additional validation, you may ask? Again, just use dataclasses! Assume that, for some reason, no book can possibly be published before 1930, and that a book published before 1930 invalidates all data. No problem, just use the standard method `__post_init__` on the relevant dataclass!\n\n```python\nfrom dataclasses import dataclass, field\nfrom pprint import pprint\nfrom typing import List, Optional\n\nimport serdataclasses\nfrom serdataclasses import OptionalProperty\n\nLIBRARIES = [\n    {\n        "books": [\n            {\n                "author": "Susy Smith",\n                "tags": ["boring"],\n                "title": "Hello, World!",\n                "year": 1929,\n            },\n            {"author": "Beth John", "title": "The great showman"},\n            {"author": None, "title": "My favorite pony"},\n        ],\n        "name": "Clark County Library",\n    },\n    {\n        "books": [\n            {\n                "author": "Smitty",\n                "tags": ["swell"],\n                "title": "The great fun time",\n                "year": 1950,\n            }\n        ],\n        "name": "Only 1 book here",\n    },\n]\n\n@dataclass\nclass Book:\n    author: Optional[str]\n    title: str\n    year: OptionalProperty[int]\n    tags: List[str] = field(default_factory=list)\n\n    def __post_init__(self) -> None:\n        if self.year and self.year < 1930:\n            raise ValueError(\n                f"Received illegal year {self.year}, cannot be before 1930"\n            )\n\n@dataclass\nclass Library:\n    books: List[Book]\n    name: str\n\nLIBRARIES_LOADED = serdataclasses.load(LIBRARIES, List[Library])\n```\n\nRunning this script should give you the following error message (traceback omitted because it can get somewhat long)\n\n```text\nserdataclasses.errors.DeserializeError: Received illegal year 1929, cannot be before 1930\n  4. "<class \'__main__.Book\'>": "{\'author\': \'Susy Smith\', \'tags\': [\'boring\'], \'title\': \'Hello, World!\', \'year\': 1929}"\n  3. \'typing.List[__main__.Book]\': "[{\'author\': \'Susy Smith\', \'tags\': [\'boring\'], \'title\': \'Hello, World!\', \'year\': 1929}, {\'author\': \'Beth John\', \'title\': \'The great showman\'}, {\'author\': None, \'title\': \'My favorite pony\'}]"\n  2. "<class \'__main__.Library\'>": "{\'books\': [{\'author\': \'Susy Smith\', \'tags\': [\'boring\'], \'title\': \'Hello, World!\', \'year\': 1929}, {\'author\': \'Beth John\', \'title\': \'The great showman\'}, {\'author\': None, \'title\': \'My favorite pony\'}], \'name\': \'Clark County Library\'}"\n  1. \'typing.List[__main__.Library]\': "[{\'books\': [{\'author\': \'Susy Smith\', \'tags\': [\'boring\'], \'title\': \'Hello, World!\', \'year\': 1929}, {\'author\': \'Beth John\', \'title\': \'The great showman\'}, {\'author\': None, \'title\': \'My favorite pony\'}], \'name\': \'Clark County Library\'}, {\'books\': [{\'author\': \'Smitty\', \'tags\': [\'swell\'], \'title\': \'The great fun time\', \'year\': 1950}], \'name\': \'Only 1 book here\'}]"\n```\n\nThe error message begins with the error message received, followed by a reverse presention container types paired with the data they attempted to deserialize. This structure makes it incredibly easy to see not only what your error is, but where exactly it occured in both the data `serdataclasses.load` receives but also in the types `serdataclasses.load` uses to attempt to deserialize the received data.\n\nIn serde, when working with resources external to your system, errors are inevitable. These error messages should hopefully make debugging your errors less annoying.\n\n## Local Development\n\nLocal development for this project is simple.\n\n**Dependencies**\n\nInstall the following tools manually.\n\n- [Poetry](https://github.com/sdispater/poetry#installation)\n- [GNU Make](https://www.gnu.org/software/make/)\n\n_Recommended_\n\n- [asdf](https://github.com/asdf-vm/asdf)\n\n**Set up development environment**\n\n```bash\nmake setup\n```\n\n**Run Tests**\n\n```bash\nmake test\n```\n\n## Notes\n\n- Initially inspired by [undictify](https://github.com/Dobiasd/undictify) and a PR I helped with. serdataclasses\'s goals are different; it\'s focused on serde instead of general function signature overrides.\n- I also notice some striking similarities with a library called [typedload](https://github.com/ltworf/typedload) (great minds think alike, I guess :p). I renamed my top-level functions to "load" and "dump" in typedload\'s homage. Unfortunately, as of version `1.20`, typedload does not handle all types of dataclasses elegantly (mainly, InitVar). Since typedload supports Python 3.5+, it never will elegantly handle all dataclasses without lots of unfortunate conditionals in the codebase. If you must use Python 3.7-, I suggest looking into typedload.\n\n## Written by\n\nSamuel Roeca *samuel.roeca@gmail.com*\n',
    'author': 'Sam Roeca',
    'author_email': 'samuel.roeca@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pappasam/serdataclasses',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
