/*
 * Decompiled with CFR 0.152.
 */
package nl.esciencecenter.xenon.adaptors.schedulers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;
import nl.esciencecenter.xenon.XenonException;
import nl.esciencecenter.xenon.adaptors.XenonProperties;
import nl.esciencecenter.xenon.adaptors.schedulers.BadParameterException;
import nl.esciencecenter.xenon.adaptors.schedulers.InteractiveProcessFactory;
import nl.esciencecenter.xenon.adaptors.schedulers.JobExecutor;
import nl.esciencecenter.xenon.adaptors.schedulers.QueueStatusImplementation;
import nl.esciencecenter.xenon.filesystems.FileSystem;
import nl.esciencecenter.xenon.filesystems.Path;
import nl.esciencecenter.xenon.schedulers.IncompleteJobDescriptionException;
import nl.esciencecenter.xenon.schedulers.InvalidJobDescriptionException;
import nl.esciencecenter.xenon.schedulers.JobDescription;
import nl.esciencecenter.xenon.schedulers.JobStatus;
import nl.esciencecenter.xenon.schedulers.NoSuchJobException;
import nl.esciencecenter.xenon.schedulers.NoSuchQueueException;
import nl.esciencecenter.xenon.schedulers.QueueStatus;
import nl.esciencecenter.xenon.schedulers.Scheduler;
import nl.esciencecenter.xenon.schedulers.Streams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobQueueScheduler
extends Scheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(JobQueueScheduler.class);
    private static final String SINGLE_QUEUE_NAME = "single";
    private static final String MULTI_QUEUE_NAME = "multi";
    private static final String UNLIMITED_QUEUE_NAME = "unlimited";
    public static final int MIN_POLLING_DELAY = 100;
    public static final int MAX_POLLING_DELAY = 60000;
    private final String adaptorName;
    private final FileSystem filesystem;
    private final Path workingDirectory;
    private final List<JobExecutor> singleQ = new LinkedList<JobExecutor>();
    private final List<JobExecutor> multiQ = new LinkedList<JobExecutor>();
    private final List<JobExecutor> unlimitedQ = new LinkedList<JobExecutor>();
    private final ExecutorService singleExecutor;
    private final ExecutorService multiExecutor;
    private final ExecutorService unlimitedExecutor;
    private final long pollingDelay;
    private final long startupTimeout;
    private final InteractiveProcessFactory factory;
    private final AtomicLong jobID = new AtomicLong(0L);
    private final ArrayList<List<JobExecutor>> queues = new ArrayList();

    public JobQueueScheduler(String uniqueID, String adaptorName, String location, InteractiveProcessFactory factory, FileSystem filesystem, Path workingDirectory, int multiQThreads, long pollingDelay, long startupTimeout, XenonProperties properties) throws BadParameterException {
        super(uniqueID, adaptorName, location, properties);
        LOGGER.debug("Creating JobQueueScheduler for Adaptor {} with multiQThreads: {} and pollingDelay: {}", new Object[]{adaptorName, multiQThreads, pollingDelay});
        this.adaptorName = adaptorName;
        this.filesystem = filesystem;
        this.workingDirectory = workingDirectory;
        this.factory = factory;
        this.pollingDelay = pollingDelay;
        this.startupTimeout = startupTimeout;
        this.queues.add(this.singleQ);
        this.queues.add(this.multiQ);
        this.queues.add(this.unlimitedQ);
        if (multiQThreads < 1) {
            throw new BadParameterException(adaptorName, "Number of slots for the multi queue cannot be smaller than one!");
        }
        if (pollingDelay < 100L || pollingDelay > 60000L) {
            throw new BadParameterException(adaptorName, "Polling delay must be between 100 and 60000!");
        }
        DaemonThreadFactory threadFactory = new DaemonThreadFactory();
        this.unlimitedExecutor = Executors.newCachedThreadPool(threadFactory);
        this.singleExecutor = Executors.newSingleThreadExecutor(threadFactory);
        this.multiExecutor = Executors.newFixedThreadPool(multiQThreads, threadFactory);
    }

    public long getCurrentJobID() {
        return this.jobID.get();
    }

    private void getJobs(List<JobExecutor> list, List<String> out) {
        for (JobExecutor e : list) {
            out.add(e.getJobIdentifier());
        }
    }

    @Override
    public String getDefaultQueueName() throws XenonException {
        return SINGLE_QUEUE_NAME;
    }

    @Override
    public String[] getJobs(String ... queueNames) throws NoSuchQueueException {
        LOGGER.debug("{}: getJobs for queues {}", (Object)this.adaptorName, (Object)queueNames);
        LinkedList<String> out = new LinkedList<String>();
        if (queueNames == null || queueNames.length == 0) {
            this.getJobs(this.singleQ, out);
            this.getJobs(this.multiQ, out);
            this.getJobs(this.unlimitedQ, out);
        } else {
            for (String name : queueNames) {
                if (SINGLE_QUEUE_NAME.equals(name)) {
                    this.getJobs(this.singleQ, out);
                    continue;
                }
                if (MULTI_QUEUE_NAME.equals(name)) {
                    this.getJobs(this.multiQ, out);
                    continue;
                }
                if (UNLIMITED_QUEUE_NAME.equals(name)) {
                    this.getJobs(this.unlimitedQ, out);
                    continue;
                }
                throw new NoSuchQueueException(this.adaptorName, "Queue \"" + name + "\" does not exist");
            }
        }
        LOGGER.debug("{}: getJobs for queues {} returns {}", new Object[]{this.adaptorName, queueNames, out});
        return out.toArray(new String[out.size()]);
    }

    private JobExecutor findJob(List<JobExecutor> queue, String jobIdentifier) throws XenonException {
        LOGGER.debug("{}: findJob for job {}", (Object)this.adaptorName, (Object)jobIdentifier);
        for (JobExecutor e : queue) {
            if (!jobIdentifier.equals(e.getJobIdentifier())) continue;
            return e;
        }
        return null;
    }

    private JobExecutor findJob(String jobIdentifier) throws XenonException {
        LOGGER.debug("{}: findJob for job {}", (Object)this.adaptorName, (Object)jobIdentifier);
        this.assertNonNullOrEmpty(jobIdentifier, "Job identifier cannot be null or empty");
        JobExecutor e = null;
        for (List<JobExecutor> queue : this.queues) {
            e = this.findJob(queue, jobIdentifier);
            if (e == null) continue;
            return e;
        }
        throw new NoSuchJobException(this.adaptorName, "Job " + jobIdentifier + " does not exist!");
    }

    private boolean cleanupJob(List<JobExecutor> queue, String jobIdentifier) {
        LOGGER.debug("{}: cleanupJob for job {}", (Object)this.adaptorName, (Object)jobIdentifier);
        Iterator<JobExecutor> itt = queue.iterator();
        while (itt.hasNext()) {
            JobExecutor e = itt.next();
            if (!e.getJobIdentifier().equals(jobIdentifier)) continue;
            itt.remove();
            return true;
        }
        return false;
    }

    private void cleanupJob(String jobIdentifier) {
        for (List<JobExecutor> queue : this.queues) {
            if (!this.cleanupJob(queue, jobIdentifier)) continue;
            return;
        }
    }

    @Override
    public JobStatus getJobStatus(String jobIdentifier) throws XenonException {
        LOGGER.debug("{}: getJobStatus for job {}", (Object)this.adaptorName, (Object)jobIdentifier);
        JobStatus status = this.findJob(jobIdentifier).getStatus();
        if (status.isDone()) {
            this.cleanupJob(jobIdentifier);
        }
        return status;
    }

    @Override
    public JobStatus waitUntilDone(String jobIdentifier, long timeout) throws XenonException {
        LOGGER.debug("{}: Waiting for job {} for {} ms.", new Object[]{this.adaptorName, jobIdentifier, timeout});
        JobExecutor ex = this.findJob(jobIdentifier);
        this.assertPositive(timeout, "Illegal timeout ");
        JobStatus status = ex.waitUntilDone(timeout);
        if (status.isDone()) {
            LOGGER.debug("{}: Job {} is done after {} ms.", new Object[]{this.adaptorName, jobIdentifier, timeout});
            this.cleanupJob(jobIdentifier);
        } else {
            LOGGER.debug("{}: Job {} is NOT done after {} ms.", new Object[]{this.adaptorName, jobIdentifier, timeout});
        }
        return status;
    }

    @Override
    public JobStatus waitUntilRunning(String jobIdentifier, long timeout) throws XenonException {
        LOGGER.debug("{}: Waiting for job {} to start for {} ms.", new Object[]{this.adaptorName, jobIdentifier, timeout});
        JobExecutor ex = this.findJob(jobIdentifier);
        this.assertPositive(timeout, "Illegal timeout ");
        JobStatus status = ex.waitUntilRunning(timeout);
        if (status.isDone()) {
            LOGGER.debug("{}: Job {} is done within {} ms.", new Object[]{this.adaptorName, jobIdentifier, timeout});
            this.cleanupJob(jobIdentifier);
        } else {
            LOGGER.debug("{}: Job {} is NOT done after {} ms.", new Object[]{this.adaptorName, jobIdentifier, timeout});
        }
        return status;
    }

    private void verifyJobDescription(JobDescription description, boolean interactive) throws XenonException {
        String queue = description.getQueueName();
        if (queue == null) {
            queue = SINGLE_QUEUE_NAME;
            description.setQueueName(SINGLE_QUEUE_NAME);
        }
        if (!(SINGLE_QUEUE_NAME.equals(queue) || MULTI_QUEUE_NAME.equals(queue) || UNLIMITED_QUEUE_NAME.equals(queue))) {
            throw new NoSuchQueueException(this.adaptorName, "Queue " + queue + " not available locally!");
        }
        String executable = description.getExecutable();
        if (executable == null) {
            throw new IncompleteJobDescriptionException(this.adaptorName, "Executable missing in JobDescription!");
        }
        int nodeCount = description.getNodeCount();
        if (nodeCount != 1) {
            throw new InvalidJobDescriptionException(this.adaptorName, "Illegal node count: " + nodeCount);
        }
        int processesPerNode = description.getProcessesPerNode();
        if (processesPerNode != 1) {
            throw new InvalidJobDescriptionException(this.adaptorName, "Illegal processes per node count: " + processesPerNode);
        }
        int maxTime = description.getMaxRuntime();
        if (maxTime < 0) {
            throw new InvalidJobDescriptionException(this.adaptorName, "Illegal maximum runtime: " + maxTime);
        }
        if (interactive) {
            if (description.getStdin() != null) {
                throw new InvalidJobDescriptionException(this.adaptorName, "Illegal stdin redirect for interactive job!");
            }
            if (description.getStdout() != null && !description.getStdout().equals("stdout.txt")) {
                throw new InvalidJobDescriptionException(this.adaptorName, "Illegal stdout redirect for interactive job!");
            }
            if (description.getStderr() != null && !description.getStderr().equals("stderr.txt")) {
                throw new InvalidJobDescriptionException(this.adaptorName, "Illegal stderr redirect for interactive job!");
            }
        }
    }

    private JobExecutor submit(JobDescription description, boolean interactive) throws XenonException {
        LOGGER.debug("{}: Submitting job", (Object)this.adaptorName);
        this.verifyJobDescription(description, interactive);
        String jobIdentifier = this.adaptorName + "-" + this.jobID.getAndIncrement();
        LOGGER.debug("{}: Created Job {}", (Object)this.adaptorName, (Object)jobIdentifier);
        JobExecutor executor = new JobExecutor(this.adaptorName, this.filesystem, this.workingDirectory, this.factory, new JobDescription(description), jobIdentifier, interactive, this.pollingDelay, this.startupTimeout);
        String queueName = description.getQueueName();
        LOGGER.debug("{}: Submitting job to queue {}", (Object)this.adaptorName, (Object)queueName);
        if (UNLIMITED_QUEUE_NAME.equals(queueName)) {
            this.unlimitedQ.add(executor);
            this.unlimitedExecutor.execute(executor);
        } else if (MULTI_QUEUE_NAME.equals(queueName)) {
            this.multiQ.add(executor);
            this.multiExecutor.execute(executor);
        } else {
            this.singleQ.add(executor);
            this.singleExecutor.execute(executor);
        }
        return executor;
    }

    @Override
    public String submitBatchJob(JobDescription description) throws XenonException {
        return this.submit(description, false).getJobIdentifier();
    }

    @Override
    public Streams submitInteractiveJob(JobDescription description) throws XenonException {
        JobExecutor executor = this.submit(description, true);
        LOGGER.debug("{}: Waiting for interactive job to start.", (Object)this.adaptorName);
        executor.waitUntilRunning(0L);
        if (executor.isDone() && !executor.hasRun()) {
            this.cleanupJob(executor.getJobIdentifier());
            throw new XenonException(this.adaptorName, "Interactive job failed to start!", executor.getError());
        }
        return executor.getStreams();
    }

    @Override
    public JobStatus cancelJob(String jobIdentifier) throws XenonException {
        LOGGER.debug("{}: Cancel job {}", (Object)this.adaptorName, (Object)jobIdentifier);
        JobExecutor e = this.findJob(jobIdentifier);
        boolean killed = e.kill();
        JobStatus status = killed ? e.getStatus() : e.waitUntilDone(this.pollingDelay);
        if (status.isDone()) {
            this.cleanupJob(jobIdentifier);
        }
        return status;
    }

    @Override
    public QueueStatus getQueueStatus(String queueName) throws XenonException {
        LOGGER.debug("{}: getQueueStatus {}", (Object)this.adaptorName, (Object)queueName);
        if (queueName == null) {
            throw new IllegalArgumentException("Adaptor " + this.adaptorName + ": Queue name is null!");
        }
        if (SINGLE_QUEUE_NAME.equals(queueName)) {
            return new QueueStatusImplementation(this, SINGLE_QUEUE_NAME, null, null);
        }
        if (MULTI_QUEUE_NAME.equals(queueName)) {
            return new QueueStatusImplementation(this, MULTI_QUEUE_NAME, null, null);
        }
        if (UNLIMITED_QUEUE_NAME.equals(queueName)) {
            return new QueueStatusImplementation(this, UNLIMITED_QUEUE_NAME, null, null);
        }
        throw new NoSuchQueueException(this.adaptorName, "No such queue: " + queueName);
    }

    @Override
    public String[] getQueueNames() {
        return new String[]{SINGLE_QUEUE_NAME, MULTI_QUEUE_NAME, UNLIMITED_QUEUE_NAME};
    }

    @Override
    public QueueStatus[] getQueueStatuses(String ... queueNames) throws XenonException {
        String[] names = queueNames;
        if (names == null) {
            throw new IllegalArgumentException("Adaptor " + this.adaptorName + ": Queue names are null!");
        }
        if (names.length == 0) {
            names = new String[]{SINGLE_QUEUE_NAME, MULTI_QUEUE_NAME, UNLIMITED_QUEUE_NAME};
        }
        QueueStatus[] result = new QueueStatus[names.length];
        for (int i = 0; i < names.length; ++i) {
            if (names[i] == null) {
                result[i] = null;
                continue;
            }
            try {
                result[i] = this.getQueueStatus(names[i]);
                continue;
            }
            catch (XenonException e) {
                result[i] = new QueueStatusImplementation(this, names[i], e, null);
            }
        }
        return result;
    }

    public void end() {
        this.singleExecutor.shutdownNow();
        this.multiExecutor.shutdownNow();
        this.unlimitedExecutor.shutdownNow();
    }

    @Override
    public boolean usesFileSystem() {
        return true;
    }

    @Override
    public FileSystem getFileSystem() throws XenonException {
        return this.filesystem;
    }

    @Override
    public void close() throws XenonException {
        this.factory.close();
    }

    @Override
    public boolean isOpen() throws XenonException {
        return this.factory.isOpen();
    }

    private class DaemonThreadFactory
    implements ThreadFactory {
        private DaemonThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = Executors.defaultThreadFactory().newThread(runnable);
            thread.setDaemon(true);
            return thread;
        }
    }
}

