"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNagApp", version: "0.11.8" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNag", version: "0.11.8" };
//# sourceMappingURL=data:application/json;base64,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