# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['simple_django_htmx', 'simple_django_htmx.templatetags']

package_data = \
{'': ['*'], 'simple_django_htmx': ['templates/simple_django_htmx/*']}

install_requires = \
['Django>=3.2']

setup_kwargs = {
    'name': 'simple-django-htmx',
    'version': '1.0.0',
    'description': 'Simple package that facilitates the usage of HTMX with Django',
    'long_description': '# simple-django-htmx\n<br>\n\n## Overview\n\nA package to simplify the usage of HTMX with Django. Easily add HTMX reuqests witout needing additional urls, and reduce clutter in views by offloading all responsibility to an <em>hx_request</em>.\n\n<br>\n\n## Why use?\n\n\n- Avoid cluttering up urls with endpoints for HTMX\n- Reduce clutter in views by avoiding if/elses that are checking if the incoming request is an HTMX request\n- Built in functionality for HTMX requests that handle forms\n\n<br>\n\n## How to use\n\n\n- ### **Installation**\n       pip install simple-django-htmx\n    - Add simple_django_htmx to your list of installed apps\n    - You will also need to have HTMX installed. See https://htmx.org/\n  \n<br>\n\n- ### **Quick Example**\n  \n<br>\n\n1. Create an HXRequest:\n\n        class EditUserHxRequest(FormHXRequest):\n            name = "edit_user"\n            GET_template = "update_user.html"\n            POST_template = "user_row.html"\n            form_class = EditUserForm\n\n2. Create the template where this HXRequest will be used. Notice how the template tag is used to render the HXRequest.\n   <br>\n\n    user_list.html\n\n        {% load hx_tags %}\n        <tr >\n            <td>\n                {{user.name}}\n            </td>\n            <td>\n                {{user.email}}\n            </td>\n            <td>\n                {{user.address}}\n            </td>\n            <td>\n                <button {% render_hx  \'edit_user\' \'get\' object=user %} hx-target="closest tr" hx-swap="outerHTML">Edit</button>\n            </td>  \n        </tr>\n\n3. Load the HXRequest into the view that it\'s being used in. The view needs to inherit from HtmxViewMixin and the view needs to provide a list of hx_requests that will be used in the view.\n   \n         class UserListView(HtmxVIewMixin, ListView):\n            template_name = "user_list.html"\n            model = User\n            hx_requests = [\n               EditUserHxRequest\n            ]\n\n4. Voila! on the click of the Edit button the GET_template from EditUserHxRequest will be loaded.\n   \n   - The view is neat and clean and there was no need to create a url for the edit button to hit.\n  \n\n<br>\n\n## A Deeper Dive\n\n\n### **---HXRequest Classes---**\n\n<br>\n\n### **HXRequest**\n\n---\n\n<br>\n\nA wrapper for HTMX requests. Instead of the GET/POST getting dircted to a view, it goes to an HXRequest. \n\n- **Attributes**\n     - name: Unique string that is used to identify which HXRequest the HTMX reuqest should direct to\n     - GET_template: The template that an <em>hx-get</em> fetches\n     - POST_template: The template an <em>hx-post</em> fetches\n     - hx_object_name: Default is hx_object. If an object is passed into the `render_hx` template tag the obejct is put into the context of the GET/POST templates. The hx_object_name is the key in the contedxt for this object.\n\n<br>\n\n- **Methods**\n    - get_context_data: Same as a view\'s get_context_data. HXRequest\'s add in a few additional items into the context.\n        - hx_kwargs -> These are any kwargs passed into `render_hx` \n        - hx_object (or the name given to it using `hx_object_name`) -> The object passed into `render_hx`. In the `FormHXRequest` this object acts like the object of a Django UpdateView.\n        - hx_request: The HXRequest \n\n<br>\n\n### **FormHXRequest**\n\n---\n\n<br>\n\nActs like a a Django UpdateView. When a form is valid it saves the form and by default returns the POST_template of the HXRequest. If the form is invalid it returns the GET_temlate by default. Can customize what the view returns by overriding `form_valid` or `form_invalid` and return anything from those functions.\n\n- **Attributes**\n  - form_class: This is passed into the context as form and is instantiated with the object passed into `render_hx`. On POST it is instantiated with the request.POST as well.\n\n\n<br>\n\n- **Methods**\n  \n  - form_valid: Method called when the form is valid. By default it just calls `form.save()`. Has access to kwargs sent in through the `render_hx` template tag.\n  - form_invalid: Method called when the form is invalid. No default. Has access to kwargs sent in through the `render_hx` template tag.\n  - get_form_kwargs: Injects kwargs into the form. Can override to put items into the form. i.e. to set initial values. Has access to kwargs sent in through the `render_hx` template tag.\n\n<br>\n\n### **DeleteHXRequest**\n\n---\n\n<br>\n\nDeletes the object passed into `render_hx`. Can override `handle_delete` for custom functionality.\n\n<br>\n\n### **---Other---**\n\n<br>\n\n### **HtmxVIewMixin** -> The mixin intercepts the GET and POST and if it finds hx_request in the GET params it redircts to the HXRequest\'s GET and POST methods.\n\n<br>\n\n### **render_hx** -> Template tag to use for HXReuqests. Takes in:\n- Name of the HXReqest\n- Type of reuqest, get or post\n- object (optional), the object that is used by the request. It is treated just like the object of an UpdateView\n- kwargs, addional params that can be passed in. These kwargs can be used in get_form_kwargs, form_valid, of form_invalid. \n    - Example -> The template has a user and is looping through the contact methods of the user. There is an HXRequest on each contact method that lets you edit the contact method. Would need to pass in the contact method as the object. Addionally the page has an add new button which allows you to add a new contact method to this user, you would need to pass an addional kwarg for the user because when saving the new contact method, the back end needs to know which user we are saving this for.\n\n            <button {% hx_request_name=\'render_hx create_update_contact_method\' method="get" object="contact_method" user=user %} hx-target="closest tr" ></button>\n\n<br>\n\n## Future Features\n\n\n- Aysnchronous messaging.\n- Auto filling initial of form fields with kwargs if the kwarg key matches the form field.',
    'author': 'yaakovLowenstein',
    'author_email': 'lowensteinyaakov@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/yaakovLowenstein/simple-django-htmx',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
