# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['findpapers',
 'findpapers.models',
 'findpapers.searchers',
 'findpapers.tools',
 'findpapers.utils']

package_data = \
{'': ['*']}

install_requires = \
['colorama>=0.4.3,<0.5.0',
 'edlib>=1.3.8,<2.0.0',
 'inquirer>=2.7.0,<3.0.0',
 'lxml>=4.5.2,<5.0.0',
 'requests>=2.24.0,<3.0.0',
 'typer>=0.3.2,<0.4.0',
 'xmltodict>=0.12.0,<0.13.0']

entry_points = \
{'console_scripts': ['findpapers = findpapers.cli:main']}

setup_kwargs = {
    'name': 'findpapers',
    'version': '0.2.0',
    'description': 'Findpapers is an application that helps researchers who are looking for references for their research.',
    'long_description': '# Findpapers\n\nFindpapers is an application that helps researchers who are looking for references for their research. The application will perform searches in several databases (currently ACM, arXiv, IEEE, PubMed, and Scopus) from a user-defined search query.\n\nIn summary, this tool will help you to perform the process below:\n\n![Workflow](docs/workflow.png)\n\n# Requirements\n\n- Python 3.7+\n\n# Installation\n\n```console\n$ pip install findpapers\n```\n\n# How to use it?\n\nAll application actions are command-line based. The available commands are \n\n- ```findpapers search```: Search for papers metadata using a query. This search will be made by matching the query with the paper\'s title, abstract, and keywords.\n\n- ```findpapers refine```: Refine the search results by selecting/classifying the papers\n\n- ```findpapers download```: Download full-text papers using the search results\n\n- ```findpapers bibtex```: Generate a BibTeX file from the search results\n\nYou can control the commands logging verbosity by the **-v** (or **--verbose**) argument.\n\nIn the following sections, we will show how to use the findpapers commands. However, all the commands have the **--help** argument to display some summary about their usage, E.g., ```findpapers search --help```.\n\n## Search query construction\n\nFirst of all, we need to know how to build the search queries. The search queries must follow the rules:\n\n- All the query terms need to be not empty and enclosed by square brackets. E.g., **[term a]**\n\n- The query can contain boolean operators, but they must be uppercase. The allowed operators are AND, OR, and NOT. E.g., **[term a] AND [term b]**\n\n- All the operators must have one space before and after them. E.g., **[term a] OR [term b] OR [term c]**\n\n- The NOT operator must always be preceded by an AND operator E.g., **[term a] AND NOT [term b]**\n\n- A subquery needs to be enclosed by parentheses. E.g., **[term a] AND ([term b] OR [term c])**\n\n- The composition of terms is only allowed through boolean operators. Queries like "**[term a] [term b]**" are invalid\n\nYou can use some wildcards in the query too. Use **?** to replace a single character or **\\*** to replace any number of characters:\n\n- **[son?]** will match song, sons, ...\n\n- **[son\\*]** will match song, sons, sonic, songwriting, ...\n\nLet\'s see some examples of valid and invalid queries:\n\n| Query  | Valid? |\n| ------------- | ------------- |\n| ([term a] OR [term b])  |  Yes  |\n| [term a] OR [term b]  |  Yes  |\n| [term a] AND [term b]  |  Yes  |\n| [term a] AND NOT ([term b] OR [term c])  |  Yes  |\n| [term a] OR ([term b] AND ([term\\*] OR [term ?]))  |  Yes |\n| [term a]  |  Yes  |\n| ([term a] OR [term b]  |  **No** (missing parentheses)  |\n| [term a] or [term b] |  **No** (lowercase operator)  |\n| term a OR [term b] |  **No** (missing square brackets)  |\n| [term a] [term b] |  **No** (missing boolean operator)  |\n| [term a] XOR [term b] |  **No** (invalid boolean operator)  |\n| [term a] OR NOT [term b] |  **No** (NOT operator must be preceded by AND)  |\n| [] AND [term b] |  **No** (empty term)  |\n\n\n## Basic example (TL;DR)\n\n- Searching for papers:\n\n```console\n$ findpapers search /some/path/search.json "[happiness] AND ([joy] OR [peace of mind]) AND NOT [stressful]"\n```\n\n- Refining search results:\n\n```console\n$ findpapers refine /some/path/search.json\n```\n\n- Downloading full-text from selected papers:\n\n```console\n$ findpapers download /some/path/search.json /some/path/papers/ -s\n```\n\n- Generating BibTeX file from selected papers:\n\n```console\n$ findpapers bibtex /some/path/search.json /some/path/mybib.bib -s\n```\n\n## Advanced example\n\nThis advanced usage documentation can be a bit boring to read (and write), so I think it\'s better to go for a storytelling approach here.\n\nLet\'s take a look at Dr. McCartney\'s research. He\'s a computer scientist interested in AI and music, so he created a search query to collect papers that can help with his research and export it to an environment variable.\n\n```console\n$ export QUERY="([artificial intelligence] OR [AI] OR [machine learning] OR [ML] OR [deep learning] OR [DL]) AND ([music] OR [song])"\n```\n\nDr. McCartney is interested in testing his query, so he decides to collect only 20 papers to test whether the query is suitable for his research.\n\n```console\n$ findpapers search /some/path/search_paul.json --query "$QUERY" --limit 20\n```\n\nBut after taking a look at the results contained in the ```search_paul.json``` file, he notices two problems:\n - Only one database was used to collect the 20 papers\n - Some collected papers were about drums, but he doesn\'t like drums and drummers\n\nSo he decides to solve these problems by reformulating his query.\n\n```console\n$ export QUERY="([artificial intelligence] OR [AI] OR [machine learning] OR [ML] OR [deep learning] OR [DL]) AND ([music] OR [song]) AND NOT [drum*]"\n```\n\nAnd he will perform the search limiting the number of papers that can be collected by 4 per database.\n\n```console\n$ findpapers search /some/path/search_paul.json --query "$QUERY" --limit-db 4\n```\n\nNow his query returned the papers he wanted, but he realized one thing, no papers were collected from Scopus nor IEEE databases. Then he noticed that he needed to pass his Scopus and IEEE API access keys when calling the search command. So he went to https://dev.elsevier.com and https://developer.ieee.org, generated the access keys, and used them in the search.\n\n```console\n$ export IEEE_TOKEN=SOME_SUPER_SECRET_TOKEN\n\n$ export SCOPUS_TOKEN=SOME_SUPER_SECRET_TOKEN\n\n$ findpapers search /some/path/search_paul.json --query "$QUERY" --limit-db 4 --token-ieee "$IEEE_TOKEN" --token-scopus "$SCOPUS_TOKEN"\n```\n\nNow everything is working as he expected, so it\'s time to do the final papers search. So he defines that he wants to collect only works published between 2000 and 2020. He also decides that he only wants papers collected from ACM, IEEE, and Scopus.\n\n```console\n$ findpapers search /some/path/search_paul.json --query "$QUERY" --token-ieee "$IEEE_TOKEN" --token-scopus "$SCOPUS_TOKEN" --since 2000-01-01 --until 2020-12-31 --databases "acm,ieee,scopus"\n```\n\nThe searching process took a long time, but after many cups of coffee, Dr. McCartney finally has a good list of papers with the potential to help in his research. All the information collected is in the ```search_paul.json``` file. He can now access this file manually to filter which works are most interesting for him, but he prefers to use the Findpapers ```refine``` command for this.\n\nFirst, he wants to filter the papers looking only at their basic information.\n\n```console\n$ findpapers refine /some/path/search_paul.json\n```\n\n![refine-01](docs/refine-01.jpeg)\n\nAfter completing the first round filtering of the collected papers, he decides to do new filtering on the selected ones looking at the paper\'s extra info and abstract now. He also chooses to perform some classification while doing this further filtering. And to help in this process, he decides to highlight some keywords contained in the abstract.\n\n```console\n$ export CATEGORIES_CONTRIBUTION="Contribution:Metric,Tool,Model,Method"\n\n$ export CATEGORIES_RESEARCH_TYPE="Research Type:Validation Research,Solution Proposal,Philosophical,Opinion,Experience,Other"\n\n$ export HIGHLIGHTS="propose,achiev,accuracy,method,metric,result,limitation"\n\n$ findpapers refine /some/path/search_paul.json --selected --abstract --extra-info --categories "$CATEGORIES_CONTRIBUTION" --categories "$CATEGORIES_CONTRIBUTION" --highlights "$HIGHLIGHTS"\n```\n\n![refine-02](docs/refine-02.jpeg)\n\nNow that he has selected all the papers he wanted, he will try to download the full-text from all of them that have a "Model" or "Tool" as a contribution.\n\n```console\n$ findpapers download /some/path/search_paul.json /some/path/papers --selected --categories "Contribution:Tool,Model"\n```\n\nHe also wants to generate the BibTeX file from these papers.\n\n```console\n$ findpapers bibtex /some/path/search_paul.json /some/path/mybib.bib --selected --categories "Contribution:Tool,Model"\n```\n\nBut when he compared the papers\' data in the ```/some/path/mybib.bib```  file to the PDF files in the ```/some/path/papers``` folder, he noticed that many papers had not been downloaded.\n\nSo when checking the ```/some/path/papers/download.log``` file he could see the link of all papers that were not downloaded correctly, and noticed that some of them were not downloaded due to some limitation of Findpapers (currently the tool has a set of heuristics to perform the download that may not work in all cases). However, the vast majority of papers were not downloaded because they were behind a paywall. Dr. McCartney has access to these jobs within the network at the university where he works, but he is at home right now.\n\nBut he discovers two things that save him from this mess. First, the university provides a proxy for tunneling requests. Second, Findpapers accepts the configuration of a proxy URL via a variables environment.\n\n```console\nexport FINDPAPERS_PROXY=https://mccartney:super_secret_pass@liverpool.ac.uk:1234\n\n$ findpapers download /some/path/search_paul.json /some/path/papers --selected --categories "Contribution:Tool,Model"\n```\n\nNow the vast majority of the papers he has access have been downloaded correctly.\n\nAnd at the end of it, he decides to download all the selected works and generate their BibTeX file too.\n\n```console\n$ findpapers download /some/path/search_paul.json /some/path/papers --selected\n\n$ findpapers bibtex /some/path/search_paul.json /some/path/mybib.bib --selected\n```\n\nAs you could see, all the information collected and enriched by the Findpapers is persisted in a single JSON file. From this file, it is possible to create interesting visualizations about the collected data ...\n\n![charts](docs/charts.png)\n\n... so, use your imagination!\n\nThat\'s all, folks! We have reached the end of our journey. I hope Dr. McCartney can continue his research and publish his work without any major problems now.\n\nWith this story, we saw all the commands of the tool. I know this documentation is kind of weird, but I haven\'t had time to write more formal documentation of the tool yet. But you can help us to improve this, take a look at the next section and see how you can do that.\n\n\n## Want to help?\n\nSee the [contribution guidelines](https://gitlab.com/jonatasgrosman/findpapers/-/blob/master/CONTRIBUTING.md)\nif you\'d like to contribute to Findpapers project.\n\nYou don\'t even need to know how to code to contribute to the project. Even the improvement of our documentation is an outstanding contribution.\n\nIf this project has been useful for you, please share it with your friends. This project could be helpful for them too.\n\nAnd, if you like this project and wanna motivate the maintainers, give us a :star:. This kind of recognition will make us very happy with the work that we\'ve done :heart:\n\n---\n\n**Note**: If you\'re seen this project from GitHub, this is just a mirror, \nthe official project source code is hosted [here](https://gitlab.com/jonatasgrosman/findpapers) on GitLab.',
    'author': 'Jonatas Grosman',
    'author_email': 'jonatasgrosman@gmail.com',
    'maintainer': 'Jonatas Grosman',
    'maintainer_email': 'jonatasgrosman@gmail.com',
    'url': 'https://gitlab.com/jonatasgrosman/findpapers',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
