#    _____           ______  _____ 
#  / ____/    /\    |  ____ |  __ \
# | |        /  \   | |__   | |__) | Caer - Modern Computer Vision
# | |       / /\ \  |  __|  |  _  /  Languages: Python, C, C++
# | |___   / ____ \ | |____ | | \ \  http://github.com/jasmcaus/caer
#  \_____\/_/    \_ \______ |_|__\_

# Licensed under the MIT License <http://opensource.org/licenses/MIT>
# SPDX-License-Identifier: MIT
# Copyright (c) 2020 The Caer Authors <http://github.com/jasmcaus>

import math
import time
import cv2 as cv

from .._internal import _check_target_size
from ..path import list_videos, exists, mkdir
from ..resize import resize
from ..globals import FRAME_COUNT, FPS
from ..io import imsave


def extract_frames(input_folder, 
                   output_folder, 
                   target_size=None, 
                   include_subdirs=False,
                   label_counter = None, 
                   max_video_count=None, 
                   frames_per_sec=None, 
                   frame_interval=None,
                   dest_filetype='jpg'):
    """ Function to extract frames from videos within a directory
    and save them as separate frames in an output directory.
    Args:
        input_folder: Input video directory.
        output_folder: Output directory to save the frames.
        target_size: Destination Image Size (tuple of size 2)
        label_counter: Starting label counter
        max_video_count: Number of videos to process.
        frames_per_sec: Number of frames to process per second. 
        frame_interval: Interval between the frames to be processed.
        dest_filetype: Processed image filetype (png, jpg) --> Default: png
    Returns:
        label_counter (after processing)
    """

    dest_filetype.replace('.', '')
    processed_videos = 0
    vid_count = 0 # to check if < max_video_count

    if not exists(input_folder):
        raise ValueError('Input folder does not exist', input_folder)
    
    if target_size is not None:
        _ = _check_target_size(target_size)
    
    video_list = list_videos(input_folder, include_subdirs=include_subdirs, use_fullpath=True, verbose=0)

    if len(video_list) == 0:
        raise ValueError(f'No videos found at {input_folder}')

    if label_counter is None:
        label_counter = 0

    if max_video_count is None:
        max_video_count = len(video_list)

    if not exists(output_folder):
        mkdir(output_folder)

    # Begin Timer
    start = time.time()

    for vid_filepath in video_list:
        if vid_count < max_video_count:
            capture = cv.VideoCapture(vid_filepath)
            video_frame_counter = 0
            vid_count += 1

            # Find the number of frames and FPS
            video_frame_count = int(capture.get(FRAME_COUNT)) - 1
            video_fps = math.ceil(capture.get(FPS))
            file = vid_filepath[vid_filepath.rindex('/')+1:]
            
            if frames_per_sec is not None:
                if frame_interval is None:
                    interval = _determine_interval(video_fps/frames_per_sec) # eg: 30//15
            
                else:
                    interval = frame_interval

            # if frames_per_sec and frame_interval are both None, we assume that each frame should be processed
            else:
                interval = 1
            
            # processed_frames = (video_frame_count//video_fps) * frames_per_sec

            print(f'{vid_count}. Reading \'{file}\'. Frame Count: {video_frame_count}. FPS: {video_fps}. Processed frames: {video_frame_count//interval}')
            
            # Start converting the video
            while capture.isOpened():
                _, frame = capture.read()

                if target_size is not None:                    
                    frame = resize(frame, target_size=target_size)
                
                # Write the results back to output location as per specified frames per second
                if video_frame_counter % interval == 0:
                    imsave(f'{output_folder}/{file}_{label_counter}.{dest_filetype}', frame)
                    video_frame_counter += 1
                    label_counter += 1
                    # print('Frame counter: ', video_frame_counter)
                
                video_frame_counter += 1

                # If there are no more frames left
                if video_frame_counter > (video_frame_count-1):
                    capture.release()
                    processed_videos += 1
                    break
    # End timer
    end = time.time()
    
    # Printing stats
    taken = end-start
    if processed_videos > 1:
        print('[INFO] {} videos extracted in {:.0f}m {:.0f}s'.format(processed_videos, 
                                                                    taken // 60,
                                                                    taken % 60 ))
    else:
        print('[INFO] {} video extracted in {:.0f}m {:.0f}s'.format(processed_videos, 
                                                                    taken // 60,
                                                                    taken % 60 ))

    return label_counter


def _determine_interval(x):
    y = '{:.1f}'.format(x)
    inde = y.find('.') + 1
    if inde == -1: # if no '.' (if an integer)
        return x
    if int(y[inde]) < 5:
        return math.floor(x)
    else:
        return math.ceil(x)


__all__ = [
    'extract_frames'
]