## Query Filter Parser

## What are query filter expressions?
Query Filter Expressions are used to express data filtering options at the source in AppFlow. The domain specific language
(DSL) that defines the filter expressions grammar is AppFlow specific. For example: `price > 100 and accountName contains "Alexa"`.
The query expression DSL is a normalized DSL and not specific to DSLs used by specific underlying applications. The connector's
job is to provide the necessary translation from AppFlow DSL to application specific DSL or other syntax the underlying
application requires.

## Why are filter expressions needed in the custom connector SDK?
AppFlow works in incremental query model. It takes the user input for filtering in a flow and translates it to a normalized
DSL and passes it on to connectors. Connectors then take that normalized expression and translate it to application specific
format.

## What is the end-to-end flow that AppFlow uses for querying data from the source connectors?
When an AppFlow user creates a flow definition with custom connector as source and adds some filters in it, AppFlow converts
these filters into filter expression and invoke `query_data` function in the custom connector Lambda with `QueryDataRequest`
as input parameter. The request object contains metadata for the target source object and the translated filter-expression
string.

For example, AppFlow user wants to query subset of data of the `Account` entity from  Salesforce. The `QueryDataRequest`
object contains metadata associated with the `Account` entity and filter expression representing the filters defined in
the flow definition.

The connector implementation for read_data lambda function is expected to parse this filter expression into the connector
specific expression. For example, the filter expression `price > 100 AND accountName CONTAINS "Alexa"` should be
translated to Salesforce specific filter expression as: ` price > 100 AND accountName LIKE '%Alexa%' `.

The translated filter expression can then be passed as query string along with the Salesforce query REST API.

## What are the various operators that are supported in the query filter DSL?
Currently the filter expression DSL support limited set of operators:

* AND
  - attribute1 > value1 *AND* attribute2 > value2
* OR
  - attribute1 > value1 *OR* attribute2 > value2
* BETWEEN   <value1> and <value2>
  - attribute1 *BETWEEN* value1 *AND* value2
* \> (greater than)
  - attribute1 *>* value1
* \>= (greater than equal to)
  - attribute1 *>=* value1
* < (less than)
  - attribute1 *<* value1
* <= (less than equal to)
  - attribute1 *<=* value1
* = (equal to)
  - attribute1 *=* value1
* != (not equal to)
  - attribute1 *!=* value1
* NULL
  - attribute1 = *NULL*
* IN
  - attribute1 *IN* *(* value1, value2, value3 *)*
* CONTAINS
  - attribute *CONTAINS* 'Alexa'
  

## Can the filter expressions be nested?
Yes the filter expressions can be nested. The DSL supports multiple level of nesting. For instance,
`price > 100 and ( accountName contains 'Alexa' OR accountName contains 'Device')`. The connector implementation should
be able to handle multiple levels of nesting. If the nesting depth is more than what the underlying application can
support the connector implementation should throw an exception.

## What are the various datatypes that are supported for the fields in filter expression?
The expressions in the DSL consist of 1) left operand (the field-name of an entity) 2) a logical operator
(e.g. AND, OR, BETWEEN) and 3) the right operand (the field-value). Following are some of the supported data-types
for the field-value:

1. Boolean values will be represented as String. Valid values: True  or  TRUE or true or FALSE or false or False.
2. DateTime value will be represented in ISO date-time format (YYYY-MM-DDThh:mm:ss.sTZD)
3. String value will wrapped around either single or double quote ('value' or "value")
4. Empty string will be represented by ('' or "")


## How can I parse the filter expressions?
The SDK comes bundled with a parser to help parse the filter-parser expressions. The parser is available in the
custom_connector_queryfilter module. The parser is built using antlr4 based grammar and bundles the files generated by
the antlr4 compiler.

## How can I learn more about antlr?
https://www.antlr.org/

## Do I need to really be an expert in antlr in order to use the parser?
No. If you follow the example, parsing the query parser with `parse_tree_builder.py` is only few lines of code.
In addition to that the connector developer needs to provide an implementation of CustomConnectorQueryFilterParserVisitor
for which you can start by looking at the example `SalesForceQueryFilterExpressionVisitor` and start from there.

## Why does the SDK comes bundled with the query filter parser?
One of the input parameter for query_data function is `filter-expression` which uses AppFlow DSL. In order to simplify
the work needed to parse the filter expression the parser has been included.

## How do I validate the passed in filter expression?
Connector implementation can utilize the `parse` method defined in the class `parse_tree_builder.py` 
inside custom-connector-queryfilter package.  Consider this example code:

```python
# method defined in the parse_tree_builder.py class
def parse(filter_expression: str) -> ParseTree:

# Use of parse method    
parse_tree = parse(filter_expression)
filter_expression_visitor = SalesforceQueryFilterExpressionVisitor(entity_definition)
# visit the parse tree to convert the filter expression into salesforce filter query
filter_expression_visitor.visit(parse_tree)
```

If the filterExpression is badly formed the parse() function will throw `InvalidFilterExpressionException` exception. 
This method return the ParseTree which can be handed over to a visitor implementation (see: CustomConnectorQueryFilterParserVisitor) 
to form connector specific query string. Please refer to the custom-connector-example package to learn more about how 
to use this parser.

## How do I implement the visitor implementation for navigating through the ParseTree?
The `filterExpression` is parsed by `CustomConnectorQueryFilterParser` and results into ParseTree. Connector implementation 
should extend the `CustomConnectorQueryFilterParserVisitor` class and override required method to visit parse tree. 
Please checkout `custom_connector_example.query.visitor.SalesForceQueryFilterExpressionVisitor` in the `custom-connector-example` 
module to see the details.

## How do I build the connector specific queryString?
In the bundled example, we have created `SalesForceQueryFilterExpressionVisitor class` which extends `CustomConnectorQueryFilterParser`
visitor class. We will use this class to generate the connector specific query string. This example generates Salesforce 
connector specific queryString. This example uses an instance variable `queryBuilder` (StringIO) to help construct the 
final query expression while navigating the parse tree in the visitor. AppFlow has defined following set of expressions 
in the AppFlow grammar.

```
| NOT expression                                  # notExpression
| left=expression op=andBinary right=expression   # aNDBinaryExpression
| left=expression op=orBinary right=expression    # oRBinaryExpression
| left=identifier op=gtComparator right=value     # greaterThanComparatorExpression
| left=identifier op=geComparator right=value     # greaterThanEqualToComparatorExpression
| left=identifier op=ltComparator right=value     # lesserThanComparatorExpression
| left=identifier op=leComparator right=value     # lesserThanEqualToComparatorExpression
| left=identifier op=eqComparator right=value     # equalToComparatorExpression
| left=identifier op=eqComparator right=bool      # boolEqualToComparatorExpression
| left=identifier op=neComparator right=value     # notEqualToComparatorExpression
| left=identifier op=neComparator right=bool      # boolNotEqualToComparatorExpression
| left=identifier op=likeComparator right=value   # likeComparatorExpression
| (left=identifier op=betweenComparator (l1=value op1=andBinary right=value)) #betweenExpression
| identifier #identifierExpression// Following is a leaf node in the parse tree// This allows validation and transformations of values.
| value #valueExpression
| identifier op=in LPAREN value (COMMA value)* RPAREN  # inExpression  // Supports SQL like 'IN' operator
```
`CustomConnectorQueryFilterParserVisitor` provides a specific method to visit each expression. Connector implementation 
needs to override the specific method and form the connector specific queryString. Antlr runtime library traverses the 
parseTree and invoke the methods in the visitor. For example for parsing the BETWEEN expression in 
`LastModifiedDate between 2021-08-21T05:06:07Z and 2021-08-22T05:06:07Z`, the visitor implementation needs to override
`visitBetweenExpression` from the base class. 

Here is the example implementation from `SalesForceQueryFilterExpressionVisitor:`

```python
def visitBetweenExpression(self, ctx: CustomConnectorQueryFilterParser.BetweenExpressionContext):
    if ctx.getChildCount() == 5:
        identifier = ctx.getChild(0).getText()
        lower_bound = ctx.getChild(2).getText()
        upper_bound = ctx.getChild(4).getText()
        data_type = self.get_field_data_type(ctx.getChild(0).getText()).data_type.name

        lower_bound_comparison = CONDITION_FORMAT.format(identifier,
                                                         COMPARISON_GREATER,
                                                         format_value(lower_bound, data_type, COMPARISON_GREATER))
        upper_bound_comparison = CONDITION_FORMAT.format(identifier,
                                                         COMPARISON_LESSER,
                                                         format_value(upper_bound, data_type, COMPARISON_LESSER))
        self.query_builder.write(lower_bound_comparison + LOGICAL_AND + upper_bound_comparison)
        return self.query_builder.getvalue()

    return self.visitChildren(ctx)
```
This example first checks that `betweenExpressionContext` should contain following 5 child nodes

- `LastModifiedDate` identifier (a field name in Salesforce)
- `BETWEEN` operator
- `2021-08-21T05:06:07Z` (field value)
- `AND` operator
- `2021-08-22T05:06:07Z` (field value)

Connector implementation needs to convert this expression into connector specific queryString like 
`ModifiedDate > 2021-08-21T05:06:07Z and ModifiedDate < 2021-08-22T05:06:07Z` and store in queryBuilder instance variable.
Please refer to `custom-connector-example` in SDK to know how other method can be overridden. Note that for the BETWEEN 
operator, the first operand value should always be smaller than the second operand value.

## How do I validate the field identifiers present in query filter expressions?
AppFlow passes the entity metadata information for the queries entity along with the `ConnectorConext` in the `QueryDataRequest`.
The connector implementation can utilize the entity metadata in their visitor implementation to validate the attribute names. 
This metadata can be used for validation of attribute names at source.

```python
# constructor
def __init__(self, entity_definition: EntityDefinition):
    assert entity_definition, "entity definition can't be null, as it is required for building filter query"
    self.entity_definition = entity_definition
    self.query_builder = StringIO()

# How to validate the identifier aka attributeName at the source
def get_field_data_type(self, field_name: str) -> FieldDefinition:
    """Find FieldDefinition for given field name. This definition contains datatype supported by field and other
    useful metadata information required to build filter clause.

    """
    try:
        return next(filter(lambda x: x.field_name == field_name, self.entity_definition.fields))
    except StopIteration:
        raise ValueError('Filter attribute not found in entity definition')

```

## How do I validate supported operators on specific attribute in the filter expression?
Connector implementation can use the entity metadata information to validate supported operator per field. Each 
fieldDefinition in metadata has supported `filter_operators`. Connector implementation should not allow attribute having
boolean datatype in the between expression. For example, `purchaseStatus between true and false` is a valid filter 
expression as per AppFlow grammar however it is not valid for Salesforce connector implementation. Connector implementation 
should throw exception in such scenario. Boolean attribute should only support `equal or not equal` operators.

## How do I validate the value/format of  each of the attributes in the filter expression ?
Connector implementation can use the metadata information to validate if identifier values in filter expression follow
the correct datatype and format or not. For Example, `LastModifiedDate` is date-time data type field in the Salesforce
metadata. It should also follow the ISO date-time format in the filter expression. If date-time is passed as long value
or other format in filter expression, connector implementation should throw exception. Connector implementation should
also format or convert the value as per source connector requirement. For example, if
source connector requires the date-time in epoch second than he should convert ISO date-time into epoch second while
building the connector specific query string.

## Where do I get hold of entity metadata to validate attributes and its operator ?
This metadata (entityDefinition) is passed in the ConnectorContext object inside QueryDataRequest object while invoking
the lambda to query the data.

## How do I return the connector specific query ?
Connector implementations can add an instance variable to store the connector specific query string.
In python for example the connector implementation can use StringIO to build the query and keep appending the expression
as Antlr traverses the ParseTree.
You can define the custom method in the class to return final result. In the `custom-connector-example`, we have defined
in following way

```python

# Returns the final query expression built for Salesforce
def get_result(self):
# Returns the final query expression built for Salesforce.
    return self.query_builder.getvalue()

# Usage of result
def translate_filter_expression(filter_expression: str, entity_definition: EntityDefinition) -> str:
    # Converts filter expression to Salesforce specific expression using FilterExpressionVisitor.
    if filter_expression:
        parse_tree = parse(filter_expression)
        filter_expression_visitor = SalesforceQueryFilterExpressionVisitor(entity_definition)
        filter_expression_visitor.visit(parse_tree)
        return filter_expression_visitor.get_result()
    return ''

```

## In case of error, what is a valid error code to return?
Connector implementation should return the ErrorDetails with `ErrorCode.InvalidArgument` and details capturing the actual
error. Errors could be related to syntax or datatype and format of value or invalid identifier (aka attributeName).

## Can I generate my own filter library out of Antlr grammar in other languages ?
Yes if that language is supported by antlr4. We have packaged our grammar along with the SDK. You can generate your own
parser library by compiling the grammar files with antlr4.

## I am still not clear, how can I get help?
We understand our documentation is evolving and we want to do a better job. So we welcome the feedback and always ready to
improve our documentation to unblock our customer. You can contact the AppFlow team for more information using the
`appflow-customconnector-beta@amazon.com` mailing list.


