import json
import logging
from typing import Any, Optional

from boto3.session import Session
from botocore.config import Config

from chainsaws.aws.iam import IAMAPI
from chainsaws.aws.scheduler.scheduler_models import SchedulerAPIConfig, ScheduleRequest
from chainsaws.aws.scheduler.scheduler_exception import ScheduleNotFoundException
from chainsaws.aws.sts import STSAPI

logger = logging.getLogger(__name__)


class Scheduler:
    """Low-level EventBridge Scheduler client wrapper."""

    def __init__(
        self,
        boto3_session: Session,
        config: SchedulerAPIConfig | None = None,
    ) -> None:
        self.config = config or SchedulerAPIConfig()

        client_config = Config(
            region_name=self.config.region,
            retries={"max_attempts": self.config.max_retries},
            connect_timeout=self.config.timeout,
            read_timeout=self.config.timeout,
        )

        self.client = boto3_session.client(
            "scheduler",
            config=client_config,
            region_name=self.config.region,
        )

    def create_schedule_group(self, group_name: str) -> None:
        """Create schedule group."""
        try:
            self.client.create_schedule_group(Name=group_name)
        except self.client.exceptions.ConflictException:
            # If exist, pass generating schedule_group
            pass

    def create_schedule(self, request: ScheduleRequest) -> dict[str, Any]:
        """Create schedule."""
        schedule_args = {
            "Name": request.name,
            "GroupName": request.group_name,
            "ScheduleExpression": request.schedule_expression,
            "State": "ENABLED",
            "Target": {
                "Arn": request.lambda_function_arn,
                "RoleArn": self._get_role_arn(),
            },
            "FlexibleTimeWindow": {
                "Mode": "OFF",
            },
        }

        if request.description:
            schedule_args["Description"] = request.description
        if request.input_data:
            schedule_args["Target"]["Input"] = json.dumps(request.input_data)

        try:
            return self.client.create_schedule(**schedule_args)
        except self.client.exceptions.ConflictException as e:
            logger.exception(f"Schedule '{request.name}' already exists in group '{
                request.group_name}': {e!s}")
            raise

    def _get_role_arn(self) -> str:
        """Get or create IAM role ARN for scheduler."""
        try:
            sts = STSAPI(config=self.config)
            identity = sts.get_caller_identity()
            role_name = "EventBridgeSchedulerRole"
            role_arn = f"arn:aws:iam::{identity.account}:role/{role_name}"

            iam = IAMAPI(config=self.config)
            try:
                iam.get_role(role_name)
            except Exception:  # Role doesn't exist
                self._create_scheduler_role(iam, role_name)
                logger.info(f"Created IAM role: {role_name}")

            return role_arn
        except Exception as e:
            logger.exception(f"Failed to get/create scheduler role: {e!s}")
            raise

    def _create_scheduler_role(self, iam: IAMAPI, role_name: str) -> None:
        """Create IAM role for EventBridge Scheduler."""
        trust_policy = {
            "Version": "2012-10-17",
            "Statement": [{
                "Effect": "Allow",
                "Principal": {
                    "Service": "scheduler.amazonaws.com",
                },
                "Action": "sts:AssumeRole",
            }],
        }

        permission_policy = {
            "Version": "2012-10-17",
            "Statement": [{
                "Effect": "Allow",
                "Action": [
                    "lambda:InvokeFunction",
                ],
                "Resource": ["arn:aws:lambda:*:*:function:*"],
            }],
        }

        iam.create_role(
            role_name=role_name,
            trust_policy=trust_policy,
            description="Role for EventBridge Scheduler to invoke Lambda functions (Generated By Chainsaws",
        )

        iam.put_role_policy(
            role_name=role_name,
            policy_name=f"{role_name}Policy",
            policy_document=permission_policy,
        )

    def delete_schedule(self, name: str, group_name: str) -> None:
        """Delete schedule."""
        try:
            self.client.delete_schedule(
                Name=name,
                GroupName=group_name,
            )
        except self.client.exceptions.ResourceNotFoundException as e:
            logger.error(f"Schedule '{name}' not found in group '{
                         group_name}': {e!s}")
            raise ScheduleNotFoundException(
                f"Schedule '{name}' not found") from e
        except Exception as e:
            logger.error(f"Failed to delete schedule '{name}': {e!s}")
            raise

    def update_schedule_state(self, name: str, group_name: str, state: str) -> None:
        """Update schedule state."""
        try:
            self.client.update_schedule(
                Name=name,
                GroupName=group_name,
                State=state,
            )
        except self.client.exceptions.ResourceNotFoundException as e:
            logger.error(f"Schedule '{name}' not found in group '{
                         group_name}': {e!s}")
            raise ScheduleNotFoundException(
                f"Schedule '{name}' not found") from e
        except Exception as e:
            logger.error(f"Failed to update schedule state '{name}': {e!s}")
            raise

    def list_schedules(
        self,
        group_name: str,
        next_token: Optional[str] = None,
        max_results: int = 100,
    ) -> dict[str, Any]:
        """List schedules in group."""
        try:
            params = {
                "GroupName": group_name,
                "MaxResults": max_results,
            }
            if next_token:
                params["NextToken"] = next_token

            return self.client.list_schedules(**params)
        except Exception as e:
            logger.error(f"Failed to list schedules in group '{
                         group_name}': {e!s}")
            raise

    def get_schedule(self, name: str, group_name: str) -> dict[str, Any]:
        """Get schedule details."""
        try:
            return self.client.get_schedule(
                Name=name,
                GroupName=group_name,
            )
        except self.client.exceptions.ResourceNotFoundException as e:
            logger.error(f"Schedule '{name}' not found in group '{
                         group_name}': {e!s}")
            raise ScheduleNotFoundException(
                f"Schedule '{name}' not found") from e
        except Exception as e:
            logger.error(f"Failed to get schedule '{name}': {e!s}")
            raise

    def update_schedule(
        self,
        name: str,
        group_name: str,
        schedule_expression: Optional[str] = None,
        description: Optional[str] = None,
        input_data: Optional[dict] = None,
        state: Optional[str] = None,
    ) -> dict[str, Any]:
        """Update schedule."""
        try:
            # Get current schedule
            current = self.get_schedule(name, group_name)

            # Prepare update params
            params = {
                "Name": name,
                "GroupName": group_name,
                "ScheduleExpression": schedule_expression or current["ScheduleExpression"],
                "Target": current["Target"],
                "FlexibleTimeWindow": current["FlexibleTimeWindow"],
            }

            if description is not None:
                params["Description"] = description
            elif "Description" in current:
                params["Description"] = current["Description"]

            if input_data is not None:
                params["Target"]["Input"] = json.dumps(input_data)

            if state is not None:
                params["State"] = state
            else:
                params["State"] = current["State"]

            return self.client.update_schedule(**params)
        except ScheduleNotFoundException:
            raise
        except Exception as e:
            logger.error(f"Failed to update schedule '{name}': {e!s}")
            raise
