import logging
from typing import Any, Optional

from chainsaws.aws.iam._iam_internal import IAM
from chainsaws.aws.iam.iam_models import IAMAPIConfig, RoleConfig, RolePolicyConfig
from chainsaws.aws.shared import session

logger = logging.getLogger(__name__)


class IAMAPI:
    """High-level IAM operations."""

    def __init__(self, config: IAMAPIConfig | None = None) -> None:
        self.config = config or IAMAPIConfig()
        self.boto3_session = session.get_boto_session(
            self.config.credentials if self.config.credentials else None,
        )
        self.iam = IAM(self.boto3_session, config=self.config)

    def create_role(
        self,
        role_name: str,
        trust_policy: dict[str, Any],
        description: str = "Generated by Chainsaws",
        wait: bool = True,
    ) -> dict[str, Any]:
        """Create IAM role.

        Args:
            role_name: Name of the role
            trust_policy: Trust policy document
            description: Role description
            wait: Whether to wait for role to be accessible

        Returns:
            Dict containing role information

        """
        config = RoleConfig(
            name=role_name,
            trust_policy=trust_policy,
            description=description,
        )
        return self.iam.create_role(config, wait=wait)

    def wait_until_role_exists(self, role_name: str, max_attempts: int = 20) -> None:
        """Wait until IAM role exists and is accessible.

        Args:
            role_name: Name of the role to wait for
            max_attempts: Maximum number of attempts to check role existence

        Raises:
            TimeoutError: If role does not exist after maximum attempts

        """
        return self.iam.wait_until_role_exists(role_name, max_attempts)

    def put_role_policy(
        self,
        role_name: str,
        policy_name: str,
        policy_document: dict[str, Any],
        wait: bool = True,
    ) -> dict[str, Any]:
        """Put inline policy to IAM role.

        Args:
            role_name: Name of the role
            policy_name: Name of the policy
            policy_document: Policy document
            wait: Whether to wait for policy to be attached

        Returns:
            Dict containing response information

        """
        config = RolePolicyConfig(
            role_name=role_name,
            policy_name=policy_name,
            policy_document=policy_document,
        )
        return self.iam.put_role_policy(config, wait=wait)

    def wait_until_role_policy_exists(
        self,
        role_name: str,
        policy_name: str,
        max_attempts: int = 20,
    ) -> None:
        """Wait until inline policy exists on role.

        Args:
            role_name: Name of the role
            policy_name: Name of the policy
            max_attempts: Maximum number of attempts to check policy existence

        Raises:
            TimeoutError: If policy is not attached after maximum attempts

        """
        return self.iam.wait_until_role_policy_exists(
            role_name,
            policy_name,
            max_attempts,
        )

    def get_role(self, role_name: str) -> dict[str, Any]:
        """Get IAM role.

        Args:
            role_name: Name of the role

        Returns:
            Dict containing role information

        """
        return self.iam.get_role(role_name)

    def add_lambda_permission(
        self,
        function_name: str,
        statement_id: str,
        action: str,
        principal: str,
        source_arn: Optional[str] = None,
    ) -> dict[str, Any]:
        """Add permission to Lambda function policy.

        Args:
            function_name: Name or ARN of the Lambda function
            statement_id: Statement identifier
            action: The AWS Lambda action you want to allow
            principal: The principal who is getting this permission
            source_arn: Optional ARN of the source service

        Returns:
            Dict containing response information

        Example:
            ```python
            # Allow API Gateway to invoke Lambda
            iam.add_lambda_permission(
                function_name="my-function",
                statement_id="AllowAPIGateway",
                action="lambda:InvokeFunction",
                principal="apigateway.amazonaws.com",
                source_arn="arn:aws:execute-api:region:account-id:api-id/*"
            )
            ```
        """
        try:
            from chainsaws.aws.lambda_client.lambda_client import LambdaAPI
            lambda_api = LambdaAPI()
            return lambda_api.add_permission(
                function_name=function_name,
                statement_id=statement_id,
                action=action,
                principal=principal,
                source_arn=source_arn,
            )
        except Exception as e:
            if "ResourceConflictException" in str(e):
                # Permission already exists
                logger.info(
                    f"Permission {statement_id} already exists for function {
                        function_name}"
                )
                return {}
            raise Exception(
                f"Failed to add Lambda permission: {str(e)}") from e
