//  ---------------------------------------------------------------------------
//
//  @file       TwFonts.cpp
//  @author     Philippe Decaudin - http://www.antisphere.com
//  @license    This file is part of the AntTweakBar library.
//              For conditions of distribution and use, see License.txt
//
//  ---------------------------------------------------------------------------

//#include "TwMgr.h"
#include "tiny_fonts.h"
#include <assert.h>
#include <string.h>
#include <string>

// Fedora patch: memset()
// using std::memset;

//  ---------------------------------------------------------------------------

CTexFont::CTexFont() {
  for (int i = 0; i < 256; ++i) {
    m_CharU0[i] = 0;
    m_CharU1[i] = 0;
    m_CharV0[i] = 0;
    m_CharV1[i] = 0;
    m_CharWidth[i] = 0;
  }
  m_TexWidth = 0;
  m_TexHeight = 0;
  m_TexBytes = NULL;
  m_NbCharRead = 0;
  m_CharHeight = 0;
}

//  ---------------------------------------------------------------------------

CTexFont::~CTexFont() {
  if (m_TexBytes) delete[] m_TexBytes;
  m_TexBytes = NULL;
  m_TexWidth = 0;
  m_TexHeight = 0;
  m_NbCharRead = 0;
}

//  ---------------------------------------------------------------------------

static int NextPow2(int _n) {
  int r = 1;
  while (r < _n) r *= 2;
  return r;
}

//  ---------------------------------------------------------------------------

const char *g_ErrBadFontHeight =
    "Cannot determine font height while reading font bitmap (check first pixel "
    "column)";

CTexFont *TwGenerateFont(const unsigned char *_Bitmap, int _BmWidth,
                         int _BmHeight) {
  // find height of the font
  int x, y;
  int h = 0, hh = 0;
  int r, NbRow = 0;
  for (y = 0; y < _BmHeight; ++y)
    if (_Bitmap[y * _BmWidth] == 0) {
      if ((hh <= 0 && h <= 0) || (h != hh && h > 0 && hh > 0)) {
        assert(0);
#if 0
				
                g_TwMgr->SetLastError(g_ErrBadFontHeight);
#endif
        return NULL;
      } else if (h <= 0)
        h = hh;
      else if (hh <= 0)
        break;
      hh = 0;
      ++NbRow;
    } else
      ++hh;

  // find width and position of each character
  int w = 0;
  int x0[224], y0[224], x1[224], y1[224];
  int ch = 32;
  int start;
  for (r = 0; r < NbRow; ++r) {
    start = 1;
    for (x = 1; x < _BmWidth; ++x)
      if (_Bitmap[(r * (h + 1) + h) * _BmWidth + x] == 0 || x == _BmWidth - 1) {
        if (x == start) break;  // next row
        if (ch < 256) {
          x0[ch - 32] = start;
          x1[ch - 32] = x;
          y0[ch - 32] = r * (h + 1);
          y1[ch - 32] = r * (h + 1) + h - 1;
          w += x - start + 1;
          start = x + 1;
        }
        ++ch;
      }
  }
  for (x = ch - 32; x < 224; ++x) {
    x0[ch] = 0;
    x1[ch] = 0;
    y0[ch] = 0;
    y1[ch] = 0;
  }

  // Repack: build 14 rows of 16 characters.
  // - First, find the largest row
  int l, lmax = 1;
  for (r = 0; r < 14; ++r) {
    l = 0;
    for (x = 0; x < 16; ++x) l += x1[x + r * 16] - x0[x + r * 16] + 1;
    if (l > lmax) lmax = l;
  }
  // A little empty margin is added between chars to avoid artefact when
  // antialiasing is on
  const int MARGIN_X = 2;
  const int MARGIN_Y = 2;
  lmax += 16 * MARGIN_X;
  // - Second, build the texture
  CTexFont *TexFont = new CTexFont;
  TexFont->m_NbCharRead = ch - 32;
  TexFont->m_CharHeight = h;
  TexFont->m_TexWidth = NextPow2(lmax);
  TexFont->m_TexHeight = NextPow2(14 * (h + MARGIN_Y));
  TexFont->m_TexBytes =
      new unsigned char[TexFont->m_TexWidth * TexFont->m_TexHeight];
  memset(TexFont->m_TexBytes, 0, TexFont->m_TexWidth * TexFont->m_TexHeight);
  int xx;
  float du = 0.f;  // 0.4f;
  float dv = 0.f;  // 0.4f;
#if 0
	assert( g_TwMgr!=NULL );
    if( g_TwMgr )
    {
        if( g_TwMgr->m_GraphAPI==TW_OPENGL || g_TwMgr->m_GraphAPI==TW_OPENGL_CORE )
        {
            du = 0;
            dv = 0;
        }
        else    // texel alignment for D3D
        {
            du = 0.5f;
            dv = 0.5f;
        }
    }
#endif

  float alpha;
  for (r = 0; r < 14; ++r)
    for (xx = 0, ch = r * 16; ch < (r + 1) * 16; ++ch)
      if (y1[ch] - y0[ch] == h - 1) {
        for (y = 0; y < h; ++y)
          for (x = x0[ch]; x <= x1[ch]; ++x) {
            alpha = ((float)(_Bitmap[x + (y0[ch] + y) * _BmWidth])) / 256.0f;
            // alpha = alpha*sqrtf(alpha); // powf(alpha, 1.5f);   // some gamma
            // correction
            TexFont->m_TexBytes[(xx + x - x0[ch]) + (r * (h + MARGIN_Y) + y) *
                                                        TexFont->m_TexWidth] =
                (unsigned char)(alpha * 256.0f);
          }
        TexFont->m_CharU0[ch + 32] =
            (float(xx) + du) / float(TexFont->m_TexWidth);
        xx += x1[ch] - x0[ch] + 1;
        TexFont->m_CharU1[ch + 32] =
            (float(xx) + du) / float(TexFont->m_TexWidth);
        TexFont->m_CharV0[ch + 32] =
            (float(r * (h + MARGIN_Y)) + dv) / float(TexFont->m_TexHeight);
        TexFont->m_CharV1[ch + 32] =
            (float(r * (h + MARGIN_Y) + h) + dv) / float(TexFont->m_TexHeight);
        TexFont->m_CharWidth[ch + 32] = x1[ch] - x0[ch] + 1;
        xx += MARGIN_X;
      }

  const unsigned char Undef = 127;  // default character used as for undifined
                                    // ones (having ascii codes from 0 to 31)
  for (ch = 0; ch < 32; ++ch) {
    TexFont->m_CharU0[ch] = TexFont->m_CharU0[Undef];
    TexFont->m_CharU1[ch] = TexFont->m_CharU1[Undef];
    TexFont->m_CharV0[ch] = TexFont->m_CharV0[Undef];
    TexFont->m_CharV1[ch] = TexFont->m_CharV1[Undef];
    TexFont->m_CharWidth[ch] = TexFont->m_CharWidth[Undef] / 2;
  }

  return TexFont;
}

//  ---------------------------------------------------------------------------

CTexFont *g_DefaultSmallFont = NULL;
CTexFont *g_DefaultNormalFont = NULL;
CTexFont *g_DefaultNormalFontAA = NULL;

CTexFont *g_DefaultLargeFont = NULL;
CTexFont *g_DefaultFixed1Font = NULL;

// Small font
const int FONT0_BM_W = 211;
const int FONT0_BM_H = 84;
static const unsigned char s_Font0[] = {
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   0,   255, 0,   255, 0,   255, 0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   255, 0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   255, 255, 0,   0,   0,   255, 0,   0,   255, 255, 255, 0,
    0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 0,   255, 255, 255, 255,
    0,   0,   255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255, 0,
    0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   255, 0,   255, 0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   255, 255, 255, 255, 0,   255, 0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 255, 255, 255, 0,   255, 0,   0,   0,   255, 0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 255, 0,   0,   0,   0,
    255, 255, 0,   255, 0,   255, 255, 0,   0,   0,   255, 255, 0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   255, 255, 0,   0,   255, 0,   0,   255, 0,   255,
    255, 255, 0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   0,
    255, 255, 0,   0,   0,   255, 255, 255, 0,   0,   255, 0,   0,   255, 0,
    0,   0,   255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 255, 0,   0,
    0,   0,   255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 255, 255, 255, 255,
    0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 255,
    0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 255, 255, 255,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   255, 255,
    255, 0,   255, 255, 255, 255, 0,   255, 255, 255, 0,   0,   0,   0,   0,
    255, 0,   255, 255, 255, 0,   0,   0,   255, 255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127,
    0,   127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 0,   127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    0,   127, 127, 127, 0,   127, 0,   127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 255, 0,   255, 0,   0,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,
    255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   255, 255,
    255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   255, 255, 255, 255,
    255, 0,   0,   0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   255,
    0,   255, 255, 255, 0,   0,   255, 255, 0,   255, 0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   255, 255, 255, 255,
    0,   0,   0,   255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   255, 255, 255, 255, 0,   255, 0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   255,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,
    0,   255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   255, 255,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 255, 0,   0,   255, 255, 0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   255, 0,   255, 0,   255,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   255, 0,   255, 0,   255, 0,   0,   255, 255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 255, 0,   255, 0,   255, 0,   255, 0,   0,   255, 0,   0,
    255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 255, 255, 255, 0,   0,   255,
    255, 255, 255, 0,   0,   255, 0,   0,   255, 255, 255, 0,   255, 255, 255,
    255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 255, 0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   255, 0,   255, 0,
    255, 0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,
    0,   255, 0,   255, 0,   255, 0,   255, 0,   0,   255, 255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 255, 0,   255, 0,   255, 0,   255, 0,   255, 255, 255,
    255, 255, 255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   255, 0,   0,   0,   255, 255, 0,   255, 0,   0,   0,   0,   0,   255,
    0,   255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   255, 0,   255, 0,   255, 0,   0,   255, 255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 255, 0,   0,   255, 255, 255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 255,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 255, 255, 255, 0,   0,   0,   0,   255,
    255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 255, 255, 0,   255, 255, 0,   0,
    255, 0,   0,   0,   255, 0,   255, 255, 255, 255, 255, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   255, 255, 255, 255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   255,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 255, 255, 255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 0,   127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   12,  0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   255, 255, 0,   0,   255, 255, 255, 0,   0,   0,   255,
    255, 0,   0,   255, 255, 255, 0,   0,   255, 255, 0,   0,   255, 255, 255,
    0,   255, 255, 255, 0,   255, 255, 255, 0,   0,   255, 0,   255, 255, 0,
    255, 0,   0,   255, 0,   255, 0,   255, 255, 255, 0,   255, 255, 0,   0,
    255, 255, 255, 0,   0,   0,   255, 255, 0,   0,   255, 255, 255, 0,   0,
    0,   255, 255, 255, 0,   255, 0,   255, 255, 255, 255, 0,   255, 255, 0,
    255, 0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   255, 0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255,
    255, 255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255,
    0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   255, 0,   0,   255, 0,   255, 255, 0,   255, 0,   0,   0,   255, 0,
    0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 255, 0,   0,   255, 0,   0,   255, 4,   4,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   255, 255, 255, 255, 0,   255,
    0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 255, 0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   255, 255, 0,   0,   0,   255, 255, 4,   255, 255,
    0,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    255, 255, 0,   255, 255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   255,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   0,   0,   255, 255, 255, 0,   255, 255, 255,
    0,   0,   0,   255, 255, 0,   0,   255, 255, 255, 0,   0,   255, 255, 255,
    0,   255, 0,   0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 0,   255,
    0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   255, 0,   0,   255,
    0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 255, 0,   0,   255,
    255, 255, 0,   0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 255, 255,
    0,   0,   255, 0,   0,   255, 255, 255, 0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   20,  0,   255,
    0,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   0,
    0,   0,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 0,
    127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   255, 255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 255, 255, 0,   0,
    255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   255,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 0,   0,
    255, 255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 0,   0,   255,
    255, 255, 255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255,
    0,   255, 0,   0,   255, 255, 0,   255, 0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   255, 255, 255, 255, 255, 0,   255, 0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    255, 255, 255, 255, 255, 255, 0,   0,   255, 0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 255, 0,   255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   255, 255, 0,   0,   255, 0,   255, 255, 0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 255, 255, 255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,
    255, 255, 255, 0,   255, 0,   0,   0,   255, 255, 0,   255, 255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 255, 255, 0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   0,   0,
    0,   0,   255, 255, 0,   255, 0,   255, 255, 0,   255, 255, 0,   0,   0,
    255, 255, 255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 255, 255,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 255,
    255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 255, 0,   255, 255, 255, 255, 0,   255, 255, 255, 255,
    255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   255, 255, 255,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255, 0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   255, 0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 0,   255, 255,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255, 0,   255,
    255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255, 0,   0,   255,
    255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 0,
    255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255, 0,   255, 255,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   255, 255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 0,   127, 0,   127,
    127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   255, 255, 255, 0,   0,   255, 0,
    255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 0,   255, 255,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 255, 255, 0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 0,   255, 0,   0,
    0,   0,   0,   255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   255, 0,
    0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   255, 255, 255, 0,
    0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   255, 255, 0,   0,   255, 255, 0,   255,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 255,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   255,
    0,   255, 0,   255, 0,   0,   255, 255, 255, 0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    255, 0,   255, 0,   255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   255, 255, 255,
    0,   255, 255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255,
    255, 0,   255, 255, 0,   0,   0,   255, 0,   255, 0,   0,   255, 255, 0,
    0,   255, 255, 0,   0,   0,   255, 0,   255, 0,   255, 255, 0,   0,   255,
    255, 0,   255, 0,   255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 255,
    255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    255, 255, 0,   255, 0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 255,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 255, 0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 255, 255, 255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   255, 0,   0,   255, 255, 255, 0,   255, 255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 255,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 127, 127, 127, 0,   127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127,
    0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 255, 0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   255, 255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255,
    0,   0,   127, 0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   0,
    255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,   255, 255, 255,
    255, 255, 0,   255, 255, 255, 255, 255, 0,   255, 255, 255, 0,   255, 255,
    255, 0,   255, 255, 255, 0,   255, 255, 255, 0,   255, 255, 255, 255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 255, 0,
    0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255,
    0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255,
    255, 0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    0,   255, 0,   127, 0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255,
    0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    255, 0,   0,   255, 255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 255, 255, 0,   0,   255,
    0,   0,   255, 0,   127, 0,   255, 0,   0,   255, 0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,
    255, 0,   255, 0,   0,   127, 0,   255, 0,   0,   255, 0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   255, 255, 255, 255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,
    255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 255,
    255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   255, 0,   0,   255, 0,   127, 255, 255, 255, 255, 255, 255, 0,   255,
    255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255, 255,
    255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255, 255, 255,
    255, 255, 255, 0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   127, 255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 255,
    255, 0,   0,   255, 0,   0,   255, 0,   127, 255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 255, 255, 255, 0,   0,
    0,   255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 0,   255, 255, 255,
    255, 255, 0,   255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,
    255, 255, 255, 0,   255, 255, 255, 0,   255, 255, 255, 0,   255, 255, 255,
    0,   255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 0,   0,
    0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 0,
    0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   255, 0,   255, 255, 255, 0,   0,   0,   0,   255, 255, 255, 255,
    0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   255, 255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,
    255, 0,   255, 255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   255, 0,   255, 0,   255, 255, 0,   255, 0,   0,
    255, 255, 0,   255, 0,   255, 255, 0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   255, 0,   255, 0,   255, 255, 0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255,
    0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255,
    0,   0,   255, 255, 255, 0,   255, 255, 0,   0,   0,   255, 255, 0,   0,
    255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   0,   0,
    255, 255, 0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 255, 255, 0,   0,   0,   255, 255, 0,   0,
    0,   255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   0,
    0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255,
    255, 255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   255, 0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 0,   255, 0,
    0,   255, 255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255,
    0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   0,
    255, 255, 255, 0,   0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   0,
    255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 0,   255, 0,   0,
    0,   255, 255, 255, 255, 0,   255, 255, 255, 255, 0,   255, 255, 255, 255,
    0,   255, 255, 255, 255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   255, 0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    255, 255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   255, 255, 255, 0,   0,   255, 255, 255,
    0,   0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   0,   255, 255, 255,
    0,   0,   255, 255, 255, 0,   0,   255, 255, 0,   255, 255, 255, 0,   0,
    255, 255, 0,   0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   0,   255,
    255, 255, 0,   0,   255, 255, 255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 255, 0,   0,   255, 0,   0,   255, 0,   0,
    255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,   0,   0,
    255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   0,   255, 255,
    255, 0,   0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   0,   0,   255,
    0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 0,   127,
    0,   127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0};

// Normal font
const int FONT1_BM_W = 253;
const int FONT1_BM_H = 106;
static const unsigned char s_Font1[] = {
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 255, 0,   0,   0,   255,
    255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 0,   0,   255, 255, 255,
    255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 255, 255, 0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   255, 255, 255, 0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 0,   0,
    255, 0,   255, 0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255,
    255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 255, 0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 0,   0,
    0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   255, 255, 0,   0,   255, 0,   0,   255, 255, 0,   0,   0,
    255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 255, 255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255,
    255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 255, 0,   0,   0,   255, 255, 255, 0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   255, 255,
    255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 255, 255,
    0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127,
    0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 0,   255, 0,
    0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   255, 255,
    255, 255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    255, 255, 255, 255, 0,   0,   0,   0,   255, 255, 255, 255, 0,   255, 255,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 255, 0,   255, 255,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   255, 255, 0,   255, 255, 0,   0,   0,   0,   255, 0,   0,
    0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,
    0,   0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    255, 255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 255, 0,   0,   0,   0,
    255, 255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   255, 255, 0,   255, 255, 0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   255, 0,   255, 255, 255, 255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 255,
    255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,   255, 0,   0,   0,
    255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   255, 0,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   255, 255, 255, 255, 0,   0,   0,   0,   255, 255, 255, 255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    255, 0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   255, 255, 0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 255,
    255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 255, 255, 255, 255,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   255, 255, 0,   0,   255, 0,   255, 0,   0,   0,   255, 0,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 0,   255,
    255, 255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 255, 0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 255,
    255, 255, 255, 0,   0,   0,   0,   255, 255, 255, 255, 0,   255, 255, 255,
    255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 255, 0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 255, 255, 0,   255, 255, 255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 255, 255, 255, 255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,
    255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 255,
    255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   255, 255, 255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255,
    255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   255, 0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   255, 255,
    255, 0,   0,   255, 0,   255, 255, 255, 0,   0,   0,   255, 255, 255, 255,
    0,   0,   255, 255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,
    255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,   255, 0,   255, 255,
    255, 0,   0,   255, 0,   255, 255, 0,   255, 0,   0,   0,   255, 0,   255,
    0,   255, 255, 255, 255, 0,   255, 255, 255, 0,   0,   255, 0,   255, 255,
    255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   255, 0,   255, 255, 255,
    0,   0,   0,   255, 255, 255, 255, 255, 0,   255, 0,   255, 0,   0,   255,
    255, 255, 0,   255, 255, 255, 255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 255,
    255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   255, 255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   255,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255,
    0,   0,   0,   255, 0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   255, 255, 255, 255, 255, 255, 0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,
    0,   255, 0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,
    255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 255,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 255, 0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,   0,   0,   255,
    255, 255, 255, 0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 255, 255,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 255, 255, 0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   255, 0,   255, 0,   0,
    0,   255, 255, 255, 0,   0,   0,   0,   255, 255, 0,   0,   255, 255, 255,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   255, 0,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 0,   127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   255, 255, 255,
    255, 0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 255,
    255, 255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,
    0,   0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   255,
    255, 255, 255, 255, 255, 255, 255, 255, 0,   255, 0,   0,   0,   255, 0,
    255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   255, 255, 255, 0,
    255, 0,   0,   0,   255, 0,   0,   255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   255, 255, 0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   127, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 0,   0,
    255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 255,
    0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   127, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,
    255, 0,   0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   255, 255,
    255, 255, 0,   255, 255, 255, 0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   255, 255, 255, 255, 0,   0,   255, 0,   0,   0,
    255, 0,   0,   127, 255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,
    0,   127, 255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   255, 0,   0,
    255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 255, 255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255,
    255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   127,
    255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 0,   255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 255, 255, 255,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   127, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   127, 0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   127, 0,   0,   255, 255, 255, 255,
    0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   255, 255, 0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 255,
    255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   0,
    0,   255, 255, 255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 255,
    255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 0,   255, 255, 255, 255, 0,   0,   0,   255, 255,
    255, 255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 127,
    127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255,
    255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,
    0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 255, 0,   0,   255, 0,   0,   0,   0,   255,
    255, 255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 255, 255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 255, 255, 255, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 255, 255, 255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255,
    255, 255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255,
    0,   0,   255, 255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 255,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,
    255, 255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,
    255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 255, 0,   255, 0,   0,
    255, 0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 0,   255, 255,
    255, 0,   255, 0,   0,   255, 255, 255, 0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 0,
    0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 255, 255, 255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   255, 255, 0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 255,
    0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   255, 255, 0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    255, 255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 255, 255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255,
    0,   255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,
    255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255,
    255, 255, 0,   255, 255, 255, 0,   255, 255, 255, 0,   255, 255, 255, 0,
    0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 0,   0,   0,   0,   255,
    0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,
    255, 255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255,
    255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   255, 255, 0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 255,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 255,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 0,
    255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255,
    255, 255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255,
    255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   0,
    255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,
    0,   255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 255, 255, 255, 255, 0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   255,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,
    255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255,
    255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255, 255,
    255, 0,   255, 255, 255, 0,   255, 255, 255, 0,   255, 255, 255, 0,   0,
    255, 255, 255, 255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 255, 0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 255,
    0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,   255,
    255, 255, 255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 255,
    0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255, 255,
    255, 0,   0,   0,   0,   0,   255, 255, 255, 0,   0,   0,   0,   0,   255,
    255, 255, 0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 255, 0,   0,
    0,   255, 255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255,
    255, 0,   0,   0,   0,   0,   255, 255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,
    255, 255, 0,   0,   0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   255, 0,   255, 0,   255, 0,   255, 255, 0,   255, 0,   0,   255,
    0,   255, 0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255,
    0,   0,   0,   255, 0,   255, 255, 0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,
    0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   255, 255, 255, 0,   0,   0,   255, 255,
    255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   0,
    0,   255, 255, 255, 0,   0,   0,   255, 255, 255, 0,   0,   0,   255, 255,
    255, 0,   0,   255, 255, 0,   0,   0,   0,   255, 255, 255, 255, 0,   0,
    255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   255,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 0,
    255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 255, 0,   255, 0,   0,
    255, 0,   255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,
    255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   255,
    255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   0,   0,
    0,   255, 0,   0,   0,   0,   0,   0,   255, 255, 255, 255, 0,   0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   255, 0,   255, 0,   255, 255, 255, 0,   0,   255, 0,   0,   0,   255,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    255, 0,   0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,
    0,   255, 0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   255, 255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   255, 255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   255, 0,   0,   0,   255, 0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255,
    255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,
    0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 255, 255, 255, 255,
    255, 0,   255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 255, 255, 0,
    255, 255, 255, 255, 255, 255, 0,   255, 255, 255, 255, 255, 255, 0,   255,
    255, 255, 255, 255, 255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,
    255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255,
    0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255, 255, 255, 255, 0,
    255, 0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255,
    0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,
    0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,
    255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255,
    0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   0,   0,
    255, 0,   0,   255, 0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 255, 0,   0,   0,
    255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   255, 0,   255, 0,   0,   0,
    255, 0,   255, 0,   0,   0,   255, 255, 0,   0,   0,   255, 0,   255, 0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   255, 0,
    0,   0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,
    0,   0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,
    0,   255, 0,   255, 0,   0,   0,   0,   255, 0,   255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 0,   0,
    0,   255, 0,   0,   255, 0,   0,   0,   255, 255, 0,   255, 0,   0,   0,
    255, 255, 0,   255, 0,   0,   0,   255, 255, 0,   255, 0,   0,   0,   255,
    255, 0,   0,   0,   255, 0,   0,   0,   255, 0,   0,   0,   0,   255, 0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   255, 255, 255, 255, 0,
    0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255,
    255, 255, 0,   0,   255, 255, 255, 255, 0,   0,   255, 255, 255, 255, 0,
    0,   255, 255, 255, 0,   0,   255, 255, 255, 0,   0,   0,   255, 255, 255,
    255, 0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,
    0,   255, 0,   255, 0,   0,   255, 0,   0,   255, 0,   0,   255, 255, 255,
    255, 0,   0,   255, 0,   0,   0,   0,   255, 0,   0,   255, 255, 255, 255,
    0,   0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,
    0,   0,   255, 255, 255, 255, 0,   0,   0,   255, 255, 255, 255, 0,   0,
    0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   255, 255, 255, 255, 0,
    0,   0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 255, 255, 0,   255,
    0,   0,   255, 255, 255, 0,   255, 0,   0,   255, 255, 255, 0,   255, 0,
    0,   0,   255, 0,   0,   0,   255, 255, 255, 255, 255, 0,   0,   0,   0,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 0,   0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   255, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   255, 0,   0,
    0,   0,   255, 0,   0,   0,   0,   0,   0,   0,   255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 0,   127,
    127, 0,   127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0};

// Normal font anti-aliased
const int FONT1AA_BM_W = 264;
const int FONT1AA_BM_H = 106;
static const unsigned char s_Font1AA[] = {
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,
    4,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 97,
    206, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    168, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  251, 89,  0,
    0,   89,  255, 125, 89,  255, 125, 0,   0,   0,   0,   7,   199, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   138, 225, 21,  59,  238, 42,  206, 125, 0,   0,   0,   0,
    7,   199, 34,  89,  166, 0,   0,   0,   0,   168, 34,  0,   0,   0,   175,
    255, 255, 166, 0,   0,   7,   202, 89,  0,   0,   0,   0,   59,  245, 255,
    251, 89,  0,   0,   0,   59,  238, 34,  0,   12,  232, 89,  0,   0,   89,
    247, 34,  0,   59,  245, 206, 199, 124, 255, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   202, 89,  0,   12,  235, 255, 247, 34,  0,
    0,   0,   0,   12,  232, 89,  0,   0,   12,  235, 255, 255, 251, 89,  0,
    7,   206, 255, 255, 255, 125, 0,   0,   0,   0,   138, 251, 89,  0,   0,
    59,  245, 255, 255, 255, 251, 89,  0,   0,   89,  255, 255, 166, 0,   89,
    255, 255, 255, 255, 255, 201, 0,   0,   59,  245, 255, 255, 125, 0,   0,
    12,  235, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    89,  255, 255, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   138, 225, 21,
    59,  238, 34,  175, 125, 0,   0,   0,   0,   59,  192, 0,   172, 89,  0,
    0,   59,  245, 255, 255, 251, 89,  89,  247, 34,  12,  228, 34,  0,   138,
    166, 0,   0,   0,   0,   12,  235, 125, 0,   175, 225, 21,  0,   0,   59,
    238, 34,  0,   138, 201, 0,   0,   0,   0,   175, 166, 0,   0,   0,   89,
    255, 201, 0,   0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    215, 21,  0,   175, 166, 0,   138, 201, 0,   0,   7,   206, 255, 251, 89,
    0,   0,   59,  192, 0,   0,   138, 247, 34,  59,  192, 0,   0,   89,  251,
    89,  0,   0,   59,  245, 251, 89,  0,   0,   59,  241, 89,  0,   0,   0,
    0,   0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   7,   206, 166,
    0,   7,   206, 125, 0,   89,  247, 34,  7,   206, 166, 0,   138, 225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 166, 0,   0,   0,   0,   0,   0,   0,   89,  125, 0,   0,   175, 201,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   138, 225, 21,  12,  206, 21,  175, 125, 0,
    0,   89,  255, 255, 255, 255, 255, 255, 166, 59,  241, 89,  168, 34,  138,
    125, 89,  225, 21,  7,   202, 89,  12,  228, 34,  0,   0,   0,   0,   12,
    232, 89,  0,   138, 201, 0,   0,   0,   12,  206, 21,  7,   202, 89,  0,
    0,   0,   0,   59,  215, 21,  59,  245, 206, 199, 124, 255, 125, 0,   0,
    0,   0,   7,   202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   12,  232, 89,  0,
    59,  238, 34,  0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   12,  232, 132, 241,
    89,  0,   0,   59,  241, 89,  0,   0,   0,   0,   7,   206, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   12,  232, 89,  0,   12,
    232, 89,  59,  241, 89,  0,   59,  241, 89,  0,   138, 247, 34,  0,   0,
    138, 247, 34,  0,   0,   0,   0,   0,   12,  235, 247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,
    4,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   172, 89,  59,
    192, 0,   0,   59,  238, 34,  168, 34,  0,   0,   89,  247, 34,  12,  228,
    34,  138, 166, 0,   0,   0,   0,   0,   0,   138, 251, 159, 247, 34,  0,
    0,   0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   7,   202, 89,
    0,   0,   7,   199, 34,  0,   0,   0,   0,   0,   0,   7,   202, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   7,   202, 89,  0,   59,  241, 89,  0,   59,  241, 89,  0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,
    0,   138, 201, 0,   7,   206, 125, 59,  241, 89,  0,   0,   59,  245, 255,
    255, 251, 89,  0,   12,  235, 255, 255, 255, 125, 0,   0,   0,   0,   7,
    206, 166, 0,   0,   0,   175, 251, 89,  138, 201, 0,   59,  241, 89,  0,
    12,  235, 125, 0,   138, 247, 34,  0,   0,   138, 247, 34,  0,   0,   0,
    59,  245, 247, 34,  0,   0,   0,   0,   7,   206, 255, 255, 255, 255, 255,
    255, 125, 0,   0,   0,   0,   138, 255, 201, 0,   0,   0,   0,   0,   0,
    89,  251, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   138, 201, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   12,  206, 21,  138, 125, 0,   0,   7,   206, 255,
    247, 34,  0,   0,   0,   175, 255, 255, 166, 59,  215, 21,  175, 255, 255,
    125, 0,   0,   138, 171, 206, 166, 0,   175, 201, 0,   0,   0,   0,   89,
    201, 0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 255, 255, 255, 255, 255, 255, 125, 0,   0,   0,   0,
    138, 255, 255, 251, 89,  0,   0,   0,   0,   0,   59,  215, 21,  0,   59,
    241, 89,  0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   0,   0,
    0,   0,   12,  235, 166, 0,   0,   0,   138, 255, 255, 125, 0,   175, 201,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   175, 247, 34,  59,  241,
    89,  0,   89,  247, 34,  0,   0,   0,   89,  247, 34,  0,   0,   0,   89,
    255, 255, 255, 125, 0,   12,  235, 166, 0,   59,  245, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   175, 225, 21,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   89,  251, 89,  0,   0,   7,   206, 225, 21,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   89,  201, 0,   0,   0,   0,   0,   0,   0,   12,  235, 255,
    255, 255, 255, 255, 225, 21,  0,   0,   0,   175, 255, 251, 89,  0,   0,
    0,   0,   0,   175, 125, 89,  225, 21,  59,  238, 34,  89,  225, 21,  12,
    235, 166, 175, 166, 0,   0,   0,   0,   89,  201, 0,   0,   0,   0,   0,
    0,   175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   138, 166, 0,   0,   59,  241, 89,  0,   59,  241, 89,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   12,  235, 166, 0,   0,
    0,   0,   0,   0,   59,  241, 97,  206, 255, 255, 255, 255, 255, 125, 0,
    0,   0,   0,   0,   59,  241, 89,  59,  238, 34,  0,   12,  235, 125, 0,
    0,   12,  235, 125, 0,   0,   0,   12,  232, 89,  0,   59,  245, 125, 0,
    89,  255, 255, 232, 241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   59,  245, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 201, 0,   0,   0,
    0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   138, 125, 12,  206, 21,  0,   0,
    0,   0,   0,   168, 34,  175, 166, 0,   0,   0,   0,   59,  215, 21,  138,
    201, 0,   12,  228, 34,  138, 225, 21,  0,   12,  235, 251, 89,  0,   0,
    0,   0,   59,  215, 21,  0,   0,   0,   0,   12,  232, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,
    0,   0,   12,  232, 89,  0,   59,  238, 34,  0,   0,   0,   59,  241, 89,
    0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   0,   12,  235,
    125, 0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   59,  241,
    89,  12,  232, 89,  0,   12,  232, 89,  0,   0,   138, 225, 21,  0,   0,
    0,   59,  238, 34,  0,   7,   206, 166, 0,   0,   0,   0,   89,  225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    235, 247, 34,  0,   0,   7,   206, 255, 255, 255, 255, 255, 255, 125, 0,
    0,   138, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,
    0,   0,   172, 89,  89,  166, 0,   0,   0,   89,  166, 0,   168, 42,  206,
    125, 0,   0,   0,   7,   202, 89,  0,   89,  225, 21,  59,  238, 34,  89,
    251, 89,  0,   0,   175, 255, 201, 0,   0,   0,   0,   7,   202, 89,  0,
    0,   0,   0,   59,  215, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   202, 89,  0,   0,   0,   0,   138, 247, 34,  0,   0,   0,
    0,   0,   7,   206, 201, 0,   12,  228, 34,  0,   0,   0,   175, 166, 0,
    138, 201, 0,   0,   0,   0,   59,  241, 89,  0,   0,   12,  235, 166, 0,
    0,   0,   0,   89,  166, 0,   0,   89,  251, 89,  0,   0,   0,   59,  241,
    89,  0,   0,   59,  192, 0,   0,   175, 225, 21,  0,   175, 201, 0,   138,
    225, 21,  0,   12,  235, 125, 0,   0,   0,   0,   12,  235, 166, 0,   59,
    241, 89,  0,   0,   0,   7,   206, 166, 0,   0,   138, 247, 34,  0,   0,
    59,  245, 125, 0,   0,   0,   0,   0,   0,   0,   12,  232, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 166, 0,   0,   0,   0,
    0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,
    4,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   12,  206, 21,  138, 125,
    0,   0,   0,   12,  235, 255, 255, 255, 166, 0,   0,   0,   0,   138, 201,
    0,   0,   0,   175, 255, 255, 125, 0,   0,   138, 255, 255, 255, 125, 12,
    235, 247, 0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   175, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   0,   7,   206, 201, 0,
    89,  201, 0,   0,   0,   0,   12,  235, 255, 247, 34,  0,   0,   7,   206,
    255, 255, 255, 225, 21,  89,  255, 255, 255, 255, 255, 166, 59,  245, 255,
    255, 251, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   12,  235, 255,
    255, 225, 21,  0,   0,   12,  235, 255, 251, 89,  0,   0,   175, 225, 21,
    0,   0,   0,   0,   0,   59,  245, 255, 255, 125, 0,   0,   89,  255, 255,
    166, 0,   0,   0,   138, 247, 34,  0,   0,   138, 225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206,
    166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    168, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    12,  232, 89,  0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   168, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 89,  255,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   89,  201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  228, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  228, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,
    4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 0,   127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 255, 255, 125, 138, 166, 0,   0,   0,   89,  255, 255,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   89,  255, 255, 255, 255, 166, 0,   0,   0,   0,   0,   12,  235, 225,
    21,  0,   0,   59,  245, 255, 255, 255, 251, 89,  0,   0,   0,   59,  245,
    255, 255, 251, 89,  59,  245, 255, 255, 255, 247, 34,  0,   0,   59,  245,
    255, 255, 255, 255, 127, 81,  245, 255, 255, 255, 255, 127, 0,   0,   59,
    245, 255, 255, 255, 166, 0,   59,  241, 89,  0,   0,   0,   59,  241, 89,
    89,  255, 255, 255, 125, 7,   206, 255, 251, 89,  59,  241, 89,  0,   0,
    89,  255, 166, 59,  241, 89,  0,   0,   0,   0,   59,  245, 225, 21,  0,
    0,   7,   206, 251, 89,  59,  245, 247, 34,  0,   0,   59,  241, 89,  0,
    0,   138, 255, 255, 255, 166, 0,   0,   59,  245, 255, 255, 255, 225, 21,
    0,   0,   0,   138, 255, 255, 255, 166, 0,   0,   59,  245, 255, 255, 255,
    251, 89,  0,   0,   0,   59,  245, 255, 255, 201, 89,  255, 255, 255, 255,
    255, 255, 255, 125, 59,  241, 89,  0,   0,   0,   59,  241, 97,  206, 166,
    0,   0,   0,   0,   175, 201, 175, 201, 0,   0,   7,   206, 201, 0,   0,
    0,   175, 171, 206, 225, 21,  0,   0,   59,  245, 166, 245, 125, 0,   0,
    0,   89,  251, 89,  89,  255, 255, 255, 255, 255, 127, 0,   228, 34,  0,
    0,   59,  215, 21,  0,   0,   0,   0,   12,  228, 34,  0,   0,   0,   59,
    245, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   175, 225, 21,  0,   0,   0,
    175, 225, 21,  0,   0,   0,   89,  232, 241, 89,  0,   0,   59,  241, 89,
    0,   0,   138, 225, 21,  0,   89,  255, 125, 0,   0,   59,  192, 59,  241,
    89,  0,   0,   175, 251, 89,  0,   59,  241, 89,  0,   0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   0,   89,  255, 125, 0,   0,   7,   199, 34,
    59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   59,  241, 89,  0,   0,
    0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,
    0,   0,   0,   0,   59,  245, 255, 125, 0,   0,   89,  255, 251, 89,  59,
    245, 255, 201, 0,   0,   59,  241, 89,  0,   138, 251, 89,  0,   12,  235,
    166, 0,   59,  241, 89,  0,   7,   206, 225, 21,  0,   138, 251, 89,  0,
    12,  235, 166, 0,   59,  241, 89,  0,   0,   138, 247, 34,  0,   12,  235,
    125, 0,   7,   176, 21,  0,   0,   59,  241, 89,  0,   0,   0,   59,  241,
    89,  0,   0,   0,   59,  241, 89,  138, 225, 21,  0,   0,   12,  235, 125,
    89,  225, 21,  0,   59,  245, 247, 34,  0,   12,  232, 89,  12,  235, 166,
    0,   7,   206, 166, 0,   89,  247, 34,  0,   7,   206, 125, 0,   0,   0,
    0,   0,   7,   206, 166, 12,  228, 34,  0,   0,   7,   202, 89,  0,   0,
    0,   0,   12,  228, 34,  0,   0,   12,  235, 133, 206, 166, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   138, 201, 0,   138, 255, 255, 255, 125, 138, 166, 0,   0,   7,
    206, 166, 175, 166, 0,   0,   59,  241, 89,  0,   0,   89,  247, 34,  7,
    206, 166, 0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   175, 225,
    21,  59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,
    7,   206, 166, 0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,
    59,  241, 89,  0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241,
    89,  59,  241, 89,  0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241,
    159, 225, 21,  0,   175, 166, 241, 89,  59,  241, 132, 241, 89,  0,   59,
    241, 89,  12,  235, 166, 0,   0,   0,   89,  247, 34,  59,  241, 89,  0,
    0,   89,  247, 34,  12,  235, 166, 0,   0,   0,   89,  247, 34,  59,  241,
    89,  0,   0,   59,  241, 89,  0,   59,  238, 34,  0,   0,   0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   0,   0,   59,  241,
    89,  59,  241, 89,  0,   0,   89,  225, 21,  59,  241, 89,  0,   89,  206,
    202, 89,  0,   59,  238, 34,  0,   89,  251, 89,  138, 225, 21,  0,   0,
    175, 201, 0,   138, 225, 21,  0,   0,   0,   0,   0,   175, 225, 21,  12,
    228, 34,  0,   0,   0,   138, 166, 0,   0,   0,   0,   12,  228, 34,  0,
    7,   206, 166, 0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 7,   202, 89,  89,  225,
    21,  7,   206, 125, 12,  206, 21,  0,   59,  238, 34,  89,  247, 34,  0,
    59,  241, 89,  0,   0,   175, 201, 0,   59,  241, 89,  0,   0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,
    0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   59,  241,
    89,  0,   0,   0,   59,  241, 89,  59,  241, 102, 232, 89,  0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 102, 232, 89,  59,  215, 81,
    241, 89,  59,  241, 89,  138, 225, 21,  59,  241, 89,  59,  241, 89,  0,
    0,   0,   59,  241, 89,  59,  241, 89,  0,   7,   206, 201, 0,   59,  241,
    89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   175, 201, 0,
    0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  7,   206, 166, 0,   0,
    175, 166, 0,   7,   206, 125, 0,   175, 125, 175, 166, 0,   138, 201, 0,
    0,   0,   175, 255, 251, 89,  0,   0,   0,   59,  245, 166, 241, 89,  0,
    0,   0,   0,   0,   89,  247, 34,  0,   12,  228, 34,  0,   0,   0,   89,
    201, 0,   0,   0,   0,   12,  228, 34,  12,  235, 201, 0,   0,   0,   59,
    245, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 59,  215, 21,  175, 125, 0,   7,   206, 125, 7,   199,
    34,  0,   138, 201, 0,   12,  235, 125, 0,   59,  245, 255, 255, 255, 247,
    34,  0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   59,  241, 89,  59,  245, 255, 255, 255, 255, 127, 59,  245, 255, 255,
    255, 255, 127, 59,  241, 89,  0,   0,   0,   0,   0,   0,   59,  245, 255,
    255, 255, 255, 255, 251, 89,  0,   59,  241, 89,  0,   0,   0,   59,  241,
    89,  59,  245, 255, 247, 34,  0,   0,   0,   59,  241, 89,  0,   0,   0,
    0,   59,  241, 89,  138, 201, 175, 166, 59,  241, 89,  59,  241, 89,  12,
    235, 125, 59,  241, 89,  59,  241, 89,  0,   0,   0,   12,  235, 125, 59,
    245, 255, 255, 255, 201, 0,   0,   59,  241, 89,  0,   0,   0,   12,  235,
    125, 59,  245, 255, 255, 255, 125, 0,   0,   0,   0,   59,  245, 255, 255,
    125, 0,   0,   0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   0,
    0,   59,  241, 89,  0,   138, 225, 21,  59,  241, 89,  0,   0,   175, 201,
    7,   202, 89,  89,  201, 0,   175, 166, 0,   0,   0,   12,  235, 166, 0,
    0,   0,   0,   0,   138, 255, 166, 0,   0,   0,   0,   0,   59,  245, 125,
    0,   0,   12,  228, 34,  0,   0,   0,   12,  228, 34,  0,   0,   0,   12,
    228, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 59,  215,
    21,  175, 125, 0,   7,   206, 125, 7,   199, 34,  7,   206, 125, 0,   0,
    175, 201, 0,   59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,
    0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241,
    89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241, 89,
    0,   59,  245, 255, 251, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,
    0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  175, 225,
    21,  0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  12,  235,
    247, 34,  59,  241, 89,  59,  241, 89,  0,   89,  247, 94,  241, 89,  59,
    241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  12,  235,
    166, 0,   0,   0,   0,   0,   0,   0,   138, 251, 89,  0,   0,   59,  241,
    89,  0,   0,   0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   12,
    232, 89,  138, 225, 21,  0,   0,   89,  225, 81,  215, 21,  12,  228, 47,
    232, 89,  0,   0,   0,   175, 255, 251, 89,  0,   0,   0,   0,   59,  241,
    89,  0,   0,   0,   0,   7,   206, 201, 0,   0,   0,   12,  228, 34,  0,
    0,   0,   0,   175, 125, 0,   0,   0,   12,  228, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 12,  228, 34,  89,  201, 0,   7,   206,
    125, 59,  215, 21,  59,  245, 255, 255, 255, 255, 247, 34,  59,  241, 89,
    0,   0,   59,  241, 89,  7,   206, 166, 0,   0,   0,   0,   0,   59,  241,
    89,  0,   0,   0,   138, 225, 21,  59,  241, 89,  0,   0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   59,  241, 89,
    59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   59,  241, 89,  0,   0,
    0,   59,  241, 89,  59,  241, 89,  7,   206, 201, 0,   0,   59,  241, 89,
    0,   0,   0,   0,   59,  241, 89,  0,   175, 166, 0,   59,  241, 89,  59,
    241, 89,  0,   7,   206, 200, 241, 89,  12,  235, 166, 0,   0,   0,   89,
    247, 34,  59,  241, 89,  0,   0,   0,   0,   0,   12,  235, 166, 0,   0,
    0,   89,  247, 34,  59,  241, 89,  0,   59,  245, 125, 0,   0,   0,   0,
    0,   0,   12,  232, 89,  0,   0,   59,  241, 89,  0,   0,   0,   12,  232,
    89,  0,   0,   0,   59,  238, 34,  0,   0,   175, 171, 206, 166, 0,   0,
    0,   12,  232, 159, 201, 0,   7,   202, 132, 215, 21,  0,   0,   89,  247,
    34,  175, 225, 21,  0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   138,
    225, 21,  0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   89,  201, 0,
    0,   0,   12,  228, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   138, 201, 7,   206, 255, 251, 226, 255, 255, 166, 0,   138, 201,
    0,   0,   0,   12,  235, 125, 59,  241, 89,  0,   0,   138, 247, 34,  0,
    89,  255, 125, 0,   0,   59,  192, 59,  241, 89,  0,   0,   138, 251, 89,
    0,   59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,
    0,   89,  255, 125, 0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,
    59,  241, 89,  0,   59,  241, 89,  0,   0,   0,   89,  247, 34,  59,  241,
    89,  0,   12,  235, 166, 0,   59,  241, 89,  0,   0,   0,   0,   59,  241,
    89,  0,   0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   59,  245,
    251, 89,  0,   138, 251, 89,  0,   59,  245, 166, 0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   138, 251, 89,  0,   59,  245, 166, 0,   59,  241,
    89,  0,   0,   138, 251, 89,  0,   89,  166, 0,   0,   89,  247, 34,  0,
    0,   59,  241, 89,  0,   0,   0,   0,   138, 225, 21,  0,   7,   206, 166,
    0,   0,   0,   89,  255, 251, 89,  0,   0,   0,   7,   206, 255, 125, 0,
    0,   138, 255, 201, 0,   0,   12,  235, 125, 0,   12,  235, 166, 0,   0,
    0,   59,  241, 89,  0,   0,   0,   89,  251, 89,  0,   0,   0,   0,   12,
    228, 34,  0,   0,   0,   0,   12,  228, 34,  0,   0,   12,  228, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   7,   206, 225, 21,
    0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,   175, 201,
    59,  245, 255, 255, 255, 247, 34,  0,   0,   0,   59,  245, 255, 255, 251,
    89,  59,  245, 255, 255, 255, 225, 21,  0,   0,   59,  245, 255, 255, 255,
    255, 127, 81,  241, 89,  0,   0,   0,   0,   0,   0,   59,  245, 255, 255,
    255, 201, 0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  89,  255, 255,
    255, 138, 235, 255, 255, 125, 0,   59,  241, 89,  0,   0,   89,  255, 201,
    59,  245, 255, 255, 255, 255, 166, 59,  241, 89,  0,   0,   0,   0,   59,
    241, 89,  59,  241, 89,  0,   0,   0,   175, 251, 89,  0,   0,   138, 255,
    255, 255, 166, 0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,
    138, 255, 255, 255, 166, 0,   0,   59,  241, 89,  0,   0,   0,   175, 251,
    89,  12,  235, 255, 255, 251, 89,  0,   0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   59,  245, 255, 251, 89,  0,   0,   0,   0,   12,  235, 201,
    0,   0,   0,   0,   0,   138, 251, 89,  0,   0,   89,  255, 125, 0,   7,
    206, 225, 21,  0,   0,   89,  255, 125, 0,   0,   59,  241, 89,  0,   0,
    0,   175, 255, 255, 255, 255, 255, 127, 0,   228, 34,  0,   0,   0,   0,
    0,   175, 125, 0,   0,   12,  228, 34,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   89,  255, 255, 255, 255, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  228, 34,  0,   0,   0,   0,   0,   89,  201, 0,   0,   12,
    228, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   138, 255, 255, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 255,
    125, 0,   0,   0,   12,  228, 124, 255, 255, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   245, 255, 255, 255, 255, 255, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   89,  255, 255, 166, 0,   0,   0,   0,   0,   0,   0,   59,  241, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   175, 255, 201, 0,   12,  228, 34,  0,   0,   89,  255, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,
    4,   4,   4,   4,   4,   4,   4,   4,   116, 116, 4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241,
    89,  0,   89,  251, 89,  59,  241, 89,  0,   0,   0,   0,   59,  241, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  225, 21,  0,
    0,   12,  228, 34,  0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,
    4,   28,  244, 252, 52,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   12,  228, 34,  0,   0,
    0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   4,   4,   4,   4,   4,   4,   4,   4,   180, 252, 164, 4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 255, 255, 125, 0,
    59,  241, 194, 255, 251, 89,  0,   0,   7,   206, 255, 255, 201, 0,   12,
    235, 255, 255, 251, 89,  0,   12,  235, 255, 251, 89,  7,   206, 255, 255,
    247, 34,  0,   12,  235, 255, 255, 251, 89,  59,  241, 194, 255, 255, 125,
    0,   59,  241, 89,  89,  255, 251, 89,  59,  241, 89,  0,   138, 251, 89,
    59,  241, 89,  59,  241, 159, 255, 255, 125, 89,  255, 255, 166, 0,   59,
    241, 194, 255, 255, 125, 0,   0,   0,   12,  235, 255, 247, 34,  0,   59,
    241, 194, 255, 255, 125, 0,   0,   12,  235, 255, 255, 251, 89,  59,  241,
    159, 255, 201, 0,   138, 255, 255, 247, 34,  206, 255, 255, 255, 166, 59,
    241, 89,  0,   59,  241, 97,  206, 166, 0,   0,   12,  235, 125, 175, 201,
    0,   7,   206, 166, 0,   7,   206, 133, 206, 225, 21,  0,   89,  255, 255,
    166, 0,   0,   12,  235, 125, 138, 255, 255, 255, 255, 166, 0,   0,   138,
    166, 0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   89,  225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,
    4,   4,   4,   76,  252, 244, 20,  4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  247, 34,  59,  245, 166, 0,   138, 225,
    21,  7,   206, 201, 0,   0,   0,   7,   206, 166, 0,   59,  241, 89,  7,
    206, 125, 0,   89,  225, 21,  59,  241, 89,  0,   0,   7,   206, 166, 0,
    59,  241, 89,  59,  245, 166, 0,   89,  247, 34,  59,  241, 89,  0,   59,
    241, 89,  59,  241, 89,  138, 225, 21,  0,   59,  241, 89,  59,  245, 201,
    0,   89,  255, 201, 0,   89,  247, 34,  59,  245, 166, 0,   89,  247, 34,
    0,   7,   206, 166, 0,   138, 225, 21,  59,  245, 166, 0,   138, 247, 34,
    7,   206, 166, 0,   59,  241, 89,  59,  245, 201, 0,   0,   59,  238, 34,
    0,   130, 34,  59,  241, 89,  0,   0,   59,  241, 89,  0,   59,  241, 89,
    89,  247, 34,  0,   89,  247, 34,  138, 225, 21,  12,  235, 225, 21,  12,
    232, 89,  7,   206, 166, 12,  235, 125, 89,  247, 34,  0,   89,  247, 34,
    0,   0,   0,   89,  247, 34,  0,   0,   138, 166, 0,   0,   0,   12,  228,
    34,  0,   0,   0,   0,   89,  225, 21,  0,   0,   7,   206, 247, 34,  0,
    0,   89,  201, 0,   0,   4,   4,   68,  12,  4,   4,   4,   220, 252, 108,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   0,
    0,   59,  241, 89,  0,   59,  241, 89,  59,  238, 34,  0,   59,  238, 34,
    59,  241, 89,  0,   0,   59,  241, 89,  0,   59,  241, 89,  59,  241, 89,
    0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 159, 201,
    0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,
    241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,  0,   59,
    241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241,
    89,  59,  241, 89,  0,   0,   59,  241, 89,  0,   0,   0,   59,  241, 89,
    0,   0,   59,  241, 89,  0,   59,  241, 89,  12,  235, 125, 0,   175, 166,
    0,   59,  238, 34,  89,  171, 202, 89,  89,  225, 21,  0,   59,  241, 226,
    201, 0,   12,  235, 125, 0,   175, 166, 0,   0,   0,   12,  235, 125, 0,
    0,   59,  238, 34,  0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   7,
    206, 125, 0,   7,   202, 89,  12,  235, 166, 0,   175, 125, 0,   0,   4,
    60,  244, 172, 4,   4,   132, 252, 212, 4,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   89,  255, 255, 255, 251, 89,  59,  241, 89,
    0,   59,  241, 89,  59,  238, 34,  0,   0,   0,   59,  238, 34,  0,   59,
    241, 89,  59,  245, 255, 255, 255, 251, 0,   59,  241, 89,  0,   0,   59,
    238, 34,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241,
    89,  0,   59,  241, 89,  59,  245, 255, 225, 21,  0,   0,   59,  241, 89,
    59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,
    59,  241, 89,  0,   59,  238, 34,  0,   12,  232, 89,  59,  241, 89,  0,
    12,  232, 89,  59,  238, 34,  0,   59,  241, 89,  59,  241, 89,  0,   0,
    0,   175, 255, 255, 201, 0,   59,  241, 89,  0,   0,   59,  241, 89,  0,
    59,  241, 89,  0,   175, 201, 12,  232, 89,  0,   7,   206, 125, 172, 89,
    138, 166, 138, 201, 0,   0,   0,   138, 247, 34,  0,   0,   175, 201, 12,
    232, 89,  0,   0,   7,   206, 166, 0,   0,   175, 225, 21,  0,   0,   0,
    0,   12,  228, 34,  0,   0,   0,   0,   0,   0,   175, 225, 34,  206, 21,
    0,   0,   175, 255, 166, 0,   0,   0,   4,   52,  244, 252, 140, 36,  244,
    252, 60,  4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   59,
    241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  238, 34,  59,  241,
    89,  0,   0,   0,   59,  241, 89,  0,   59,  241, 89,  59,  238, 34,  0,
    0,   0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   59,  241, 89,
    59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,
    241, 97,  206, 201, 0,   0,   59,  241, 89,  59,  241, 89,  0,   59,  241,
    89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,  241,
    89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,
    0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   0,   0,   59,  245, 125,
    59,  241, 89,  0,   0,   59,  241, 89,  0,   59,  241, 89,  0,   59,  238,
    124, 225, 21,  0,   0,   175, 176, 206, 21,  59,  215, 187, 125, 0,   0,
    59,  245, 255, 201, 0,   0,   89,  247, 124, 225, 21,  0,   0,   138, 225,
    21,  0,   0,   0,   59,  241, 89,  0,   0,   0,   12,  228, 34,  0,   0,
    0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   4,   4,   76,  252, 252, 220, 252, 164, 4,   4,   4,   4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   89,  251, 89,
    59,  241, 89,  0,   175, 201, 0,   7,   206, 201, 0,   0,   0,   7,   206,
    166, 0,   138, 251, 89,  7,   206, 166, 0,   7,   199, 34,  59,  241, 89,
    0,   0,   7,   206, 166, 0,   138, 251, 89,  59,  241, 89,  0,   59,  241,
    89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  12,  235, 166, 0,
    59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,  59,
    241, 89,  0,   59,  241, 89,  0,   7,   206, 166, 0,   138, 225, 21,  59,
    241, 89,  0,   138, 225, 21,  7,   206, 166, 0,   89,  251, 89,  59,  241,
    89,  0,   0,   89,  125, 0,   12,  232, 89,  12,  232, 89,  0,   12,  12,
    235, 125, 0,   175, 251, 89,  0,   7,   206, 255, 125, 0,   0,   0,   89,
    255, 201, 0,   7,   206, 247, 34,  0,   7,   206, 166, 59,  245, 125, 0,
    7,   206, 255, 125, 0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   138,
    166, 0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   89,  225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   100,
    252, 252, 244, 28,  4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   175, 255, 255, 232, 241, 89,  59,  245, 255, 255, 247, 34,
    0,   0,   12,  235, 255, 255, 201, 0,   59,  245, 255, 200, 241, 89,  0,
    12,  235, 255, 251, 89,  0,   59,  241, 89,  0,   0,   0,   12,  235, 255,
    200, 241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,
    241, 89,  59,  241, 89,  0,   59,  245, 201, 59,  241, 89,  59,  241, 89,
    0,   59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,
    0,   0,   12,  235, 255, 247, 34,  0,   59,  245, 166, 255, 247, 34,  0,
    0,   59,  245, 255, 166, 241, 89,  59,  241, 89,  0,   0,   59,  245, 255,
    255, 166, 0,   0,   138, 255, 255, 125, 0,   89,  255, 255, 166, 241, 89,
    0,   0,   138, 247, 34,  0,   0,   0,   59,  245, 125, 0,   0,   138, 225,
    21,  7,   206, 225, 21,  0,   138, 251, 0,   0,   138, 247, 34,  0,   0,
    175, 255, 255, 255, 255, 166, 0,   0,   138, 166, 0,   0,   0,   12,  228,
    34,  0,   0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   4,   4,   4,   4,   132, 252, 108, 4,   4,   4,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   59,  241,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   138, 201, 0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   138,
    201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,
    4,   4,   4,   4,   116, 4,   4,   4,   4,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    166, 255, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   255, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 201,
    0,   12,  228, 34,  0,   0,   89,  255, 251, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,
    4,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  138,
    201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 125, 59,  238, 34,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  255, 225, 21,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 251,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   59,  238, 34,  138, 201, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   199, 34,  0,   0,   0,   0,   0,   7,   199, 34,
    0,   0,   0,   0,   138, 255, 201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   251, 89,  0,   0,   138, 255, 251, 97,  206, 201, 0,   0,   138,
    251, 102, 235, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,
    255, 201, 12,  228, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   206, 166, 12,  232, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   175, 166, 12,  235, 127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    175, 255, 255, 255, 225, 21,  59,  245, 255, 255, 255, 255, 255, 125, 0,
    0,   0,   0,   0,   0,   0,   7,   206, 255, 247, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   199,
    34,  0,   0,   0,   0,   0,   7,   199, 34,  0,   0,   0,   138, 225, 21,
    175, 166, 0,   0,   175, 255, 255, 166, 0,   0,   7,   202, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  245, 255, 255, 201, 0,   0,
    0,   0,   0,   0,   0,   0,   59,  245, 255, 255, 255, 255, 255, 255, 255,
    255, 125, 0,   59,  245, 255, 255, 255, 255, 255, 125, 0,   0,   89,  255,
    255, 255, 255, 255, 225, 21,  59,  245, 255, 255, 255, 255, 255, 125, 0,
    0,   0,   59,  245, 255, 255, 255, 255, 255, 125, 7,   206, 166, 0,   0,
    175, 171, 206, 166, 89,  247, 34,  0,   175, 201, 59,  241, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  228, 34,  175, 255, 125, 0,   0,
    89,  255, 255, 255, 125, 175, 251, 89,  89,  255, 125, 0,   7,   206, 255,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  245, 255, 255, 255, 255, 255, 125, 0,
    0,   7,   206, 255, 125, 59,  245, 125, 0,   0,   0,   89,  251, 89,  0,
    0,   0,   0,   0,   0,   0,   127, 7,   206, 225, 21,  0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   0,   0,
    175, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  245, 255, 255, 255, 255, 125, 0,   59,  245,
    255, 255, 255, 255, 125, 0,   0,   0,   0,   0,   0,   0,   89,  247, 34,
    12,  228, 34,  0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  235, 125, 0,   7,   176, 21,  0,   0,   0,   0,   0,   0,   138,
    251, 89,  0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   59,  115, 0,
    0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   0,   7,   206, 166, 0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   59,  115, 0,   0,
    0,   0,   59,  115, 0,   89,  201, 0,   12,  232, 89,  89,  201, 7,   202,
    89,  12,  232, 89,  138, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   199, 34,  0,   172,
    132, 196, 199, 163, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   0,   0,
    89,  247, 34,  0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    127, 89,  247, 34,  0,   0,   0,   0,   0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   7,   199, 34,  0,   0,   0,   0,   0,   7,   199, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  225, 21,  7,   202, 89,  12,  228, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  0,   0,
    0,   0,   0,   0,   0,   130, 34,  59,  241, 89,  0,   0,   0,   138, 201,
    0,   0,   0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,
    0,   0,   0,   0,   0,   175, 225, 21,  0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   12,
    228, 34,  59,  192, 0,   12,  228, 34,  138, 166, 59,  215, 21,  175, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   199, 34,  0,   172, 89,  175, 166, 138, 125, 0,
    138, 255, 255, 247, 34,  12,  146, 0,   0,   0,   0,   89,  255, 255, 255,
    125, 12,  235, 255, 255, 125, 0,   0,   0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   138, 255, 255, 255, 255, 127, 0,   175, 201, 0,   138, 225,
    21,  0,   0,   0,   0,   0,   0,   0,   0,   127, 245, 255, 255, 255, 255,
    255, 125, 0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,
    0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   199, 34,  0,   0,
    0,   0,   0,   7,   199, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    89,  247, 34,  12,  228, 34,  138, 166, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   175, 225,
    21,  138, 225, 21,  0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   89,  247,
    34,  0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   59,
    115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   199,
    34,  0,   172, 89,  0,   0,   138, 125, 59,  238, 34,  0,   130, 34,  7,
    206, 201, 0,   0,   59,  241, 89,  0,   12,  235, 255, 125, 0,   59,  241,
    89,  0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,
    89,  247, 34,  0,   59,  245, 166, 241, 89,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 138, 225, 21,  0,   0,   0,   0,   0,   59,  115, 0,
    0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   89,  255, 255, 255, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   199, 34,  0,   0,   0,   0,   0,   7,   199, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 255, 255, 166, 59,
    215, 21,  175, 255, 255, 125, 0,   89,  255, 255, 201, 0,   0,   0,   59,
    245, 255, 255, 125, 0,   12,  235, 166, 0,   0,   138, 225, 21,  0,   0,
    0,   138, 255, 255, 255, 255, 247, 34,  0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  245, 125, 0,   0,   0,   59,  115, 0,
    0,   0,   0,   59,  115, 0,   0,   0,   59,  115, 0,   0,   0,   0,   59,
    115, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   59,  245, 255, 251, 102, 0,   255, 255, 255, 255, 255,
    0,   245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 127, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   138, 247, 34,  138, 201,
    0,   0,   0,   175, 201, 0,   0,   0,   175, 166, 0,   0,   59,  115, 0,
    0,   0,   0,   59,  115, 0,   0,   0,   12,  235, 125, 0,   0,   0,   138,
    255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 245, 255,
    255, 255, 255, 225, 21,  0,   59,  115, 0,   0,   0,   0,   59,  115, 0,
    0,   0,   0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   199,
    34,  0,   0,   0,   59,  245, 255, 255, 255, 255, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 89,  225, 21,  59,  238,
    47,  232, 89,  7,   206, 125, 0,   0,   0,   0,   0,   138, 251, 89,  12,
    235, 166, 0,   0,   138, 225, 21,  0,   0,   0,   138, 201, 0,   0,   0,
    0,   0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   7,
    206, 201, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,
    0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  245,
    255, 251, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 255, 255,
    201, 0,   0,   0,   138, 247, 34,  175, 201, 0,   0,   0,   138, 255, 255,
    255, 255, 255, 166, 0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,
    0,   7,   206, 166, 0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 89,  251, 89,  0,   0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   0,   0,   138,
    201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   7,   199, 34,  0,   0,   0,   0,   0,
    7,   199, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   59,  215, 21,  138, 201, 0,   12,  228, 47,  228, 34,  0,   175, 166,
    0,   0,   0,   0,   0,   12,  232, 89,  0,   0,   175, 225, 21,  59,  241,
    89,  0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   59,  115, 0,
    0,   0,   0,   59,  115, 0,   0,   0,   138, 225, 21,  0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   0,   59,  115, 0,   0,
    0,   0,   59,  115, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 166, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  245, 125, 7,   206, 201, 0,
    0,   138, 201, 0,   0,   0,   175, 201, 0,   0,   0,   0,   0,   0,   0,
    59,  115, 0,   0,   0,   0,   59,  115, 0,   0,   138, 225, 21,  0,   0,
    0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 7,   206, 247, 34,  0,   0,   0,   0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   59,  245, 125, 0,   0,   175, 166, 0,   0,   0,   59,  245,
    125, 175, 225, 29,  206, 166, 0,   89,  247, 34,  7,   206, 166, 0,   0,
    0,   7,   199, 34,  0,   0,   0,   0,   0,   7,   199, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  0,   89,  225,
    21,  59,  238, 47,  232, 89,  7,   206, 125, 0,   89,  166, 0,   0,   89,
    247, 34,  0,   0,   0,   130, 34,  0,   138, 255, 125, 0,   0,   138, 201,
    0,   0,   0,   0,   0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,
    0,   89,  251, 89,  0,   0,   0,   0,   0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   0,   0,   59,  115, 0,   0,   0,   0,   59,  115, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,
    125, 0,   12,  232, 89,  12,  146, 0,   0,   0,   59,  241, 89,  0,   12,
    235, 247, 34,  0,   0,   89,  125, 0,   0,   59,  115, 0,   0,   0,   0,
    59,  115, 0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   175, 255, 255,
    255, 225, 21,  59,  245, 255, 255, 255, 255, 255, 125, 0,   138, 225, 21,
    0,   12,  235, 125, 0,   0,   0,   138, 225, 34,  235, 125, 7,   206, 166,
    0,   89,  247, 34,  7,   206, 166, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   138, 201, 0,   0,   0,   175, 255, 255, 125, 0,   89,  255,
    255, 201, 0,   0,   12,  235, 255, 255, 251, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   89,  255, 255, 255, 255, 255, 255, 255, 255, 255, 125, 0,
    59,  245, 255, 255, 255, 255, 255, 125, 0,   0,   175, 255, 255, 255, 255,
    255, 247, 34,  59,  245, 255, 255, 255, 255, 255, 125, 0,   0,   0,   59,
    245, 255, 255, 255, 255, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  245, 255, 255, 166, 0,   0,
    0,   0,   0,   0,   0,   89,  255, 255, 255, 125, 59,  245, 255, 255, 201,
    0,   0,   0,   59,  245, 255, 255, 255, 255, 255, 125, 0,   175, 255, 255,
    255, 255, 127, 0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   175, 166, 0,   255, 255, 201, 0,   0,   0,
    0,   175, 166, 59,  238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    228, 34,  0,   0,   0,   0,   0,   0,   0,   12,  228, 34,  138, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    245, 255, 255, 255, 255, 255, 251, 127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  228, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   138, 225, 21,  0,   0,   0,   138, 125, 0,   0,   0,   0,
    59,  245, 255, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   138, 225,
    21,  0,   0,   175, 166, 0,   12,  228, 34,  0,   0,   59,  245, 255, 255,
    247, 34,  0,   89,  225, 29,  206, 166, 0,   0,   0,   0,   0,   89,  255,
    255, 255, 255, 125, 0,   0,   0,   7,   206, 255, 255, 247, 34,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 255, 255, 255, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 255,
    166, 0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   0,   0,   12,  235,
    255, 125, 0,   0,   175, 255, 255, 225, 21,  0,   0,   0,   12,  235, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 255, 255,
    255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  228, 34,  0,   0,   89,  255, 255, 225, 21,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   89,  225, 21,
    0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   89,  225, 21,  0,   0,
    0,   12,  235, 255, 255, 166, 0,   0,   7,   206, 125, 0,   0,   0,   0,
    0,   0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   138, 225, 21,
    0,   0,   0,   138, 125, 0,   0,   0,   12,  235, 125, 0,   59,  115, 0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   59,  215, 21,  0,
    12,  228, 34,  0,   12,  235, 125, 0,   0,   168, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   175, 225, 21,  0,   0,   0,   175, 225, 21,
    0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   175, 225, 21,  0,   0,   0,   175, 225, 21,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  238, 34,  7,   206, 125, 0,   0,   0,   7,
    202, 89,  0,   0,   0,   0,   7,   199, 34,  59,  238, 34,  0,   0,   0,
    7,   202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  255, 255, 255, 125, 175, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 247, 34,  0,   59,
    241, 89,  0,   175, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 255, 166, 0,   0,   12,  232, 89,  0,   0,   0,   0,   7,   206, 255,
    166, 0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   59,  215,
    21,  0,   89,  201, 0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 255, 251,
    89,  0,   59,  241, 89,  0,   0,   0,   138, 201, 0,   0,   0,   138, 201,
    0,   0,   89,  225, 21,  175, 125, 0,   0,   12,  228, 34,  0,   12,  235,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    166, 0,   89,  255, 255, 247, 34,  89,  201, 0,   0,   89,  255, 255, 255,
    166, 0,   0,   0,   0,   168, 34,  7,   151, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 7,   206, 255,
    255, 225, 21,  89,  201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   89,
    166, 0,   0,   138, 166, 0,   0,   0,   7,   202, 89,  0,   0,   0,   0,
    0,   0,   0,   59,  238, 34,  0,   7,   206, 255, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  238, 34,  0,   0,   175, 166, 0,   175, 255,
    255, 255, 125, 175, 125, 0,   138, 247, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   12,  228, 34,  0,   89,  201, 0,   0,   89,  225, 0,
    81,  115, 0,   134, 89,  0,   0,   0,   0,   0,   138, 166, 0,   0,   138,
    166, 0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   0,   138, 166, 0,
    0,   0,   0,   0,   0,   59,  245, 247, 34,  0,   12,  232, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    89,  201, 0,   7,   206, 201, 138, 125, 138, 125, 0,   59,  241, 89,  0,
    0,   0,   0,   175, 255, 255, 255, 225, 21,  0,   0,   7,   206, 166, 215,
    21,  0,   0,   12,  228, 34,  0,   0,   138, 255, 255, 251, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  206, 21,  59,  241, 89,  0,   134,
    89,  0,   172, 89,  59,  238, 34,  0,   138, 166, 0,   0,   7,   206, 201,
    12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  206, 21,  7,   202, 89,  12,  235, 125, 0,   172, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  7,   206, 125, 12,
    235, 255, 255, 255, 255, 255, 255, 125, 0,   0,   0,   12,  235, 125, 0,
    0,   0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   0,   0,   59,
    238, 34,  0,   0,   175, 166, 0,   175, 255, 255, 255, 125, 175, 125, 0,
    138, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  228,
    34,  0,   89,  201, 0,   0,   89,  225, 0,   29,  206, 166, 59,  245, 125,
    0,   0,   0,   0,   138, 166, 0,   12,  228, 34,  0,   175, 225, 21,  0,
    0,   0,   138, 166, 0,   12,  228, 42,  206, 255, 255, 166, 0,   0,   0,
    0,   12,  228, 34,  138, 166, 0,   89,  247, 34,  0,   0,   0,   0,   59,
    238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   138, 201, 0,   59,  241, 89,
    138, 125, 0,   0,   59,  245, 255, 255, 255, 125, 0,   0,   138, 166, 0,
    89,  201, 0,   0,   0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,
    0,   7,   206, 125, 0,   138, 251, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   89,  166, 0,   138, 201, 0,   0,   0,   0,   0,   89,  166, 59,  215,
    21,  0,   175, 166, 0,   59,  245, 125, 89,  251, 89,  0,   0,   12,  235,
    255, 255, 255, 255, 255, 255, 125, 138, 255, 255, 251, 89,  127, 166, 0,
    7,   202, 89,  12,  232, 89,  0,   89,  166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   138, 255, 255, 166, 0,   0,   0,   0,   7,   202, 89,  0,
    0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   7,   206, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  0,   0,   175, 166,
    0,   89,  255, 255, 255, 125, 175, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  228, 34,  0,   59,  241, 89,  0,
    175, 201, 0,   0,   0,   175, 201, 7,   206, 201, 0,   0,   0,   138, 166,
    0,   175, 166, 0,   138, 200, 215, 21,  0,   0,   0,   138, 166, 0,   175,
    166, 7,   151, 0,   89,  247, 34,  0,   0,   0,   59,  238, 47,  228, 34,
    59,  219, 209, 34,  0,   0,   0,   89,  255, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   138, 225, 21,  59,  238, 34,  138, 125, 0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   138, 166, 0,   89,  201, 0,   0,   59,  245,
    255, 255, 255, 255, 125, 0,   0,   0,   0,   0,   59,  238, 34,  0,   7,
    206, 125, 0,   0,   0,   0,   0,   0,   0,   0,   89,  166, 0,   138, 201,
    0,   0,   0,   0,   0,   89,  166, 0,   175, 255, 255, 223, 166, 0,   12,
    235, 125, 59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   175,
    125, 0,   0,   0,   0,   0,   138, 125, 0,   7,   206, 255, 255, 125, 0,
    0,   59,  157, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   0,   7,   206, 255,
    255, 255, 166, 7,   206, 255, 255, 201, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  238, 34,  0,   0,   175, 166, 0,   0,   89,  255, 255, 125,
    175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    89,  255, 255, 251, 89,  0,   89,  255, 255, 225, 21,  0,   0,   0,   175,
    225, 29,  206, 166, 0,   0,   0,   138, 166, 59,  215, 21,  59,  215, 81,
    215, 21,  0,   0,   0,   138, 166, 59,  215, 21,  0,   0,   0,   89,  225,
    21,  59,  245, 255, 255, 125, 138, 166, 7,   202, 97,  199, 34,  0,   0,
    89,  251, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   138, 225, 21,
    59,  241, 89,  138, 125, 0,   0,   0,   89,  247, 34,  0,   0,   0,   0,
    175, 255, 255, 255, 225, 21,  0,   0,   0,   12,  232, 89,  0,   0,   0,
    12,  228, 34,  0,   12,  235, 225, 21,  59,  215, 21,  0,   0,   0,   0,
    0,   0,   0,   0,   12,  206, 21,  59,  241, 89,  0,   134, 89,  0,   172,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 201, 12,  235, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,   0,
    12,  206, 21,  7,   202, 89,  7,   206, 125, 0,   172, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  0,
    0,   175, 166, 0,   0,   0,   0,   175, 125, 175, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   7,   206, 166, 59,  245, 125, 0,   0,   0,
    0,   0,   0,   175, 125, 12,  228, 34,  59,  215, 21,  0,   0,   0,   0,
    0,   175, 125, 0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   59,
    238, 34,  175, 125, 7,   199, 34,  0,   0,   175, 201, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   138, 225, 21,  7,   206, 201, 138, 125, 138,
    125, 7,   202, 89,  0,   0,   0,   0,   138, 201, 0,   0,   0,   138, 166,
    0,   0,   0,   12,  232, 89,  0,   0,   0,   12,  228, 34,  0,   0,   0,
    175, 255, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    166, 0,   89,  255, 255, 247, 34,  89,  201, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   168, 34,  7,   151, 0,   0,   0,   0,   0,   0,
    0,   0,   175, 125, 0,   0,   0,   0,   0,   0,   138, 166, 7,   202, 89,
    0,   89,  247, 124, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   12,  235, 255, 255, 255, 255, 255, 255, 125, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  245, 125, 0,   7,   206, 166, 0,   0,   0,
    0,   175, 125, 175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  115, 0,   134, 89,  0,   0,   0,   0,   0,   0,   59,  215, 21,  59,
    245, 255, 255, 255, 225, 21,  0,   0,   0,   59,  215, 21,  0,   0,   59,
    238, 34,  0,   0,   0,   0,   0,   0,   175, 125, 7,   206, 255, 255, 255,
    251, 89,  0,   138, 247, 34,  0,   59,  157, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    138, 225, 21,  0,   7,   206, 255, 255, 251, 89,  138, 255, 255, 255, 255,
    255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  232, 89,
    0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 225, 21,  0,   0,   0,
    175, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   175, 225, 21,  0,   0,   0,   175, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    238, 198, 255, 251, 194, 166, 0,   0,   0,   0,   175, 125, 175, 125, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,   89,  225, 21,  0,
    0,   0,   7,   206, 125, 0,   0,   12,  235, 255, 255, 255, 166, 0,   0,
    0,   89,  225, 21,  0,   0,   0,   12,  228, 34,  0,   0,   0,   175, 255,
    255, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    138, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  228, 34,
    0,   7,   176, 21,  0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  255, 255, 255, 255, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    89,  255, 255, 255, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   175, 125, 175, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   59,  215, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   12,  228, 34,  0,   7,   206, 255, 255, 251,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   175, 125,
    175, 125, 0,   0,   0,   0,   0,   0,   59,  245, 251, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   12,  235, 201,
    0,   0,   0,   0,   12,  235, 251, 89,  0,   0,   0,   0,   175, 255, 125,
    89,  201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  245,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 0,
    0,   0,   0,   0,   0,   138, 251, 89,  0,   0,   0,   12,  235, 251, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,
    89,  255, 125, 0,   89,  255, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 247, 34,  172, 89,
    0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    89,  255, 125, 0,   0,   0,   0,   0,   0,   89,  255, 225, 21,  0,   0,
    0,   0,   12,  235, 247, 34,  172, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 0,
    0,   0,   0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   89,
    255, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   89,  247,
    34,  0,   0,   0,   0,   0,   175, 166, 0,   0,   0,   0,   7,   206, 125,
    59,  241, 89,  0,   0,   89,  201, 12,  235, 247, 34,  0,   0,   7,   206,
    166, 59,  241, 89,  0,   0,   12,  228, 34,  59,  215, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   89,  247,
    34,  0,   0,   0,   7,   206, 125, 59,  241, 89,  0,   7,   206, 166, 59,
    238, 34,  0,   0,   175, 166, 0,   0,   59,  241, 89,  0,   89,  247, 34,
    138, 201, 59,  238, 34,  138, 201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   175, 125, 89,  255, 201, 0,   0,   0,   0,   0,   0,   0,
    175, 201, 0,   0,   0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   0,
    0,   0,   59,  238, 34,  138, 225, 21,  0,   0,   0,   175, 125, 89,  255,
    201, 0,   0,   0,   0,   59,  238, 34,  138, 201, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   0,   0,
    138, 201, 0,   0,   0,   0,   0,   59,  238, 34,  138, 225, 21,  0,   0,
    0,   59,  238, 34,  138, 201, 0,   0,   0,   0,   12,  232, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    12,  228, 34,  59,  215, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 255, 255, 125, 0,   0,   0,   0,   127, 0,   0,
    12,  235, 225, 21,  0,   0,   0,   0,   12,  235, 225, 21,  0,   0,   0,
    0,   12,  235, 225, 21,  0,   0,   0,   0,   12,  235, 225, 21,  0,   0,
    0,   0,   12,  235, 225, 21,  0,   0,   0,   0,   12,  235, 225, 21,  0,
    0,   0,   0,   0,   175, 255, 255, 255, 255, 255, 255, 255, 166, 0,   0,
    138, 255, 255, 255, 251, 89,  59,  245, 255, 255, 255, 255, 127, 81,  245,
    255, 255, 255, 255, 225, 21,  59,  245, 255, 255, 255, 255, 127, 81,  245,
    255, 255, 255, 255, 127, 111, 255, 255, 255, 125, 89,  255, 255, 255, 125,
    89,  255, 255, 255, 125, 89,  255, 255, 255, 125, 7,   206, 255, 255, 255,
    255, 125, 0,   0,   59,  245, 247, 34,  0,   0,   59,  241, 89,  0,   0,
    0,   138, 255, 255, 255, 166, 0,   0,   0,   0,   138, 255, 255, 255, 166,
    0,   0,   0,   0,   0,   138, 255, 255, 255, 166, 0,   0,   0,   0,   138,
    255, 255, 255, 166, 0,   0,   0,   0,   138, 255, 255, 255, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255,
    255, 255, 210, 235, 166, 59,  241, 89,  0,   0,   0,   59,  241, 89,  59,
    241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,
    241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 132, 245, 125, 0,   0,
    0,   89,  251, 89,  12,  232, 89,  0,   0,   0,   0,   7,   206, 166, 0,
    89,  251, 89,  0,   0,   0,   127, 0,   0,   89,  232, 241, 89,  0,   0,
    0,   0,   89,  232, 241, 89,  0,   0,   0,   0,   89,  232, 241, 89,  0,
    0,   0,   0,   89,  232, 241, 89,  0,   0,   0,   0,   89,  232, 241, 89,
    0,   0,   0,   0,   89,  232, 241, 89,  0,   0,   0,   0,   12,  232, 89,
    89,  225, 21,  0,   0,   0,   0,   0,   175, 247, 34,  0,   0,   59,  192,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   0,
    59,  241, 89,  0,   7,   206, 166, 0,   0,   59,  245, 201, 0,   59,  245,
    255, 201, 0,   0,   59,  241, 89,  0,   0,   138, 251, 89,  0,   12,  235,
    166, 0,   0,   138, 251, 89,  0,   12,  235, 166, 0,   0,   0,   138, 251,
    89,  0,   12,  235, 166, 0,   0,   138, 251, 89,  0,   12,  235, 166, 0,
    0,   138, 251, 89,  0,   12,  235, 166, 0,   0,   0,   138, 166, 0,   0,
    0,   12,  228, 34,  0,   0,   175, 247, 34,  0,   0,   175, 225, 21,  59,
    241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,
    241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,
    0,   0,   59,  241, 89,  89,  247, 34,  0,   7,   206, 125, 0,   12,  232,
    89,  0,   0,   0,   0,   59,  241, 89,  0,   12,  232, 89,  0,   0,   0,
    127, 0,   7,   206, 166, 175, 166, 0,   0,   0,   7,   206, 166, 175, 166,
    0,   0,   0,   7,   206, 166, 175, 166, 0,   0,   0,   7,   206, 166, 175,
    166, 0,   0,   0,   7,   206, 166, 175, 166, 0,   0,   0,   0,   175, 166,
    175, 166, 0,   0,   0,   0,   138, 225, 21,  89,  225, 21,  0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   59,
    241, 89,  0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   7,   206,
    166, 0,   0,   0,   59,  245, 125, 59,  241, 132, 241, 89,  0,   59,  241,
    89,  0,   12,  235, 166, 0,   0,   0,   89,  247, 34,  12,  235, 166, 0,
    0,   0,   89,  247, 34,  0,   12,  235, 166, 0,   0,   0,   89,  247, 34,
    12,  235, 166, 0,   0,   0,   89,  247, 34,  12,  235, 166, 0,   0,   0,
    89,  247, 34,  0,   0,   12,  235, 125, 0,   12,  235, 125, 0,   0,   59,
    241, 89,  0,   0,   138, 176, 235, 166, 59,  241, 89,  0,   0,   0,   59,
    241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,
    0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  0,
    175, 201, 0,   138, 225, 21,  0,   12,  235, 255, 255, 255, 225, 21,  59,
    238, 34,  0,   138, 225, 21,  0,   0,   0,   127, 0,   59,  238, 34,  89,
    247, 34,  0,   0,   59,  238, 34,  89,  247, 34,  0,   0,   59,  238, 34,
    89,  247, 34,  0,   0,   59,  238, 34,  89,  247, 34,  0,   0,   59,  238,
    34,  89,  247, 34,  0,   0,   59,  241, 89,  89,  225, 21,  0,   0,   7,
    206, 125, 0,   89,  225, 21,  0,   0,   0,   0,   138, 225, 21,  0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   0,   59,  241,
    89,  0,   0,   59,  241, 89,  0,   7,   206, 166, 0,   0,   0,   7,   206,
    166, 59,  241, 89,  138, 225, 21,  59,  241, 89,  0,   59,  241, 89,  0,
    0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  0,
    59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,
    59,  241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   0,   0,
    12,  235, 138, 235, 125, 0,   0,   0,   138, 225, 21,  0,   59,  215, 21,
    175, 201, 59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,
    0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  59,
    241, 89,  0,   0,   0,   59,  241, 89,  0,   59,  245, 166, 241, 89,  0,
    0,   12,  232, 89,  0,   0,   175, 225, 59,  238, 47,  235, 225, 21,  0,
    0,   0,   0,   127, 0,   138, 201, 0,   12,  235, 125, 0,   0,   138, 201,
    0,   12,  235, 125, 0,   0,   138, 201, 0,   12,  235, 125, 0,   0,   138,
    201, 0,   12,  235, 125, 0,   0,   138, 201, 0,   12,  235, 125, 0,   0,
    138, 225, 21,  12,  235, 125, 0,   0,   89,  247, 34,  0,   89,  255, 255,
    255, 255, 251, 89,  138, 225, 21,  0,   0,   0,   0,   0,   59,  245, 255,
    255, 255, 255, 127, 59,  245, 255, 255, 255, 255, 166, 0,   59,  245, 255,
    255, 255, 255, 127, 59,  245, 255, 255, 255, 255, 127, 0,   59,  241, 89,
    0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   0,   59,  241, 89,
    7,   206, 255, 255, 255, 166, 0,   0,   175, 201, 59,  241, 89,  12,  235,
    125, 59,  241, 89,  0,   59,  241, 89,  0,   0,   0,   12,  235, 125, 59,
    241, 89,  0,   0,   0,   12,  235, 125, 0,   59,  241, 89,  0,   0,   0,
    12,  235, 125, 59,  241, 89,  0,   0,   0,   12,  235, 125, 59,  241, 89,
    0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   12,  235, 125, 0,   0,
    0,   0,   138, 225, 21,  7,   199, 34,  0,   138, 225, 81,  241, 89,  0,
    0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  59,
    241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,
    241, 89,  0,   0,   138, 255, 166, 0,   0,   0,   12,  232, 89,  0,   0,
    89,  247, 59,  238, 34,  0,   59,  245, 125, 0,   0,   0,   127, 7,   206,
    125, 0,   0,   175, 201, 0,   7,   206, 125, 0,   0,   175, 201, 0,   7,
    206, 125, 0,   0,   175, 201, 0,   7,   206, 125, 0,   0,   175, 201, 0,
    7,   206, 125, 0,   0,   175, 201, 0,   7,   206, 125, 0,   0,   175, 201,
    0,   7,   206, 255, 255, 255, 255, 225, 21,  0,   0,   0,   0,   138, 225,
    21,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241,
    89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  241,
    89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,
    0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   7,   206, 166, 0,   0,
    0,   7,   206, 166, 59,  241, 89,  0,   89,  247, 94,  241, 89,  0,   59,
    241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,
    241, 89,  0,   59,  241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,
    0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,
    0,   0,   0,   12,  235, 138, 235, 125, 0,   0,   0,   138, 225, 21,  175,
    125, 0,   0,   175, 201, 59,  241, 89,  0,   0,   0,   59,  241, 89,  59,
    241, 89,  0,   0,   0,   59,  241, 89,  59,  241, 89,  0,   0,   0,   59,
    241, 89,  59,  241, 89,  0,   0,   0,   59,  241, 89,  0,   0,   59,  241,
    89,  0,   0,   0,   12,  232, 89,  0,   7,   206, 201, 59,  238, 34,  0,
    0,   138, 201, 0,   0,   0,   127, 59,  245, 255, 255, 255, 255, 247, 34,
    59,  245, 255, 255, 255, 255, 247, 34,  59,  245, 255, 255, 255, 255, 247,
    34,  59,  245, 255, 255, 255, 255, 247, 34,  59,  245, 255, 255, 255, 255,
    247, 34,  59,  245, 255, 255, 255, 255, 247, 34,  59,  241, 89,  0,   0,
    89,  225, 21,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    59,  241, 89,  0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   0,
    59,  241, 89,  0,   7,   206, 166, 0,   0,   0,   59,  241, 89,  59,  241,
    89,  0,   7,   206, 200, 241, 89,  0,   12,  235, 166, 0,   0,   0,   89,
    247, 34,  12,  235, 166, 0,   0,   0,   89,  247, 34,  0,   12,  235, 166,
    0,   0,   0,   89,  247, 34,  12,  235, 166, 0,   0,   0,   89,  247, 34,
    12,  235, 166, 0,   0,   0,   89,  247, 34,  0,   0,   12,  235, 125, 0,
    12,  235, 125, 0,   0,   59,  241, 159, 166, 0,   0,   12,  235, 166, 12,
    232, 89,  0,   0,   0,   59,  238, 34,  12,  232, 89,  0,   0,   0,   59,
    238, 34,  12,  232, 89,  0,   0,   0,   59,  238, 34,  12,  232, 89,  0,
    0,   0,   59,  238, 34,  0,   0,   59,  241, 89,  0,   0,   0,   12,  235,
    255, 255, 255, 201, 0,   59,  238, 34,  0,   0,   138, 201, 0,   0,   0,
    127, 138, 201, 0,   0,   0,   12,  235, 125, 138, 201, 0,   0,   0,   12,
    235, 125, 138, 201, 0,   0,   0,   12,  235, 125, 138, 201, 0,   0,   0,
    12,  235, 125, 138, 201, 0,   0,   0,   12,  235, 125, 138, 201, 0,   0,
    0,   12,  235, 125, 175, 201, 0,   0,   0,   89,  225, 21,  0,   0,   0,
    0,   0,   175, 247, 34,  0,   0,   59,  192, 59,  241, 89,  0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   0,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   59,
    241, 89,  0,   0,   59,  241, 89,  0,   0,   59,  241, 89,  0,   7,   206,
    166, 0,   0,   59,  245, 201, 0,   59,  241, 89,  0,   0,   59,  245, 251,
    89,  0,   0,   138, 251, 89,  0,   59,  245, 166, 0,   0,   138, 251, 89,
    0,   59,  245, 166, 0,   0,   0,   138, 251, 89,  0,   59,  245, 166, 0,
    0,   138, 251, 89,  0,   59,  245, 166, 0,   0,   138, 251, 89,  0,   59,
    245, 166, 0,   0,   0,   138, 166, 0,   0,   0,   12,  228, 34,  0,   0,
    175, 247, 34,  0,   7,   206, 225, 21,  0,   138, 225, 21,  0,   7,   206,
    166, 0,   0,   138, 225, 21,  0,   7,   206, 166, 0,   0,   138, 225, 21,
    0,   7,   206, 166, 0,   0,   138, 225, 21,  0,   7,   206, 166, 0,   0,
    0,   59,  241, 89,  0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   59,
    238, 34,  0,   12,  235, 125, 0,   0,   0,   127, 206, 125, 0,   0,   0,
    0,   175, 206, 206, 125, 0,   0,   0,   0,   175, 206, 206, 125, 0,   0,
    0,   0,   175, 206, 206, 125, 0,   0,   0,   0,   175, 206, 206, 125, 0,
    0,   0,   0,   175, 206, 206, 125, 0,   0,   0,   0,   175, 232, 245, 125,
    0,   0,   0,   89,  255, 255, 255, 255, 255, 166, 0,   0,   138, 255, 255,
    255, 251, 89,  59,  245, 255, 255, 255, 255, 127, 81,  245, 255, 255, 255,
    255, 225, 21,  59,  245, 255, 255, 255, 255, 127, 81,  245, 255, 255, 255,
    255, 127, 111, 255, 255, 255, 125, 89,  255, 255, 255, 125, 89,  255, 255,
    255, 125, 89,  255, 255, 255, 125, 7,   206, 255, 255, 255, 255, 125, 0,
    0,   59,  241, 89,  0,   0,   0,   175, 251, 89,  0,   0,   0,   138, 255,
    255, 255, 166, 0,   0,   0,   0,   138, 255, 255, 255, 166, 0,   0,   0,
    0,   0,   138, 255, 255, 255, 166, 0,   0,   0,   0,   138, 255, 255, 255,
    166, 0,   0,   0,   0,   138, 255, 255, 255, 166, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 194, 255, 255, 255, 201,
    0,   0,   0,   0,   59,  245, 255, 251, 89,  0,   0,   0,   0,   59,  245,
    255, 251, 89,  0,   0,   0,   0,   59,  245, 255, 251, 89,  0,   0,   0,
    0,   59,  245, 255, 251, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   12,  232, 89,  0,   0,   0,   0,   59,  238, 47,  235, 255, 166, 0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   12,  228, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    12,  235, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  235, 255, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   12,
    235, 201, 0,   0,   0,   138, 255, 201, 0,   0,   0,   59,  245, 225, 29,
    202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 7,   202, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   12,  235,
    201, 0,   0,   0,   89,  255, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   175, 247, 34,  0,   12,  235, 255, 255, 166, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   59,  245, 225, 29,  202, 89,  0,   0,
    138, 251, 89,  0,   0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,
    89,  255, 225, 21,  0,   0,   59,  245, 225, 29,  202, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,
    0,   0,   89,  255, 125, 0,   0,   0,   89,  255, 225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 251, 89,  0,   59,  238,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   175, 166, 0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   89,  225,
    21,  175, 201, 0,   7,   202, 89,  138, 255, 166, 0,   0,   89,  247, 34,
    175, 166, 0,   0,   138, 166, 7,   202, 89,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 166, 0,   0,   0,   0,   0,   175, 166, 0,   0,   0,   89,  247, 34,
    138, 201, 0,   0,   89,  247, 34,  175, 201, 0,   0,   138, 201, 0,   175,
    200, 215, 34,  235, 247, 47,  232, 0,   138, 255, 225, 111, 225, 21,  0,
    0,   172, 89,  138, 255, 166, 0,   0,   0,   0,   89,  225, 21,  0,   0,
    0,   0,   175, 201, 0,   0,   0,   0,   89,  247, 34,  138, 201, 0,   0,
    172, 89,  138, 255, 166, 0,   0,   59,  238, 34,  138, 201, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   175, 166, 0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   89,  247, 34,  138, 201, 0,   0,   0,   59,  238, 34,  138, 201, 0,
    0,   0,   89,  247, 34,  0,   0,   59,  238, 34,  0,   0,   0,   0,   0,
    138, 225, 29,  206, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235,
    255, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   89,  166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 7,   206, 255, 255, 255, 125, 0,   7,   206, 255, 255, 255, 125, 0,
    7,   206, 255, 255, 255, 125, 0,   7,   206, 255, 255, 255, 125, 0,   7,
    206, 255, 255, 255, 125, 0,   12,  235, 255, 255, 251, 89,  0,   12,  235,
    255, 255, 251, 89,  59,  245, 255, 166, 0,   0,   59,  245, 255, 255, 201,
    0,   12,  235, 255, 251, 89,  0,   0,   12,  235, 255, 251, 89,  0,   0,
    12,  235, 255, 251, 89,  0,   0,   12,  235, 255, 251, 89,  0,   59,  238,
    34,  59,  238, 34,  59,  238, 34,  59,  238, 34,  0,   59,  241, 89,  175,
    225, 21,  0,   59,  241, 194, 255, 255, 125, 0,   0,   12,  235, 255, 247,
    34,  0,   0,   12,  235, 255, 247, 34,  0,   0,   0,   12,  235, 255, 247,
    34,  0,   0,   12,  235, 255, 247, 34,  0,   0,   12,  235, 255, 247, 34,
    0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   7,   206, 255,
    255, 225, 21,  0,   59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,
    59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,  0,
    59,  241, 97,  206, 166, 0,   0,   12,  235, 125, 59,  238, 163, 255, 255,
    201, 7,   206, 166, 0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   89,
    247, 34,  0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   89,  247,
    34,  0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   89,  247, 34,
    0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   89,  255, 225, 21,
    0,   138, 201, 59,  245, 125, 0,   0,   0,   7,   206, 125, 0,   89,  225,
    21,  7,   206, 125, 0,   89,  225, 21,  7,   206, 125, 0,   89,  225, 21,
    7,   206, 125, 0,   89,  225, 21,  59,  238, 34,  59,  238, 34,  59,  238,
    34,  59,  238, 34,  0,   0,   0,   0,   12,  235, 125, 0,   59,  245, 166,
    0,   89,  247, 34,  7,   206, 166, 0,   138, 225, 21,  7,   206, 166, 0,
    138, 225, 21,  0,   7,   206, 166, 0,   138, 225, 21,  7,   206, 166, 0,
    138, 225, 21,  7,   206, 166, 0,   138, 225, 21,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   12,  232, 89,  0,   138, 251, 89,  0,   59,  241,
    89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,
    0,   59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,  89,  247, 34,
    0,   89,  247, 34,  59,  245, 166, 0,   7,   206, 166, 89,  247, 34,  0,
    89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   59,  238,
    34,  0,   0,   0,   0,   12,  232, 89,  0,   0,   59,  238, 127, 225, 21,
    0,   0,   0,   59,  238, 34,  0,   59,  238, 34,  59,  238, 34,  0,   59,
    238, 34,  59,  238, 34,  0,   59,  238, 34,  59,  238, 34,  0,   59,  238,
    34,  59,  238, 34,  59,  238, 34,  59,  238, 34,  59,  238, 34,  0,   138,
    255, 255, 255, 255, 201, 0,   59,  241, 89,  0,   59,  241, 89,  59,  241,
    89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,  241,
    89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,
    0,   59,  241, 89,  0,   12,  235, 255, 255, 255, 255, 255, 255, 166, 138,
    201, 0,   59,  157, 175, 201, 0,   59,  241, 89,  0,   59,  241, 89,  59,
    241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  0,   59,
    241, 89,  0,   59,  241, 89,  12,  235, 125, 0,   175, 166, 0,   59,  238,
    34,  0,   0,   138, 225, 34,  235, 125, 0,   175, 166, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   89,
    255, 255, 255, 251, 89,  0,   89,  255, 255, 255, 251, 89,  0,   89,  255,
    255, 255, 251, 89,  0,   89,  255, 255, 255, 251, 89,  0,   89,  255, 255,
    255, 251, 89,  0,   138, 255, 255, 255, 247, 34,  0,   175, 255, 255, 255,
    255, 255, 255, 255, 255, 251, 127, 201, 0,   0,   0,   0,   59,  245, 255,
    255, 255, 251, 89,  59,  245, 255, 255, 255, 251, 89,  59,  245, 255, 255,
    255, 251, 89,  59,  245, 255, 255, 255, 251, 89,  59,  238, 34,  59,  238,
    34,  59,  238, 34,  59,  238, 34,  138, 247, 34,  0,   0,   138, 201, 0,
    59,  241, 89,  0,   59,  241, 89,  59,  238, 34,  0,   12,  232, 89,  59,
    238, 34,  0,   12,  232, 89,  0,   59,  238, 34,  0,   12,  232, 89,  59,
    238, 34,  0,   12,  232, 89,  59,  238, 34,  0,   12,  232, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   175, 201, 7,   176, 21,  138, 201,
    0,   59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,
    59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,  0,   59,  241, 89,
    0,   175, 201, 12,  232, 89,  0,   59,  238, 34,  0,   0,   138, 225, 21,
    175, 201, 12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 59,  241, 89,  0,   59,  241, 89,  59,
    241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241,
    89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,  138, 247, 34,
    0,   59,  238, 34,  138, 225, 21,  0,   12,  232, 89,  0,   0,   0,   0,
    138, 201, 0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   59,  238,
    34,  0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   59,  238, 34,
    0,   0,   0,   0,   59,  238, 34,  59,  238, 34,  59,  238, 34,  59,  238,
    34,  175, 201, 0,   0,   0,   138, 166, 0,   59,  241, 89,  0,   59,  241,
    89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,
    0,   59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241, 89,
    59,  241, 89,  0,   59,  241, 89,  0,   0,   0,   0,   12,  235, 166, 0,
    0,   0,   138, 201, 134, 89,  0,   175, 166, 0,   59,  241, 89,  0,   59,
    241, 89,  59,  241, 89,  0,   59,  241, 89,  59,  241, 89,  0,   59,  241,
    89,  0,   59,  241, 89,  0,   59,  241, 89,  0,   89,  247, 124, 225, 21,
    0,   59,  238, 34,  0,   0,   138, 201, 0,   89,  247, 124, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 89,  247, 34,  0,   89,  251, 89,  89,  247, 34,  0,   89,  251, 89,
    89,  247, 34,  0,   89,  251, 89,  89,  247, 34,  0,   89,  251, 89,  89,
    247, 34,  0,   89,  251, 89,  175, 201, 0,   0,   175, 247, 34,  175, 201,
    0,   0,   59,  245, 225, 21,  0,   7,   199, 94,  245, 125, 0,   0,   0,
    7,   206, 166, 0,   7,   199, 34,  7,   206, 166, 0,   7,   199, 34,  7,
    206, 166, 0,   7,   199, 34,  7,   206, 166, 0,   7,   199, 34,  59,  238,
    34,  59,  238, 34,  59,  238, 34,  59,  238, 34,  138, 247, 34,  0,   12,
    232, 89,  0,   59,  241, 89,  0,   59,  241, 89,  7,   206, 166, 0,   138,
    225, 21,  7,   206, 166, 0,   138, 225, 21,  0,   7,   206, 166, 0,   138,
    225, 21,  7,   206, 166, 0,   138, 225, 21,  7,   206, 166, 0,   138, 225,
    21,  0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   59,  245, 166, 0,
    59,  241, 89,  0,   12,  235, 125, 0,   175, 251, 89,  12,  235, 125, 0,
    175, 251, 89,  12,  235, 125, 0,   175, 251, 89,  0,   12,  235, 125, 0,
    175, 251, 89,  0,   7,   206, 255, 125, 0,   0,   59,  238, 34,  0,   12,
    235, 125, 0,   7,   206, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   175, 255, 255, 232,
    241, 89,  0,   175, 255, 255, 232, 241, 89,  0,   175, 255, 255, 232, 241,
    89,  0,   175, 255, 255, 232, 241, 89,  0,   175, 255, 255, 232, 241, 89,
    12,  235, 255, 255, 166, 238, 34,  12,  235, 255, 255, 225, 21,  89,  255,
    255, 251, 89,  0,   89,  255, 255, 255, 201, 0,   12,  235, 255, 251, 89,
    0,   0,   12,  235, 255, 251, 89,  0,   0,   12,  235, 255, 251, 89,  0,
    0,   12,  235, 255, 251, 89,  0,   59,  238, 34,  59,  238, 34,  59,  238,
    34,  59,  238, 34,  0,   138, 255, 255, 255, 125, 0,   0,   59,  241, 89,
    0,   59,  241, 89,  0,   12,  235, 255, 247, 34,  0,   0,   12,  235, 255,
    247, 34,  0,   0,   0,   12,  235, 255, 247, 34,  0,   0,   12,  235, 255,
    247, 34,  0,   0,   12,  235, 255, 247, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   206, 255, 255, 225, 21,  0,   0,   0,   89,
    255, 255, 166, 241, 89,  0,   89,  255, 255, 166, 241, 89,  0,   89,  255,
    255, 166, 241, 89,  0,   0,   89,  255, 255, 166, 241, 89,  0,   0,   138,
    247, 34,  0,   0,   59,  245, 166, 255, 255, 166, 0,   0,   0,   138, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  215, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   59,  238,
    34,  0,   0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 125, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   138, 225, 21,  0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   0,
    138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 0,   127, 127, 0,   127, 127, 0,   127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0};

// Large font anti-aliased
const int FONT2AA_BM_W = 276;
const int FONT2AA_BM_H = 120;
static const unsigned char s_Font2AA[] = {
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   0,   4,
    4,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   4,   4,   0,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   59,  245, 125, 175, 225, 21,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,   12,
    235, 201, 89,  255, 166, 0,   0,   0,   0,   0,   172, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   0,   4,   4,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   138, 247, 34,  0,   12,  232,
    89,  138, 225, 21,  0,   0,   0,   0,   138, 125, 7,   199, 34,  0,   0,
    0,   0,   138, 125, 0,   0,   0,   0,   138, 255, 255, 201, 0,   0,   0,
    59,  215, 21,  0,   0,   0,   0,   59,  245, 255, 255, 166, 0,   0,   0,
    59,  241, 89,  0,   7,   206, 201, 0,   0,   89,  251, 89,  0,   59,  215,
    21,  172, 89,  59,  192, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   175, 166, 0,   0,   138, 255, 251, 89,  0,   0,   0,   0,   0,
    138, 201, 0,   0,   0,   7,   206, 255, 255, 255, 166, 0,   0,   7,   206,
    255, 255, 255, 201, 0,   0,   0,   0,   0,   0,   138, 251, 89,  0,   0,
    175, 255, 255, 255, 255, 225, 21,  0,   0,   12,  235, 255, 255, 125, 89,
    255, 255, 255, 255, 255, 251, 89,  0,   12,  235, 255, 255, 225, 21,  0,
    0,   59,  245, 255, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  245, 255, 255, 251, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   0,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    89,  247, 34,  0,   12,  232, 89,  138, 201, 0,   0,   0,   0,   7,   202,
    89,  59,  215, 21,  0,   0,   12,  235, 255, 255, 255, 166, 0,   59,  241,
    89,  12,  235, 125, 0,   0,   172, 89,  0,   0,   0,   0,   7,   206, 166,
    0,   89,  251, 89,  0,   0,   12,  228, 34,  0,   89,  247, 34,  0,   0,
    0,   175, 201, 0,   0,   89,  251, 191, 194, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  232, 89,  0,   175, 201, 0,   12,
    235, 125, 0,   0,   138, 255, 255, 201, 0,   0,   0,   12,  182, 0,   0,
    59,  245, 125, 0,   12,  206, 21,  0,   12,  235, 166, 0,   0,   0,   0,
    89,  255, 251, 89,  0,   0,   175, 201, 0,   0,   0,   0,   0,   0,   89,
    255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   89,  251, 89,  12,  235,
    166, 0,   7,   206, 201, 0,   59,  245, 125, 0,   12,  235, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   89,  166, 0,   0,   138, 251, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   4,   4,   4,   0,   4,   4,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   89,  247, 34,  0,   12,  228, 34,  89,  201,
    0,   0,   0,   0,   12,  206, 21,  89,  166, 0,   0,   12,  235, 125, 138,
    125, 59,  192, 0,   89,  247, 34,  7,   206, 166, 0,   89,  201, 0,   0,
    0,   0,   0,   12,  235, 125, 0,   12,  232, 89,  0,   0,   12,  228, 34,
    0,   175, 201, 0,   0,   0,   0,   59,  241, 89,  0,   0,   7,   206, 166,
    0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  225,
    21,  59,  241, 89,  0,   0,   138, 225, 21,  0,   0,   0,   175, 201, 0,
    0,   0,   0,   0,   0,   0,   7,   206, 201, 0,   0,   0,   0,   0,   0,
    175, 201, 0,   0,   0,   59,  241, 132, 241, 89,  0,   0,   175, 201, 0,
    0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   0,   0,   0,
    7,   206, 201, 0,   59,  241, 89,  0,   0,   138, 225, 21,  138, 225, 21,
    0,   0,   138, 225, 21,  89,  255, 125, 0,   0,   89,  255, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   0,   4,
    4,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   89,  247, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 255, 255, 255, 255, 255,
    255, 125, 59,  238, 34,  138, 125, 0,   0,   0,   89,  247, 34,  7,   206,
    166, 7,   202, 89,  0,   0,   0,   0,   0,   0,   175, 225, 21,  138, 225,
    21,  0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   7,   206,
    125, 0,   89,  251, 191, 194, 247, 34,  0,   0,   0,   0,   0,   138, 166,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   138, 166, 0,   89,  247, 34,  0,   0,   89,  247, 34,
    0,   0,   0,   175, 201, 0,   0,   0,   0,   0,   0,   0,   12,  235, 166,
    0,   0,   0,   0,   0,   59,  245, 125, 0,   0,   12,  235, 125, 59,  241,
    89,  0,   0,   175, 201, 0,   0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   12,  235, 201, 0,   0,
    175, 201, 0,   138, 225, 21,  0,   0,   89,  247, 34,  89,  255, 125, 0,
    0,   89,  255, 125, 0,   0,   0,   0,   0,   12,  235, 255, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 255, 251,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   138, 247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   4,   4,   0,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   175, 125, 7,   199, 34,  0,   0,   12,  235, 166, 138, 125, 0,   0,
    0,   59,  241, 89,  12,  235, 125, 89,  201, 12,  235, 255, 251, 89,  0,
    0,   7,   206, 255, 166, 0,   59,  241, 89,  0,   0,   0,   59,  238, 34,
    0,   0,   0,   0,   0,   175, 166, 59,  215, 21,  172, 89,  59,  192, 0,
    0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   138, 247,
    34,  0,   0,   89,  247, 34,  0,   0,   0,   175, 201, 0,   0,   0,   0,
    0,   0,   0,   89,  251, 89,  0,   0,   0,   89,  255, 247, 34,  0,   0,
    7,   206, 166, 0,   59,  241, 89,  0,   0,   175, 255, 255, 255, 225, 21,
    0,   89,  251, 226, 255, 255, 247, 34,  0,   0,   0,   7,   206, 166, 0,
    0,   0,   12,  235, 255, 255, 201, 0,   0,   89,  255, 125, 0,   0,   138,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255,
    255, 166, 0,   0,   0,   0,   0,   175, 255, 255, 255, 255, 255, 255, 225,
    21,  0,   0,   0,   0,   59,  245, 255, 201, 0,   0,   0,   0,   0,   175,
    251, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   0,   4,   4,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   89,  225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   7,   199, 34,  59,  215, 21,  0,   0,   0,
    59,  245, 255, 255, 201, 0,   0,   0,   138, 255, 255, 201, 12,  228, 34,
    175, 166, 0,   138, 201, 0,   12,  235, 125, 89,  255, 125, 59,  241, 89,
    0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   138, 201, 0,   0,
    0,   172, 89,  0,   0,   0,   7,   206, 255, 255, 255, 255, 255, 255, 247,
    34,  0,   0,   0,   0,   89,  255, 255, 255, 166, 0,   0,   0,   0,   0,
    59,  238, 34,  0,   138, 247, 34,  0,   0,   89,  247, 34,  0,   0,   0,
    175, 201, 0,   0,   0,   0,   0,   0,   59,  245, 166, 0,   0,   0,   0,
    0,   0,   12,  235, 166, 0,   138, 201, 0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   12,  235, 201, 0,   138, 251, 89,  0,   0,   175, 225, 21,
    0,   0,   89,  247, 34,  0,   0,   7,   206, 166, 0,   175, 255, 166, 0,
    0,   89,  255, 255, 255, 223, 247, 34,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   175, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206,
    225, 21,  0,   0,   175, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   0,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    59,  215, 21,  0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 255, 255,
    255, 255, 255, 166, 0,   0,   0,   0,   138, 125, 175, 225, 21,  0,   0,
    0,   0,   0,   138, 166, 7,   206, 125, 0,   89,  247, 34,  138, 225, 21,
    0,   89,  255, 166, 215, 21,  0,   0,   0,   59,  238, 34,  0,   0,   0,
    0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   138, 201, 0,   0,   89,  247, 34,  0,   0,
    89,  247, 34,  0,   0,   0,   175, 201, 0,   0,   0,   0,   0,   12,  235,
    201, 0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,  175, 255, 255,
    255, 255, 255, 255, 125, 0,   0,   0,   0,   0,   138, 247, 34,  89,  247,
    34,  0,   0,   59,  241, 89,  0,   7,   206, 166, 0,   0,   0,   138, 247,
    34,  0,   0,   138, 247, 34,  0,   0,   0,   0,   0,   138, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 255, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  245, 255, 201, 0,   0,   0,   0,   175, 201, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   4,   4,   4,   0,   4,   4,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   89,  166, 0,   175, 125, 0,   0,   0,   0,   0,   0,   138,
    125, 89,  247, 34,  0,   0,   0,   0,   12,  228, 34,  7,   206, 125, 0,
    89,  247, 34,  138, 247, 34,  0,   0,   89,  255, 166, 0,   0,   0,   0,
    59,  238, 34,  0,   0,   0,   0,   0,   175, 166, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,
    0,   59,  241, 89,  0,   0,   138, 225, 21,  0,   0,   0,   175, 201, 0,
    0,   0,   0,   12,  235, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    138, 225, 21,  0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    0,   138, 225, 21,  59,  241, 89,  0,   0,   59,  241, 89,  0,   89,  247,
    34,  0,   0,   0,   138, 247, 34,  0,   0,   89,  251, 89,  0,   0,   0,
    0,   7,   206, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 255, 225, 21,  0,   0,   175, 255, 255, 255, 255,
    255, 255, 225, 21,  0,   0,   175, 255, 251, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   0,   4,
    4,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   89,  247, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 7,   199, 34,  0,
    0,   0,   89,  201, 0,   138, 125, 175, 201, 0,   0,   0,   0,   0,   138,
    166, 0,   0,   175, 166, 0,   138, 201, 0,   89,  255, 166, 0,   0,   89,
    255, 255, 125, 0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   7,   206,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 166,
    0,   0,   0,   0,   0,   138, 255, 125, 0,   0,   0,   0,   0,   0,   175,
    247, 34,  59,  238, 34,  0,   0,   0,   175, 201, 0,   12,  235, 125, 0,
    0,   0,   0,   175, 201, 0,   0,   0,   12,  235, 166, 0,   0,   0,   0,
    0,   89,  166, 0,   0,   59,  245, 166, 0,   0,   0,   0,   0,   59,  241,
    89,  0,   59,  215, 21,  0,   12,  235, 166, 0,   7,   206, 201, 0,   0,
    175, 225, 21,  7,   206, 166, 0,   0,   0,   0,   59,  245, 166, 0,   7,
    206, 225, 21,  0,   0,   0,   0,   175, 225, 21,  0,   89,  255, 125, 0,
    0,   12,  235, 201, 0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 201, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   175, 225, 21,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   4,   4,   0,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   7,
    199, 34,  59,  215, 21,  0,   0,   0,   12,  235, 255, 255, 255, 201, 0,
    0,   0,   0,   0,   59,  215, 21,  0,   0,   12,  235, 255, 251, 89,  0,
    0,   89,  255, 255, 255, 201, 0,   89,  255, 0,   0,   0,   0,   175, 201,
    0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 201, 0,   0,
    0,   0,   0,   0,   0,   175, 247, 34,  138, 201, 0,   0,   0,   0,   0,
    138, 255, 251, 89,  0,   0,   0,   138, 255, 255, 255, 255, 166, 0,   89,
    255, 255, 255, 255, 255, 247, 34,  12,  235, 255, 255, 255, 166, 0,   0,
    0,   0,   0,   0,   59,  241, 89,  0,   12,  235, 255, 255, 255, 166, 0,
    0,   0,   7,   206, 255, 255, 225, 21,  0,   138, 247, 34,  0,   0,   0,
    0,   0,   59,  245, 255, 255, 201, 0,   0,   0,   175, 255, 255, 201, 0,
    0,   0,   89,  255, 125, 0,   0,   89,  251, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 225,
    21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,   4,
    52,  4,   4,   4,   4,   4,   4,   4,   4,   4,   0,   4,   4,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   138, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   175, 201, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 201,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   0,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   138, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 201, 0,   0,
    89,  251, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   89,  201, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   4,   4,   4,   4,   4,   4,   0,   4,   4,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   201, 0,   0,   201, 201, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,
    4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   0,   0,   0,   0,   0,   0,   127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   4,   4,   4,   4,   4,   4,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 255,
    201, 0,   138, 201, 0,   0,   0,   0,   89,  255, 255, 255, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   175, 255, 255, 255, 247, 34,  0,   0,   0,   0,
    0,   7,   206, 251, 89,  0,   0,   12,  0,   235, 255, 255, 255, 255, 201,
    0,   0,   0,   0,   59,  245, 255, 255, 255, 201, 12,  0,   235, 255, 255,
    255, 255, 166, 0,   0,   0,   12,  235, 255, 255, 255, 255, 255, 127, 12,
    235, 255, 255, 255, 255, 251, 89,  0,   0,   12,  235, 255, 255, 255, 251,
    89,  12,  235, 166, 0,   0,   0,   12,  235, 125, 89,  255, 255, 255, 201,
    0,   0,   175, 255, 255, 225, 21,  12,  235, 166, 0,   0,   7,   206, 251,
    102, 0,   235, 166, 0,   0,   0,   0,   12,  235, 251, 89,  0,   0,   0,
    89,  255, 225, 21,  12,  235, 251, 89,  0,   0,   12,  235, 125, 0,   0,
    0,   138, 255, 255, 166, 0,   0,   0,   12,  235, 255, 255, 255, 251, 89,
    0,   0,   0,   0,   175, 255, 255, 201, 0,   0,   0,   12,  235, 255, 255,
    255, 251, 89,  0,   0,   0,   12,  235, 255, 255, 255, 247, 47,  235, 255,
    255, 255, 255, 255, 255, 255, 138, 0,   235, 125, 0,   0,   0,   59,  245,
    133, 206, 166, 0,   0,   0,   0,   59,  245, 255, 133, 201, 0,   0,   0,
    138, 251, 89,  0,   0,   12,  235, 133, 206, 247, 34,  0,   0,   0,   175,
    229, 216, 225, 21,  0,   0,   0,   138, 247, 124, 255, 255, 255, 255, 255,
    255, 125, 7,   206, 125, 0,   0,   0,   59,  238, 34,  0,   0,   0,   0,
    0,   12,  235, 125, 0,   0,   0,   0,   175, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   59,  245, 166, 0,   0,   0,
    59,  245, 166, 0,   0,   0,   0,   59,  245, 255, 166, 0,   0,   12,  0,
    235, 166, 0,   0,   59,  245, 125, 0,   0,   138, 255, 125, 0,   0,   7,
    202, 102, 0,   235, 166, 0,   0,   59,  245, 225, 21,  0,   12,  235, 166,
    0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   89,
    255, 166, 0,   0,   0,   89,  127, 12,  235, 166, 0,   0,   0,   12,  235,
    125, 0,   12,  235, 125, 0,   0,   0,   0,   0,   138, 225, 21,  12,  235,
    166, 0,   7,   206, 225, 21,  12,  0,   235, 166, 0,   0,   0,   0,   12,
    235, 255, 166, 0,   0,   7,   206, 255, 225, 21,  12,  235, 255, 201, 0,
    0,   12,  235, 125, 0,   59,  245, 166, 0,   0,   138, 251, 89,  0,   12,
    235, 166, 0,   0,   138, 251, 89,  0,   89,  255, 125, 0,   0,   89,  255,
    125, 0,   12,  235, 166, 0,   0,   138, 251, 89,  0,   12,  235, 166, 0,
    0,   7,   202, 89,  0,   0,   0,   138, 225, 21,  0,   0,   12,  0,   235,
    125, 0,   0,   0,   59,  245, 125, 138, 225, 21,  0,   0,   0,   138, 225,
    151, 34,  247, 34,  0,   0,   175, 255, 125, 0,   0,   89,  247, 34,  12,
    235, 166, 0,   0,   89,  247, 34,  59,  245, 125, 0,   0,   59,  245, 125,
    0,   0,   0,   0,   0,   138, 247, 34,  7,   206, 125, 0,   0,   0,   7,
    206, 125, 0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   138, 225,
    187, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   12,
    232, 89,  0,   0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   138, 225,
    151, 225, 21,  0,   12,  0,   235, 166, 0,   0,   12,  235, 166, 0,   12,
    235, 166, 0,   0,   0,   0,   0,   12,  0,   235, 166, 0,   0,   0,   12,
    235, 166, 0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   12,  235,
    166, 0,   0,   0,   12,  235, 125, 0,   12,  235, 125, 0,   0,   0,   0,
    0,   138, 225, 21,  12,  235, 166, 0,   175, 225, 21,  0,   12,  0,   235,
    166, 0,   0,   0,   0,   12,  235, 166, 238, 34,  0,   59,  215, 187, 225,
    21,  12,  235, 166, 245, 125, 0,   12,  235, 125, 12,  235, 125, 0,   0,
    0,   0,   138, 247, 34,  12,  235, 166, 0,   0,   12,  235, 166, 12,  235,
    125, 0,   0,   0,   0,   138, 247, 34,  12,  235, 166, 0,   0,   12,  235,
    166, 0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   138, 225,
    21,  0,   0,   12,  0,   235, 125, 0,   0,   0,   59,  245, 125, 59,  241,
    89,  0,   0,   7,   206, 166, 59,  0,   241, 89,  0,   12,  232, 194, 201,
    0,   0,   138, 225, 21,  0,   89,  251, 89,  12,  235, 166, 0,   0,   138,
    247, 34,  7,   206, 201, 0,   0,   0,   0,   0,   59,  245, 125, 0,   7,
    206, 125, 0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   12,  235,
    125, 0,   0,   59,  241, 89,  12,  235, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   175, 166, 0,   59,  245, 255, 255, 247, 34,  138,
    201, 0,   0,   7,   206, 166, 59,  241, 89,  0,   12,  0,   235, 166, 0,
    0,   89,  251, 89,  0,   89,  247, 34,  0,   0,   0,   0,   0,   12,  0,
    235, 166, 0,   0,   0,   0,   138, 225, 21,  12,  235, 166, 0,   0,   0,
    0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   89,  247, 34,  0,   0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   12,  235, 125, 0,   12,
    235, 125, 0,   0,   0,   0,   0,   138, 225, 21,  12,  235, 166, 175, 247,
    34,  0,   0,   12,  0,   235, 166, 0,   0,   0,   0,   12,  235, 133, 206,
    166, 0,   175, 166, 175, 225, 21,  12,  235, 125, 138, 225, 21,  12,  235,
    125, 89,  247, 34,  0,   0,   0,   0,   59,  245, 125, 12,  235, 166, 0,
    0,   12,  235, 166, 89,  247, 34,  0,   0,   0,   0,   59,  245, 125, 12,
    235, 166, 0,   0,   12,  235, 125, 0,   89,  255, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   138, 225, 21,  0,   0,   12,  0,   235, 125, 0,   0,
    0,   59,  245, 125, 7,   206, 201, 0,   0,   59,  241, 89,  7,   0,   206,
    166, 0,   59,  215, 111, 225, 21,  7,   206, 166, 0,   0,   0,   175, 225,
    187, 225, 21,  0,   0,   12,  235, 166, 89,  247, 34,  0,   0,   0,   0,
    7,   206, 201, 0,   0,   7,   206, 125, 0,   0,   0,   0,   89,  225, 21,
    0,   0,   0,   0,   12,  235, 125, 0,   12,  235, 166, 0,   0,   59,  241,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   202, 89,  12,  235,
    125, 0,   12,  228, 34,  59,  215, 0,   0,   59,  241, 89,  7,   206, 166,
    0,   12,  0,   235, 255, 255, 255, 255, 166, 0,   0,   138, 225, 21,  0,
    0,   0,   0,   0,   12,  0,   235, 166, 0,   0,   0,   0,   89,  247, 34,
    12,  235, 255, 255, 255, 255, 247, 34,  12,  235, 255, 255, 255, 255, 247,
    0,   163, 225, 21,  0,   0,   0,   0,   0,   0,   12,  235, 255, 255, 255,
    255, 255, 255, 125, 0,   12,  235, 125, 0,   0,   0,   0,   0,   138, 225,
    21,  12,  235, 255, 247, 34,  0,   0,   0,   12,  0,   235, 166, 0,   0,
    0,   0,   12,  235, 125, 89,  225, 34,  228, 34,  175, 225, 21,  12,  235,
    125, 12,  235, 125, 12,  235, 125, 138, 225, 21,  0,   0,   0,   0,   12,
    235, 166, 12,  235, 166, 0,   0,   175, 247, 34,  138, 225, 21,  0,   0,
    0,   0,   12,  235, 166, 12,  235, 166, 0,   0,   175, 225, 21,  0,   0,
    175, 255, 255, 225, 21,  0,   0,   0,   0,   0,   138, 225, 21,  0,   0,
    12,  0,   235, 125, 0,   0,   0,   59,  245, 125, 0,   138, 247, 34,  0,
    138, 225, 21,  0,   0,   175, 201, 0,   138, 201, 12,  232, 89,  12,  235,
    125, 0,   0,   0,   12,  235, 251, 89,  0,   0,   0,   0,   89,  255, 255,
    125, 0,   0,   0,   0,   0,   138, 247, 34,  0,   0,   7,   206, 125, 0,
    0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   12,  235, 125, 7,   206,
    201, 0,   0,   0,   0,   138, 251, 89,  0,   0,   0,   0,   0,   0,   0,
    127, 7,   228, 34,  89,  225, 21,  0,   12,  228, 34,  12,  228, 0,   0,
    138, 225, 21,  0,   138, 225, 21,  12,  0,   235, 166, 0,   0,   12,  235,
    201, 0,   138, 225, 21,  0,   0,   0,   0,   0,   12,  0,   235, 166, 0,
    0,   0,   0,   89,  247, 34,  12,  235, 166, 0,   0,   0,   0,   0,   12,
    235, 166, 0,   0,   0,   0,   0,   138, 225, 21,  0,   12,  235, 255, 255,
    127, 12,  235, 166, 0,   0,   0,   12,  235, 125, 0,   12,  235, 125, 0,
    0,   0,   0,   0,   138, 225, 21,  12,  235, 229, 216, 225, 21,  0,   0,
    12,  0,   235, 166, 0,   0,   0,   0,   12,  235, 125, 12,  235, 223, 201,
    0,   175, 225, 21,  12,  235, 125, 0,   138, 225, 34,  235, 125, 138, 225,
    21,  0,   0,   0,   0,   12,  235, 166, 12,  235, 255, 255, 255, 247, 34,
    0,   138, 225, 21,  0,   0,   0,   0,   12,  235, 166, 12,  235, 255, 255,
    255, 166, 0,   0,   0,   0,   0,   0,   89,  255, 255, 247, 34,  0,   0,
    0,   138, 225, 21,  0,   0,   12,  0,   235, 125, 0,   0,   0,   59,  245,
    125, 0,   59,  245, 125, 7,   206, 166, 0,   0,   0,   89,  247, 34,  175,
    125, 7,   206, 125, 89,  247, 34,  0,   0,   0,   12,  235, 251, 89,  0,
    0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   59,  245, 125, 0,
    0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   175, 166, 0,   0,   0,
    0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 12,  228, 34,  89,  225, 21,  0,   12,
    228, 34,  59,  215, 0,   7,   206, 255, 255, 255, 255, 251, 89,  12,  0,
    235, 166, 0,   0,   0,   138, 247, 0,   124, 247, 34,  0,   0,   0,   0,
    0,   12,  0,   235, 166, 0,   0,   0,   0,   138, 225, 21,  12,  235, 166,
    0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   89,  247,
    34,  0,   0,   0,   0,   175, 127, 12,  235, 166, 0,   0,   0,   12,  235,
    125, 0,   12,  235, 125, 0,   0,   0,   0,   0,   138, 225, 21,  12,  235,
    166, 59,  245, 201, 0,   0,   12,  0,   235, 166, 0,   0,   0,   0,   12,
    235, 125, 0,   138, 251, 89,  0,   175, 225, 21,  12,  235, 125, 0,   12,
    235, 138, 235, 125, 89,  247, 34,  0,   0,   0,   0,   59,  245, 125, 12,
    235, 166, 0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   59,
    245, 125, 12,  235, 166, 0,   175, 247, 34,  0,   0,   0,   0,   0,   0,
    0,   59,  245, 166, 0,   0,   0,   138, 225, 21,  0,   0,   12,  0,   235,
    125, 0,   0,   0,   59,  241, 89,  0,   7,   206, 201, 59,  241, 89,  0,
    0,   0,   59,  241, 102, 232, 89,  0,   138, 201, 138, 225, 21,  0,   0,
    0,   175, 201, 175, 225, 21,  0,   0,   0,   0,   175, 225, 21,  0,   0,
    0,   7,   206, 201, 0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,
    0,   89,  225, 21,  0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 12,  232,
    89,  59,  241, 89,  0,   89,  247, 34,  89,  201, 0,   59,  241, 89,  0,
    0,   7,   206, 166, 12,  0,   235, 166, 0,   0,   0,   138, 225, 0,   81,
    245, 166, 0,   0,   0,   0,   0,   12,  0,   235, 166, 0,   0,   0,   12,
    235, 166, 0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   175, 127, 12,  235,
    166, 0,   0,   0,   12,  235, 125, 0,   12,  235, 125, 0,   0,   0,   0,
    0,   138, 225, 21,  12,  235, 166, 0,   89,  255, 166, 0,   12,  0,   235,
    166, 0,   0,   0,   0,   12,  235, 125, 0,   12,  182, 0,   0,   175, 225,
    21,  12,  235, 125, 0,   0,   138, 232, 245, 125, 12,  235, 125, 0,   0,
    0,   0,   138, 247, 34,  12,  235, 166, 0,   0,   0,   0,   0,   12,  235,
    125, 0,   0,   0,   0,   138, 247, 34,  12,  235, 166, 0,   7,   206, 225,
    21,  0,   0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   138, 225,
    21,  0,   0,   12,  0,   235, 166, 0,   0,   0,   89,  251, 89,  0,   0,
    138, 247, 163, 225, 21,  0,   0,   0,   7,   206, 200, 215, 21,  0,   89,
    225, 187, 166, 0,   0,   0,   89,  251, 89,  12,  235, 166, 0,   0,   0,
    0,   175, 225, 21,  0,   0,   0,   138, 247, 34,  0,   0,   0,   0,   7,
    206, 125, 0,   0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   12,  235,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 12,  175, 166, 0,   89,  255, 255, 210, 235, 255, 255,
    125, 0,   138, 225, 21,  0,   0,   0,   138, 247, 47,  0,   235, 166, 0,
    0,   59,  245, 166, 0,   0,   138, 255, 125, 0,   0,   7,   202, 102, 0,
    235, 166, 0,   0,   12,  235, 225, 21,  0,   12,  235, 166, 0,   0,   0,
    0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   138, 255, 125, 0,
    0,   0,   175, 127, 12,  235, 166, 0,   0,   0,   12,  235, 125, 0,   12,
    235, 125, 0,   0,   0,   0,   7,   206, 201, 0,   12,  235, 166, 0,   0,
    138, 255, 125, 12,  0,   235, 166, 0,   0,   0,   12,  0,   235, 125, 0,
    0,   0,   0,   0,   175, 225, 21,  12,  235, 125, 0,   0,   12,  235, 255,
    125, 0,   89,  255, 125, 0,   0,   89,  251, 89,  0,   12,  235, 166, 0,
    0,   0,   0,   0,   0,   89,  255, 125, 0,   0,   89,  255, 125, 0,   12,
    235, 166, 0,   0,   12,  235, 201, 0,   138, 166, 0,   0,   0,   138, 251,
    89,  0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   138, 247, 34,  0,
    7,   206, 225, 21,  0,   0,   12,  235, 255, 166, 0,   0,   0,   0,   0,
    175, 255, 201, 0,   0,   12,  235, 255, 125, 0,   0,   12,  235, 166, 0,
    0,   138, 251, 89,  0,   0,   0,   175, 225, 21,  0,   0,   89,  251, 89,
    0,   0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   175,
    166, 0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   59,  241, 89,  0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   59,
    245, 138, 0,   235, 255, 255, 255, 255, 125, 0,   0,   0,   0,   59,  245,
    255, 255, 255, 201, 12,  0,   235, 255, 255, 255, 255, 166, 0,   0,   0,
    12,  235, 255, 255, 255, 255, 255, 127, 12,  235, 166, 0,   0,   0,   0,
    0,   0,   0,   59,  245, 255, 255, 255, 225, 21,  12,  235, 166, 0,   0,
    0,   12,  235, 125, 89,  255, 255, 255, 210, 127, 235, 255, 255, 225, 21,
    0,   12,  235, 166, 0,   0,   0,   175, 255, 127, 0,   235, 255, 255, 255,
    247, 47,  0,   235, 125, 0,   0,   0,   0,   0,   175, 225, 21,  12,  235,
    125, 0,   0,   0,   138, 255, 125, 0,   0,   0,   175, 255, 255, 201, 0,
    0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,   0,   0,   175, 255,
    255, 201, 0,   0,   0,   12,  235, 166, 0,   0,   0,   89,  255, 225, 34,
    235, 255, 255, 255, 247, 34,  0,   0,   0,   0,   138, 225, 21,  0,   0,
    0,   0,   0,   138, 255, 255, 255, 201, 0,   0,   0,   0,   0,   175, 251,
    89,  0,   0,   0,   0,   0,   89,  255, 166, 0,   0,   7,   206, 247, 34,
    0,   7,   206, 225, 21,  0,   0,   7,   206, 225, 21,  0,   0,   175, 225,
    21,  0,   0,   138, 255, 255, 255, 255, 255, 255, 166, 7,   206, 125, 0,
    0,   0,   0,   0,   0,   138, 201, 0,   0,   0,   12,  235, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   175, 201, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   59,  238, 34,  0,
    0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   7,   206, 255, 255, 255,
    225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  251, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,
    0,   0,   7,   206, 125, 0,   0,   12,  235, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   245, 255, 255, 255, 255, 255, 255, 127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   138, 255, 255, 166, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 255, 255, 201, 0,   0,   0,   0,   0,   0,   89,  89,  255, 255, 255,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   59,  245, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   89,  247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 247, 0,   0,
    0,   0,   0,   0,   0,   12,  12,  235, 125, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   12,
    235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 255, 251, 89,  0,   7,
    206, 125, 0,   89,  255, 251, 89,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   84,  84,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    7,   206, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  12,  235, 125,
    0,   0,   0,   0,   0,   59,  245, 102, 0,   89,  247, 34,  12,  235, 125,
    0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    241, 89,  0,   0,   0,   7,   206, 125, 0,   0,   0,   138, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   4,   4,   4,   4,   4,   100, 252, 252, 84,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,
    0,   0,   0,   0,   12,  0,   235, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   12,  235, 125, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   7,   206, 125, 0,
    0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   4,   4,   4,   4,   4,   4,   4,   20,  236, 252, 164,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 255, 255, 255, 166, 0,   12,  235, 166, 245, 255,
    247, 34,  0,   0,   12,  235, 255, 255, 247, 34,  0,   34,  235, 255, 255,
    255, 225, 21,  0,   12,  235, 255, 255, 225, 29,  0,   206, 255, 255, 255,
    127, 0,   12,  235, 255, 255, 255, 225, 21,  12,  235, 138, 235, 255, 247,
    34,  0,   12,  235, 102, 175, 255, 247, 34,  12,  235, 125, 0,   59,  245,
    201, 0,   12,  235, 125, 12,  0,   235, 166, 245, 255, 225, 29,  206, 255,
    251, 89,  0,   12,  235, 138, 235, 255, 247, 34,  0,   0,   12,  235, 255,
    255, 201, 0,   0,   12,  235, 166, 245, 255, 251, 89,  0,   0,   12,  235,
    255, 255, 255, 225, 21,  12,  235, 138, 235, 247, 127, 34,  138, 255, 255,
    255, 206, 0,   206, 255, 255, 255, 201, 59,  241, 89,  0,   0,   89,  247,
    42,  206, 201, 0,   0,   0,   138, 225, 187, 201, 0,   0,   138, 225, 21,
    0,   59,  241, 187, 226, 247, 34,  0,   7,   206, 206, 206, 201, 0,   0,
    0,   138, 225, 151, 255, 255, 255, 255, 247, 0,   0,   89,  247, 34,  0,
    0,   0,   7,   206, 125, 0,   0,   0,   59,  238, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,   4,   4,   4,
    4,   4,   148, 252, 236, 20,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   12,  206, 21,  0,   59,  245,
    125, 12,  235, 247, 34,  0,   138, 225, 21,  12,  235, 166, 0,   0,   134,
    102, 0,   235, 166, 0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   175,
    201, 12,  0,   235, 125, 0,   0,   12,  235, 166, 0,   0,   138, 225, 21,
    12,  235, 247, 34,  0,   175, 201, 0,   12,  235, 102, 0,   89,  247, 34,
    12,  235, 125, 12,  235, 166, 0,   0,   12,  235, 125, 12,  0,   235, 225,
    21,  12,  235, 251, 89,  0,   175, 201, 0,   12,  235, 247, 34,  0,   175,
    201, 0,   12,  235, 166, 0,   7,   206, 201, 0,   12,  235, 225, 21,  0,
    175, 225, 21,  12,  235, 166, 0,   0,   138, 225, 21,  12,  235, 247, 34,
    0,   0,   89,  247, 34,  0,   12,  206, 34,  0,   235, 125, 0,   0,   59,
    241, 89,  0,   0,   89,  247, 34,  89,  247, 34,  0,   7,   206, 166, 138,
    225, 21,  7,   206, 251, 89,  0,   89,  225, 138, 34,  235, 201, 0,   138,
    225, 21,  89,  247, 34,  0,   7,   206, 166, 0,   0,   0,   7,   206, 166,
    0,   0,   89,  225, 21,  0,   0,   0,   7,   206, 125, 0,   0,   0,   59,
    241, 89,  0,   0,   0,   0,   138, 251, 89,  0,   0,   7,   202, 89,  0,
    0,   4,   4,   4,   4,   4,   4,   52,  252, 252, 108, 4,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   206, 102, 12,  235, 125, 0,   0,   59,  241, 89,
    138, 225, 21,  0,   0,   0,   34,  89,  225, 21,  0,   0,   138, 225, 21,
    89,  225, 21,  0,   0,   89,  247, 47,  0,   235, 125, 0,   0,   89,  225,
    21,  0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   89,  247, 34,  12,
    235, 102, 0,   89,  247, 34,  12,  235, 138, 235, 166, 0,   0,   0,   12,
    235, 125, 12,  0,   235, 125, 0,   7,   206, 166, 0,   0,   138, 225, 21,
    12,  235, 125, 0,   0,   89,  247, 34,  138, 225, 21,  0,   0,   59,  238,
    34,  12,  235, 125, 0,   0,   59,  241, 89,  89,  225, 21,  0,   0,   138,
    225, 21,  12,  235, 125, 0,   0,   0,   138, 225, 21,  0,   0,   0,   12,
    0,   235, 125, 0,   0,   59,  241, 89,  0,   0,   89,  247, 34,  12,  235,
    125, 0,   59,  241, 89,  59,  238, 34,  12,  228, 198, 166, 0,   175, 166,
    59,  0,   89,  251, 132, 241, 89,  0,   12,  235, 125, 0,   59,  238, 34,
    0,   0,   0,   138, 225, 21,  0,   12,  235, 166, 0,   0,   0,   0,   7,
    206, 125, 0,   0,   0,   0,   175, 201, 0,   0,   0,   138, 166, 12,  235,
    166, 0,   12,  232, 89,  0,   0,   12,  84,  4,   4,   4,   4,   204, 252,
    204, 4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   59,  245, 255, 255, 255, 102, 12,  235,
    125, 0,   0,   12,  235, 125, 175, 201, 0,   0,   0,   0,   0,   175, 201,
    0,   0,   0,   138, 225, 21,  175, 255, 255, 255, 255, 255, 247, 47,  0,
    235, 125, 0,   0,   175, 201, 0,   0,   0,   138, 225, 21,  12,  235, 125,
    0,   0,   89,  247, 34,  12,  235, 102, 0,   89,  247, 34,  12,  235, 255,
    225, 21,  0,   0,   0,   12,  235, 125, 12,  0,   235, 125, 0,   7,   206,
    166, 0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   89,  247, 34,  175,
    201, 0,   0,   0,   12,  232, 89,  12,  235, 125, 0,   0,   12,  235, 125,
    175, 201, 0,   0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   0,   59,
    245, 255, 247, 34,  0,   12,  0,   235, 125, 0,   0,   59,  241, 89,  0,
    0,   89,  247, 34,  0,   175, 201, 0,   138, 201, 0,   12,  235, 125, 89,
    201, 89,  225, 29,  206, 125, 12,  0,   0,   175, 255, 166, 0,   0,   0,
    175, 201, 0,   138, 201, 0,   0,   0,   89,  251, 89,  0,   138, 247, 34,
    0,   0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   89,  255,
    125, 7,   202, 89,  0,   89,  251, 89,  89,  201, 0,   0,   0,   172, 252,
    84,  4,   4,   100, 252, 252, 60,  4,   4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 166,
    0,   7,   206, 102, 12,  235, 125, 0,   0,   12,  235, 125, 175, 201, 0,
    0,   0,   0,   0,   175, 201, 0,   0,   0,   138, 225, 21,  175, 201, 0,
    0,   0,   0,   0,   12,  0,   235, 125, 0,   0,   175, 201, 0,   0,   0,
    138, 225, 21,  12,  235, 125, 0,   0,   89,  247, 34,  12,  235, 102, 0,
    89,  247, 34,  12,  235, 138, 235, 201, 0,   0,   0,   12,  235, 125, 12,
    0,   235, 125, 0,   7,   206, 166, 0,   0,   138, 225, 21,  12,  235, 125,
    0,   0,   89,  247, 34,  175, 201, 0,   0,   0,   12,  232, 89,  12,  235,
    125, 0,   0,   12,  235, 125, 175, 201, 0,   0,   0,   138, 225, 21,  12,
    235, 125, 0,   0,   0,   0,   0,   138, 255, 255, 201, 12,  0,   235, 125,
    0,   0,   59,  241, 89,  0,   0,   89,  247, 34,  0,   89,  247, 42,  206,
    125, 0,   0,   175, 166, 175, 125, 12,  232, 102, 232, 89,  0,   0,   0,
    175, 255, 201, 0,   0,   0,   89,  247, 47,  235, 125, 0,   0,   12,  235,
    166, 0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   7,   206, 125, 0,
    0,   0,   0,   138, 201, 0,   0,   12,  232, 89,  0,   0,   59,  245, 225,
    21,  0,   0,   0,   196, 252, 244, 60,  20,  236, 252, 156, 4,   4,   4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   175, 201, 0,   0,   7,   206, 102, 12,  235, 125, 0,   0,
    59,  241, 89,  138, 225, 21,  0,   0,   0,   34,  89,  225, 21,  0,   0,
    138, 225, 21,  138, 247, 34,  0,   0,   0,   0,   12,  0,   235, 125, 0,
    0,   138, 225, 21,  0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   89,
    247, 34,  12,  235, 102, 0,   89,  247, 34,  12,  235, 125, 59,  245, 125,
    0,   0,   12,  235, 125, 12,  0,   235, 125, 0,   7,   206, 166, 0,   0,
    138, 225, 21,  12,  235, 125, 0,   0,   89,  247, 34,  138, 225, 21,  0,
    0,   89,  247, 34,  12,  235, 125, 0,   0,   59,  241, 89,  138, 225, 21,
    0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   0,   0,   0,   0,   0,
    89,  247, 47,  0,   235, 125, 0,   0,   59,  241, 89,  0,   0,   89,  247,
    34,  0,   12,  235, 166, 238, 34,  0,   0,   138, 210, 228, 34,  0,   175,
    166, 215, 21,  0,   0,   89,  251, 159, 251, 89,  0,   0,   12,  235, 191,
    247, 34,  0,   0,   175, 225, 21,  0,   0,   0,   0,   138, 225, 21,  0,
    0,   0,   7,   206, 125, 0,   0,   0,   12,  232, 89,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   20,  220, 252, 236, 180,
    252, 244, 28,  4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,  0,   138, 255,
    102, 12,  235, 125, 0,   7,   206, 201, 0,   12,  235, 166, 0,   0,   134,
    132, 0,   245, 125, 0,   59,  245, 225, 21,  12,  235, 201, 0,   0,   12,
    206, 34,  0,   235, 125, 0,   0,   59,  245, 125, 0,   12,  235, 225, 21,
    12,  235, 125, 0,   0,   89,  247, 34,  12,  235, 102, 0,   89,  247, 34,
    12,  235, 125, 0,   138, 251, 89,  0,   12,  235, 125, 12,  0,   235, 125,
    0,   7,   206, 166, 0,   0,   138, 225, 21,  12,  235, 125, 0,   0,   89,
    247, 34,  12,  235, 166, 0,   7,   206, 201, 0,   12,  235, 125, 0,   7,
    206, 201, 0,   59,  245, 125, 0,   12,  235, 225, 21,  12,  235, 125, 0,
    0,   0,   138, 125, 0,   0,   138, 225, 29,  0,   206, 166, 0,   0,   7,
    206, 166, 0,   59,  245, 247, 34,  0,   0,   175, 255, 201, 0,   0,   0,
    59,  245, 225, 21,  0,   89,  255, 201, 0,   0,   12,  235, 166, 0,   175,
    225, 21,  0,   0,   138, 255, 166, 0,   0,   89,  251, 89,  0,   0,   0,
    0,   0,   89,  247, 34,  0,   0,   0,   7,   206, 125, 0,   0,   0,   59,
    238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   4,   36,  236, 252, 252, 252, 108, 4,   4,   4,   4,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    7,   206, 255, 255, 171, 206, 102, 12,  232, 226, 255, 255, 225, 21,  0,
    0,   12,  235, 255, 255, 247, 34,  0,   89,  255, 255, 247, 163, 225, 21,
    0,   7,   206, 255, 255, 247, 34,  12,  0,   235, 125, 0,   0,   0,   89,
    255, 255, 247, 163, 225, 21,  12,  235, 125, 0,   0,   89,  247, 34,  12,
    235, 102, 0,   89,  247, 34,  12,  235, 125, 0,   0,   175, 251, 34,  0,
    235, 125, 12,  0,   235, 125, 0,   7,   206, 166, 0,   0,   138, 225, 21,
    12,  235, 125, 0,   0,   89,  247, 34,  0,   12,  235, 255, 255, 201, 0,
    0,   12,  235, 255, 255, 255, 225, 21,  0,   0,   89,  255, 255, 247, 163,
    225, 21,  12,  235, 125, 0,   0,   0,   89,  255, 255, 255, 247, 34,  0,
    0,   89,  255, 255, 127, 0,   59,  245, 255, 225, 111, 247, 34,  0,   0,
    59,  245, 125, 0,   0,   0,   12,  235, 166, 0,   0,   59,  245, 125, 7,
    0,   206, 225, 21,  0,   12,  235, 201, 0,   0,   59,  241, 89,  0,   0,
    175, 255, 255, 255, 255, 247, 0,   0,   89,  247, 34,  0,   0,   0,   7,
    206, 125, 0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   4,   60,  252, 252, 204, 4,   4,
    4,   4,   4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 201, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,
    247, 34,  0,   0,   0,   7,   206, 125, 0,   0,   0,   59,  238, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   4,
    4,   76,  252, 60,  4,   4,   4,   4,   4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   199, 34,  0,   12,
    232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   7,   206, 125, 0,
    0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   76,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 251, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  247, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 251,
    89,  0,   7,   206, 125, 0,   89,  255, 247, 34,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   175, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   206, 125, 0,   175, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  245, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 166, 0,   138, 201,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 166, 0,   138, 201, 0,   7,   206, 166, 12,  235, 125, 0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,
    0,   175, 125, 0,   0,   0,   0,   0,   175, 171, 206, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 0,   31,
    206, 130, 255, 166, 175, 247, 34,  0,   0,   89,  255, 125, 175, 247, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  245, 247, 34,
    138, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   59,  241, 132, 238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  241, 132, 238, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   59,  245, 255,
    255, 255, 125, 0,   12,  235, 255, 255, 255, 255, 255, 225, 21,  0,   0,
    0,   0,   0,   0,   0,   0,   175, 255, 255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 125, 0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,   89,  225,
    21,  59,  238, 34,  0,   0,   138, 255, 255, 201, 0,   0,   0,   59,  215,
    21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 255, 255,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 255, 255, 255,
    255, 255, 255, 255, 251, 89,  0,   12,  235, 255, 255, 255, 255, 255, 225,
    21,  0,   89,  255, 255, 255, 255, 255, 255, 125, 0,   12,  235, 255, 255,
    255, 255, 255, 225, 21,  0,   0,   12,  235, 255, 255, 255, 255, 255, 225,
    21,  7,   206, 201, 0,   50,  206, 56,  255, 201, 12,  235, 125, 0,   0,
    138, 225, 29,  206, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   7,   202, 89,  89,  255, 225, 21,  0,   89,  255, 255, 255, 225, 81,
    245, 201, 0,   138, 251, 89,  0,   0,   138, 255, 166, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  235, 255, 255, 255, 255, 255, 225, 21,  0,   0,   0,   138, 255,
    166, 7,   206, 225, 21,  0,   0,   0,   138, 247, 34,  0,   0,   0,   0,
    127, 0,   89,  255, 125, 0,   0,   0,   0,   0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,   0,   175, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  241, 89,  12,
    235, 125, 0,   0,   172, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    12,  235, 166, 0,   0,   7,   202, 89,  0,   0,   0,   0,   0,   0,   89,
    255, 201, 0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   138, 247,
    34,  0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   89,  225, 21,  71,  157, 22,  191,
    225, 21,  175, 201, 0,   7,   206, 125, 59,  238, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   175, 125, 0,   59,  196, 199, 47,  206, 184, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144,
    21,  0,   0,   0,   0,   0,   0,   0,   59,  245, 125, 0,   0,   59,  245,
    125, 0,   0,   0,   0,   0,   127, 12,  235, 166, 0,   0,   0,   0,   0,
    0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   0,
    0,   0,   175, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 255, 255, 255,
    127, 34,  235, 255, 255, 255, 255, 225, 21,  0,   0,   0,   0,   0,   0,
    0,   0,   89,  247, 34,  7,   206, 166, 0,   89,  201, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   59,  115, 12,  235, 166, 0,   0,   0,   12,  235, 125, 0,   0,
    0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,
    0,   0,   0,   59,  245, 125, 0,   0,   12,  146, 0,   0,   0,   0,   0,
    144, 21,  0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   7,
    202, 89,  117, 104, 0,   29,  202, 89,  59,  215, 21,  59,  215, 21,  138,
    201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 0,   59,  192, 89,  223,
    125, 172, 89,  0,   138, 255, 255, 255, 201, 12,  182, 0,   0,   0,   0,
    0,   175, 255, 255, 125, 0,   89,  255, 255, 247, 34,  0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   138, 255, 255, 255, 255, 247, 34,
    138, 247, 34,  7,   206, 201, 0,   0,   0,   0,   0,   0,   127, 89,  251,
    89,  0,   0,   0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144,
    21,  0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   175, 125, 0,   0,   0,   0,   0,   175, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  7,   206, 166, 7,
    202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125,
    0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 89,  247, 34,  0,   0,
    0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   0,   0,   0,   7,   206, 201, 0,   0,   0,   12,
    146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  245, 255, 201, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175,
    125, 0,   59,  192, 12,  228, 34,  172, 89,  89,  247, 34,  0,   12,  206,
    29,  206, 201, 0,   0,   7,   206, 166, 0,   7,   206, 255, 225, 21,  0,
    89,  247, 34,  0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,
    0,   0,   7,   206, 166, 0,   12,  235, 166, 89,  247, 34,  0,   0,   0,
    0,   0,   0,   127, 245, 255, 255, 255, 255, 255, 201, 0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   0,   59,  245, 255,
    255, 255, 127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,
    0,   175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    241, 89,  12,  235, 125, 89,  201, 12,  235, 255, 251, 89,  0,   89,  255,
    255, 225, 21,  0,   175, 255, 255, 225, 21,  0,   0,   0,   89,  251, 89,
    0,   138, 225, 21,  0,   0,   0,   12,  235, 255, 255, 255, 255, 225, 21,
    0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   138,
    247, 34,  0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,
    0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    235, 255, 255, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   175, 125, 0,   59,  192, 0,   0,   0,   172, 89,
    138, 225, 21,  0,   0,   0,   0,   7,   206, 225, 21,  138, 225, 21,  0,
    0,   89,  251, 89,  0,   0,   12,  235, 125, 0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   0,   0,   0,   138, 225, 21,  0,   0,   89,  255,
    255, 125, 0,   0,   0,   0,   0,   0,   0,   127, 138, 225, 21,  0,   0,
    0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,
    0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 125, 0,   0,   12,  235, 255, 255, 255, 255, 225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   138, 255, 255, 201, 12,  228, 34,  175, 166,
    0,   138, 201, 7,   206, 125, 7,   206, 166, 0,   0,   0,   89,  255, 255,
    247, 34,  59,  241, 89,  0,   0,   138, 225, 21,  0,   0,   0,   12,  235,
    125, 0,   0,   0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144,
    21,  0,   0,   0,   59,  245, 125, 0,   0,   0,   0,   12,  146, 0,   0,
    0,   0,   0,   144, 21,  0,   0,   12,  146, 0,   0,   0,   0,   0,   144,
    21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  245, 255, 255, 255, 207, 235, 255, 255, 255,
    255, 255, 255, 207, 235, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
    255, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  245, 255, 247, 34,  0,   0,   0,   0,
    175, 166, 175, 201, 0,   0,   0,   59,  245, 255, 255, 255, 255, 255, 125,
    0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   89,  251,
    89,  0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    127, 245, 255, 255, 255, 255, 255, 125, 0,   0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   175, 125, 0,   0,   0,   0,   0,   175, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   138, 166, 7,   206, 125, 0,   89,  247, 94,  241, 89,  0,   138, 201,
    0,   0,   0,   0,   0,   59,  245, 166, 0,   89,  251, 89,  0,   89,  247,
    34,  0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   0,   7,   206, 201, 0,   0,   0,
    0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 255,
    255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    255, 255, 201, 0,   7,   206, 225, 21,  175, 201, 0,   0,   0,   59,  241,
    89,  0,   0,   0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144,
    21,  0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   175, 225, 21,  0,
    0,   0,   0,   0,   0,   0,   127, 89,  255, 125, 0,   0,   0,   0,   0,
    0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   0,   0,   0,
    0,   89,  247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 125, 0,
    0,   0,   0,   0,   175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   12,  228, 34,  7,   206, 125, 0,   89,  247,
    94,  241, 89,  0,   138, 201, 0,   0,   0,   0,   0,   12,  235, 166, 0,
    0,   89,  255, 125, 12,  235, 166, 0,   0,   0,   12,  235, 125, 0,   0,
    0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,
    138, 247, 34,  0,   0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,
    144, 21,  0,   0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   59,  245, 255, 201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   89,  247, 42,  206, 201, 0,   0,   89,
    225, 21,  0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   12,  146,
    0,   0,   0,   0,   0,   144, 21,  0,   0,   175, 225, 21,  0,   0,   0,
    0,   0,   175, 225, 21,  0,   0,   0,   0,   0,   0,   0,   127, 0,   175,
    251, 89,  0,   0,   0,   0,   0,   12,  146, 0,   0,   0,   0,   0,   144,
    21,  0,   59,  245, 166, 0,   0,   138, 225, 21,  0,   0,   0,   59,  245,
    166, 138, 251, 89,  7,   206, 201, 0,   12,  235, 125, 0,   59,  241, 89,
    0,   0,   0,   175, 125, 0,   0,   0,   0,   0,   175, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 0,
    0,   175, 166, 0,   138, 201, 7,   206, 125, 7,   206, 166, 138, 166, 0,
    0,   0,   138, 251, 89,  0,   0,   0,   59,  115, 0,   89,  255, 201, 0,
    0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   89,  251, 89,  0,   0,   0,   0,   0,   0,   12,
    146, 0,   0,   0,   0,   0,   144, 21,  0,   0,   12,  146, 0,   0,   0,
    0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 125, 0,   0,   138, 225,
    34,  182, 0,   0,   0,   7,   206, 166, 0,   7,   206, 255, 247, 34,  0,
    0,   175, 125, 0,   12,  146, 0,   0,   0,   0,   0,   144, 21,  0,   89,
    251, 89,  0,   0,   0,   0,   0,   0,   175, 225, 21,  0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   138, 255, 255, 255, 255, 125, 0,   12,  235,
    255, 255, 255, 255, 255, 225, 21,  0,   138, 247, 34,  0,   7,   206, 166,
    0,   0,   0,   0,   89,  247, 34,  175, 201, 0,   7,   206, 201, 0,   12,
    235, 125, 0,   59,  241, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  215, 21,  0,   0,   12,  235, 255, 251, 89,  0,   89,  255,
    255, 225, 21,  12,  235, 255, 255, 255, 247, 34,  0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 255, 255, 255, 255, 255, 255, 255, 255, 251, 89,
    0,   12,  235, 255, 255, 255, 255, 255, 225, 21,  0,   138, 255, 255, 255,
    255, 255, 255, 166, 0,   12,  235, 255, 255, 255, 255, 255, 225, 21,  0,
    0,   12,  235, 255, 255, 255, 255, 255, 225, 21,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    89,  255, 255, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   175, 255,
    255, 125, 0,   138, 255, 255, 255, 125, 0,   0,   12,  235, 255, 255, 255,
    255, 255, 225, 21,  0,   175, 255, 255, 255, 255, 247, 0,   0,   0,   175,
    225, 21,  0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175,
    166, 0,   255, 255, 201, 0,   0,   0,   0,   0,   175, 166, 12,  232, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   12,  228, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    12,  232, 89,  59,  215, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 0,   127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   245, 255, 255, 255, 255, 255, 255, 225,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   12,  235, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   89,  247, 34,
    0,   0,   0,   0,   59,  192, 0,   0,   0,   0,   0,   7,   206, 255, 255,
    225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   138, 247, 34,  0,   0,
    89,  251, 89,  0,   7,   206, 125, 0,   0,   7,   206, 255, 255, 255, 166,
    0,   89,  251, 89,  138, 247, 34,  0,   0,   0,   0,   7,   206, 255, 255,
    255, 247, 34,  0,   0,   0,   0,   175, 255, 255, 251, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 255, 255, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255,
    255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    12,  235, 255, 247, 34,  0,   0,   7,   206, 255, 251, 89,  0,   0,   7,
    206, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,
    255, 255, 255, 255, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   59,  245, 255, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  201, 0,
    0,   0,   0,   175, 166, 0,   0,   0,   0,   0,   0,   89,  201, 0,   0,
    0,   0,   175, 166, 0,   0,   0,   0,   0,   59,  245, 255, 201, 0,   0,
    0,   59,  241, 89,  0,   0,   0,   0,   0,   59,  245, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   59,  192, 0,   0,   0,
    0,   0,   175, 201, 0,   0,   144, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   7,   206, 166, 0,   7,   206, 166, 0,   0,   7,   206, 125, 0,   7,
    206, 201, 0,   0,   89,  166, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   89,  255, 125, 0,   0,   0,   59,  245, 166, 0,   0,   0,   0,   0,
    0,   12,  206, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,
    255, 125, 0,   0,   0,   59,  245, 166, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   59,  241, 89,  0,   138, 201, 0,   0,   0,   0,   0,   138,
    166, 0,   0,   0,   0,   0,   168, 34,  7,   206, 166, 0,   0,   172, 89,
    0,   175, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  255, 255, 255, 166, 89,  225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 251, 89,  0,
    0,   12,  235, 125, 0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   7,   206, 255, 201, 0,   0,   0,   89,  225, 21,  0,   0,   0,   0,
    7,   206, 255, 201, 0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   12,
    206, 21,  12,  235, 125, 0,   0,   175, 166, 0,   0,   0,   0,   0,   0,
    59,  245, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 255, 255, 255, 166, 0,   0,   12,  235, 125, 0,   0,   0,   0,   89,
    225, 21,  0,   0,   12,  232, 89,  0,   89,  247, 34,  89,  247, 34,  0,
    0,   7,   206, 125, 0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   7,
    206, 125, 0,   0,   7,   206, 255, 255, 247, 34,  0,   0,   0,   85,  89,
    0,   85,  89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   7,   206, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  201, 0,   0,   12,  228,
    34,  0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   7,
    206, 125, 0,   0,   7,   206, 255, 166, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 125, 0,   0,   89,  247, 34,  175, 255, 255, 255, 166,
    89,  225, 21,  0,   89,  255, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   202, 89,  0,   0,   89,  225, 21,  0,   12,  232, 89,  59,
    115, 0,   59,  115, 0,   0,   0,   0,   0,   89,  201, 0,   0,   7,   206,
    125, 0,   0,   0,   0,   0,   0,   0,   89,  201, 0,   0,   7,   206, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   12,  232, 89,  0,   59,  238, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    89,  225, 21,  0,   0,   138, 247, 94,  192, 12,  182, 0,   0,   12,  235,
    125, 0,   0,   0,   0,   0,   175, 255, 255, 255, 255, 166, 0,   0,   7,
    206, 171, 206, 166, 0,   0,   0,   7,   206, 125, 0,   7,   206, 251, 89,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  0,
    59,  245, 255, 255, 201, 0,   12,  228, 34,  12,  235, 166, 0,   12,  228,
    34,  0,   0,   138, 251, 89,  138, 247, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  0,   138, 255,
    255, 255, 125, 0,   12,  228, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    59,  241, 89,  0,   138, 201, 0,   0,   0,   0,   0,   138, 166, 0,   0,
    0,   0,   0,   0,   0,   175, 201, 0,   0,   0,   0,   0,   0,   175, 201,
    0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   89,  247,
    34,  175, 255, 255, 255, 166, 89,  225, 21,  0,   89,  255, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   138, 225,
    21,  0,   12,  235, 125, 12,  235, 166, 59,  245, 166, 0,   0,   0,   0,
    89,  201, 0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   0,   0,   89,
    201, 0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   0,   12,  235, 255,
    125, 0,   0,   175, 125, 0,   0,   0,   0,   0,   0,   0,   12,  235, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   89,  225, 21,  0,   12,  235, 125, 59,  192,
    0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   59,  215, 21,
    59,  238, 34,  0,   0,   0,   89,  255, 247, 34,  0,   0,   0,   7,   206,
    125, 0,   0,   7,   206, 255, 255, 247, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   59,  192, 0,   12,  235, 166, 0,   7,   176, 21,  0,   175, 125,
    59,  238, 34,  0,   12,  228, 34,  0,   138, 247, 34,  138, 247, 34,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    59,  192, 0,   0,   138, 201, 0,   89,  247, 34,  0,   175, 125, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   89,  255, 255, 225, 21,  0,   7,   206,
    255, 255, 255, 255, 255, 255, 247, 34,  0,   12,  235, 125, 0,   0,   0,
    7,   176, 21,  0,   175, 201, 0,   0,   0,   0,   0,   0,   0,   0,   12,
    235, 125, 0,   0,   89,  247, 34,  89,  255, 255, 255, 166, 89,  225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    202, 89,  0,   0,   89,  225, 21,  0,   12,  232, 89,  0,   12,  235, 166,
    12,  235, 166, 0,   0,   0,   89,  201, 0,   7,   206, 125, 0,   12,  235,
    166, 0,   0,   0,   0,   89,  201, 0,   7,   206, 125, 89,  255, 255, 255,
    125, 0,   0,   0,   0,   7,   206, 125, 89,  225, 21,  0,   138, 225, 21,
    0,   0,   0,   138, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   89,  247, 34,
    0,   59,  241, 89,  59,  192, 0,   0,   0,   12,  235, 255, 255, 255, 225,
    21,  0,   0,   138, 166, 0,   7,   202, 89,  0,   0,   0,   7,   206, 166,
    0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   12,  235, 201,
    0,   0,   0,   0,   0,   0,   0,   0,   89,  166, 0,   89,  247, 34,  0,
    0,   0,   0,   0,   89,  166, 12,  232, 89,  0,   138, 247, 34,  89,  247,
    34,  59,  238, 34,  0,   0,   12,  235, 255, 255, 255, 255, 255, 255, 247,
    34,  89,  255, 255, 255, 166, 89,  166, 0,   0,   138, 201, 0,   138, 225,
    21,  0,   89,  166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   7,
    206, 255, 255, 255, 247, 34,  0,   59,  245, 255, 247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   89,  247, 34,  0,   89,
    255, 255, 166, 89,  225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   89,  255, 255, 255, 166, 0,   12,  235, 125, 0,   138,
    225, 21,  0,   0,   12,  235, 125, 12,  235, 125, 0,   0,   89,  201, 0,
    89,  201, 0,   7,   206, 223, 166, 0,   0,   0,   0,   89,  201, 0,   89,
    201, 0,   89,  125, 0,   138, 225, 21,  12,  182, 0,   7,   206, 133, 206,
    125, 0,   89,  232, 215, 21,  0,   7,   206, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   89,  247, 34,  0,   59,  241, 89,  59,  192, 0,   0,   0,
    0,   12,  235, 125, 0,   0,   0,   0,   0,   59,  215, 21,  59,  238, 34,
    0,   59,  245, 255, 255, 255, 255, 225, 21,  0,   0,   0,   0,   0,   59,
    241, 89,  0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   89,
    166, 0,   89,  247, 34,  0,   0,   0,   0,   0,   89,  166, 0,   138, 255,
    255, 176, 228, 34,  0,   138, 247, 34,  138, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   89,  166, 0,
    0,   138, 255, 255, 225, 21,  0,   0,   89,  166, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 125, 0,
    0,   89,  247, 34,  0,   0,   0,   138, 166, 89,  225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   59,  245, 255, 247, 34,  0,   0,   12,  235, 166, 12,  235, 166,
    0,   0,   0,   0,   0,   12,  232, 89,  0,   175, 166, 138, 166, 0,   0,
    0,   0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   138, 201, 0,   0,
    89,  255, 255, 201, 89,  225, 21,  89,  225, 81,  215, 21,  0,   138, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   89,  247, 34,  0,   12,  235,
    125, 59,  192, 0,   0,   0,   0,   59,  241, 89,  0,   0,   0,   0,   0,
    175, 255, 255, 255, 255, 166, 0,   0,   0,   7,   206, 166, 0,   0,   0,
    0,   7,   206, 125, 0,   12,  235, 201, 0,   7,   206, 166, 0,   0,   0,
    0,   0,   0,   0,   0,   59,  192, 0,   12,  235, 166, 0,   7,   176, 21,
    0,   175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 251, 89,
    138, 247, 34,  0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  0,   0,
    0,   0,   0,   59,  192, 0,   0,   138, 201, 59,  245, 166, 0,   0,   175,
    125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   138, 166, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 125, 0,   0,   89,  247, 34,  0,   0,   0,   138, 166,
    89,  225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    235, 166, 59,  245, 166, 0,   0,   0,   0,   0,   0,   138, 201, 0,   138,
    201, 0,   138, 166, 0,   0,   0,   0,   0,   0,   138, 201, 0,   0,   0,
    0,   89,  247, 34,  0,   0,   0,   0,   0,   7,   206, 125, 59,  238, 34,
    59,  215, 21,  0,   175, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    89,  247, 34,  0,   0,   175, 225, 81,  192, 12,  182, 0,   7,   206, 125,
    0,   0,   0,   0,   0,   89,  225, 21,  0,   0,   12,  232, 89,  0,   0,
    7,   206, 166, 0,   0,   0,   0,   7,   206, 125, 0,   0,   59,  245, 255,
    255, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  0,
    59,  245, 255, 255, 166, 0,   12,  228, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   85,  89,  0,   85,  89,  0,   0,   0,   0,   0,   0,
    0,   59,  238, 34,  0,   0,   0,   0,   0,   7,   202, 89,  0,   138, 201,
    0,   59,  245, 225, 34,  228, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 255, 255, 255, 255, 255,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 201, 0,   0,   175, 247,
    34,  0,   0,   0,   138, 166, 89,  225, 21,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   59,  115, 0,   59,  115, 0,   0,   0,   0,   0,
    0,   12,  232, 89,  0,   175, 255, 255, 255, 255, 201, 0,   0,   0,   0,
    12,  232, 89,  0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   0,
    89,  201, 0,   89,  255, 255, 255, 255, 247, 34,  138, 251, 89,  0,   7,
    176, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   138, 247, 34,  0,   0,   0,   175, 255, 255,
    255, 166, 0,   89,  255, 255, 255, 255, 255, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   7,   206,
    125, 0,   0,   0,   0,   0,   138, 255, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   7,   206, 125, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   59,  238, 34,  0,   0,   0,   0,   0,
    0,   89,  225, 21,  0,   0,   0,   0,   0,   7,   206, 125, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    235, 191, 255, 255, 166, 238, 34,  0,   0,   0,   138, 166, 89,  225, 21,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   175, 201, 0,   0,   0,   0,   0,   138,
    166, 0,   0,   0,   0,   0,   175, 201, 0,   0,   0,   89,  255, 255, 255,
    255, 125, 0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   59,  215, 21,
    0,   0,   138, 255, 255, 255, 225, 21,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   59,  192, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   7,   206, 125, 0,   0,   0,   0,   0,   0,   175, 201,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 0,   0,
    0,   59,  245, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 125, 0,   0,   0,   59,
    245, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   138, 166, 89,  225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,  192, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 125, 0,   7,
    199, 34,  0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   7,   206, 255, 255, 255, 247, 34,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    7,   206, 255, 255, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 125, 0,
    0,   0,   0,   0,   0,   0,   0,   138, 166, 89,  225, 21,  0,   0,   0,
    0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   7,   206, 125, 0,   7,   206, 255, 255, 255, 166, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 201, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 0,   127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,   12,  235, 125,
    0,   0,   0,   0,   19,  172, 255, 190, 11,  0,   0,   0,   0,   138, 255,
    201, 7,   202, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 255, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   59,
    138, 225, 21,  0,   0,   0,   0,   0,   0,   59,  245, 201, 0,   0,   0,
    19,  172, 255, 190, 11,  0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 225, 21,  0,   0,   0,   59,  245, 201, 19,  172, 255, 190, 11,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   12,  235, 251, 89,  89,  201, 0,   0,   0,   0,   0,   175, 201, 0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,
    0,   19,  172, 255, 190, 11,  0,   0,   0,   0,   0,   175, 255, 166, 12,
    228, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  255, 125, 0,   0,   0,   0,   0,   0,
    0,   12,  175, 247, 34,  0,   0,   0,   19,  172, 255, 190, 11,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   7,   206, 125, 0,   0,
    0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   136, 190, 45,  196, 145,
    0,   0,   0,   59,  215, 21,  175, 255, 166, 0,   0,   0,   175, 225, 29,
    206, 166, 0,   0,   7,   202, 89,  7,   202, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   209, 125, 0,   0,   0,   0,   0,   0,
    138, 225, 21,  0,   0,   0,   136, 190, 45,  196, 145, 0,   0,   0,   175,
    225, 29,  206, 166, 0,   0,   12,  235, 125, 0,   0,   12,  138, 225, 21,
    136, 190, 45,  196, 145, 159, 251, 89,  138, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   175, 125, 59,  245, 247, 34,  0,   0,
    0,   0,   0,   12,  232, 89,  0,   0,   0,   0,   0,   0,   0,   175, 166,
    0,   0,   0,   0,   0,   0,   12,  136, 190, 45,  196, 145, 0,   0,   0,
    0,   138, 166, 12,  235, 255, 125, 0,   0,   0,   0,   7,   206, 166, 12,
    235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  232,
    89,  0,   0,   0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   136,
    190, 45,  196, 145, 34,  0,   0,   0,   89,  251, 89,  138, 247, 34,  0,
    0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   202, 89,  7,   202,
    89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  255, 255, 125, 0,
    0,   0,   0,   127, 0,   0,   7,   206, 251, 89,  0,   0,   0,   0,   7,
    206, 251, 89,  0,   0,   0,   0,   7,   206, 251, 89,  0,   0,   0,   0,
    0,   7,   206, 251, 89,  0,   0,   0,   0,   7,   206, 251, 89,  0,   0,
    0,   0,   12,  235, 255, 125, 0,   0,   0,   0,   0,   89,  255, 255, 255,
    255, 255, 255, 255, 255, 125, 0,   0,   0,   59,  245, 255, 255, 255, 201,
    12,  235, 255, 255, 255, 255, 255, 125, 12,  235, 255, 255, 255, 255, 255,
    125, 12,  235, 255, 255, 255, 255, 255, 125, 12,  235, 255, 255, 255, 255,
    255, 125, 89,  255, 255, 255, 201, 89,  255, 255, 255, 201, 89,  255, 255,
    255, 201, 89,  255, 255, 255, 201, 0,   175, 255, 255, 255, 255, 201, 0,
    0,   0,   12,  235, 251, 89,  0,   0,   12,  235, 125, 0,   0,   0,   138,
    255, 255, 166, 0,   0,   0,   0,   0,   0,   138, 255, 255, 166, 0,   0,
    0,   0,   0,   0,   138, 255, 255, 166, 0,   0,   0,   0,   0,   0,   138,
    255, 255, 166, 0,   0,   0,   0,   0,   0,   138, 255, 255, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    255, 255, 201, 89,  251, 89,  12,  235, 125, 0,   0,   0,   59,  245, 125,
    12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,
    59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125, 7,   206, 225,
    21,  0,   0,   0,   138, 247, 0,   235, 166, 0,   0,   0,   0,   0,   0,
    138, 225, 21,  7,   206, 166, 0,   0,   0,   127, 0,   0,   59,  245, 255,
    166, 0,   0,   0,   0,   59,  245, 255, 166, 0,   0,   0,   0,   59,  245,
    255, 166, 0,   0,   0,   0,   0,   59,  245, 255, 166, 0,   0,   0,   0,
    59,  245, 255, 166, 0,   0,   0,   0,   59,  245, 255, 166, 0,   0,   0,
    0,   0,   175, 201, 7,   206, 166, 0,   0,   0,   0,   0,   0,   0,   138,
    255, 125, 0,   0,   7,   202, 102, 235, 166, 0,   0,   0,   0,   0,   12,
    235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,
    12,  235, 166, 0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   12,
    235, 125, 0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   175,
    201, 0,   0,   7,   206, 251, 89,  0,   12,  235, 255, 201, 0,   0,   12,
    235, 125, 0,   59,  245, 166, 0,   0,   138, 251, 89,  0,   0,   59,  245,
    166, 0,   0,   138, 251, 89,  0,   0,   59,  245, 166, 0,   0,   138, 251,
    89,  0,   0,   59,  245, 166, 0,   0,   138, 251, 89,  0,   0,   59,  245,
    166, 0,   0,   138, 251, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   59,  245, 166, 0,   0,   89,  255, 166, 0,   12,  235, 125,
    0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125,
    12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,
    59,  245, 125, 0,   59,  245, 125, 0,   0,   59,  245, 125, 12,  235, 166,
    0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   175, 201, 0,   0,   0,
    127, 0,   0,   138, 225, 151, 225, 21,  0,   0,   0,   138, 225, 151, 225,
    21,  0,   0,   0,   138, 225, 151, 225, 21,  0,   0,   0,   0,   138, 225,
    151, 225, 21,  0,   0,   0,   138, 225, 151, 225, 21,  0,   0,   0,   138,
    225, 151, 225, 21,  0,   0,   0,   59,  241, 89,  7,   206, 166, 0,   0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166,
    0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235,
    166, 0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,
    12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,
    12,  235, 125, 0,   0,   175, 201, 0,   0,   0,   0,   175, 225, 21,  12,
    235, 166, 245, 125, 0,   12,  235, 125, 12,  235, 125, 0,   0,   0,   0,
    138, 247, 34,  12,  235, 125, 0,   0,   0,   0,   138, 247, 34,  12,  235,
    125, 0,   0,   0,   0,   138, 247, 34,  12,  235, 125, 0,   0,   0,   0,
    138, 247, 34,  12,  235, 125, 0,   0,   0,   0,   138, 247, 34,  0,   138,
    225, 21,  0,   0,   0,   175, 201, 0,   12,  235, 125, 0,   0,   7,   202,
    159, 247, 34,  12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125,
    0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125,
    12,  235, 125, 0,   0,   0,   59,  245, 125, 0,   0,   138, 247, 34,  7,
    206, 201, 0,   12,  235, 255, 255, 255, 251, 89,  0,   12,  235, 125, 0,
    12,  235, 125, 0,   0,   0,   127, 0,   7,   206, 166, 59,  241, 89,  0,
    0,   7,   206, 166, 59,  241, 89,  0,   0,   7,   206, 166, 59,  241, 89,
    0,   0,   0,   7,   206, 166, 59,  241, 89,  0,   0,   7,   206, 166, 59,
    241, 89,  0,   0,   7,   206, 166, 59,  241, 89,  0,   0,   0,   138, 225,
    21,  7,   206, 166, 0,   0,   0,   0,   0,   0,   89,  247, 34,  0,   0,
    0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166,
    0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,
    0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   175, 201, 0,   0,
    0,   0,   59,  241, 89,  12,  235, 125, 138, 225, 21,  12,  235, 125, 89,
    247, 34,  0,   0,   0,   0,   59,  245, 125, 89,  247, 34,  0,   0,   0,
    0,   59,  245, 125, 89,  247, 34,  0,   0,   0,   0,   59,  245, 125, 89,
    247, 34,  0,   0,   0,   0,   59,  245, 125, 89,  247, 34,  0,   0,   0,
    0,   59,  245, 125, 0,   0,   175, 225, 21,  0,   175, 225, 21,  0,   89,
    247, 34,  0,   0,   138, 166, 12,  235, 125, 12,  235, 125, 0,   0,   0,
    59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125,
    0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125,
    0,   0,   12,  235, 166, 89,  247, 34,  0,   12,  235, 166, 0,   0,   138,
    251, 89,  12,  235, 133, 206, 255, 125, 0,   0,   0,   0,   127, 0,   59,
    241, 89,  7,   206, 166, 0,   0,   59,  241, 89,  7,   206, 166, 0,   0,
    59,  241, 89,  7,   206, 166, 0,   0,   0,   59,  241, 89,  7,   206, 166,
    0,   0,   59,  241, 89,  7,   206, 166, 0,   0,   59,  241, 89,  7,   206,
    166, 0,   0,   12,  235, 125, 0,   7,   206, 255, 255, 255, 255, 247, 34,
    0,   138, 225, 21,  0,   0,   0,   0,   0,   12,  235, 255, 255, 255, 255,
    247, 34,  12,  235, 255, 255, 255, 255, 247, 34,  12,  235, 255, 255, 255,
    255, 247, 34,  12,  235, 255, 255, 255, 255, 247, 34,  0,   12,  235, 125,
    0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   12,  235, 125,
    0,   206, 255, 255, 255, 247, 34,  0,   12,  235, 125, 12,  235, 125, 12,
    235, 125, 12,  235, 125, 138, 225, 21,  0,   0,   0,   0,   12,  235, 166,
    138, 225, 21,  0,   0,   0,   0,   12,  235, 166, 138, 225, 21,  0,   0,
    0,   0,   12,  235, 166, 138, 225, 21,  0,   0,   0,   0,   12,  235, 166,
    138, 225, 21,  0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   175, 225,
    187, 225, 21,  0,   0,   138, 225, 21,  0,   59,  215, 21,  7,   206, 166,
    12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,
    59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125,
    0,   0,   0,   59,  245, 125, 0,   0,   0,   89,  255, 255, 125, 0,   0,
    12,  235, 166, 0,   0,   12,  235, 166, 12,  235, 125, 0,   7,   206, 201,
    0,   0,   0,   127, 0,   138, 225, 21,  0,   138, 225, 21,  0,   138, 225,
    21,  0,   138, 225, 21,  0,   138, 225, 21,  0,   138, 225, 21,  0,   0,
    138, 225, 21,  0,   138, 225, 21,  0,   138, 225, 21,  0,   138, 225, 21,
    0,   138, 225, 21,  0,   138, 225, 21,  0,   89,  255, 255, 255, 255, 255,
    166, 0,   0,   0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,   0,
    12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,
    0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,
    0,   0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   12,  235,
    125, 0,   0,   12,  235, 125, 0,   0,   175, 201, 0,   0,   0,   0,   12,
    235, 125, 12,  235, 125, 0,   138, 225, 34,  235, 125, 138, 225, 21,  0,
    0,   0,   0,   12,  235, 166, 138, 225, 21,  0,   0,   0,   0,   12,  235,
    166, 138, 225, 21,  0,   0,   0,   0,   12,  235, 166, 138, 225, 21,  0,
    0,   0,   0,   12,  235, 166, 138, 225, 21,  0,   0,   0,   0,   12,  235,
    166, 0,   0,   0,   0,   175, 225, 21,  0,   0,   0,   138, 225, 21,  7,
    202, 89,  0,   7,   206, 166, 12,  235, 125, 0,   0,   0,   59,  245, 125,
    12,  235, 125, 0,   0,   0,   59,  245, 125, 12,  235, 125, 0,   0,   0,
    59,  245, 125, 12,  235, 125, 0,   0,   0,   59,  245, 125, 0,   0,   0,
    7,   206, 225, 21,  0,   0,   12,  235, 166, 0,   0,   12,  235, 166, 12,
    235, 125, 0,   0,   59,  241, 89,  0,   0,   127, 7,   206, 255, 255, 255,
    255, 251, 89,  7,   206, 255, 255, 255, 255, 251, 89,  7,   206, 255, 255,
    255, 255, 251, 89,  0,   7,   206, 255, 255, 255, 255, 251, 89,  7,   206,
    255, 255, 255, 255, 251, 89,  7,   206, 255, 255, 255, 255, 251, 89,  7,
    206, 166, 0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   0,   89,  247,
    34,  0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,
    235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,
    12,  235, 166, 0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   12,
    235, 125, 0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   175,
    201, 0,   0,   0,   0,   59,  241, 89,  12,  235, 125, 0,   12,  235, 138,
    235, 125, 89,  247, 34,  0,   0,   0,   0,   59,  245, 125, 89,  247, 34,
    0,   0,   0,   0,   59,  245, 125, 89,  247, 34,  0,   0,   0,   0,   59,
    245, 125, 89,  247, 34,  0,   0,   0,   0,   59,  245, 125, 89,  247, 34,
    0,   0,   0,   0,   59,  245, 125, 0,   0,   0,   175, 225, 187, 225, 21,
    0,   0,   138, 247, 34,  175, 125, 0,   0,   12,  235, 125, 12,  235, 125,
    0,   0,   0,   59,  241, 89,  12,  235, 125, 0,   0,   0,   59,  241, 89,
    12,  235, 125, 0,   0,   0,   59,  241, 89,  12,  235, 125, 0,   0,   0,
    59,  241, 89,  0,   0,   0,   0,   175, 225, 21,  0,   0,   12,  235, 166,
    0,   0,   175, 247, 34,  12,  235, 125, 0,   0,   12,  235, 125, 0,   0,
    127, 59,  241, 89,  0,   0,   7,   206, 166, 59,  241, 89,  0,   0,   7,
    206, 166, 59,  241, 89,  0,   0,   7,   206, 166, 0,   59,  241, 89,  0,
    0,   7,   206, 166, 59,  241, 89,  0,   0,   7,   206, 166, 59,  241, 89,
    0,   0,   7,   206, 166, 59,  241, 89,  0,   0,   7,   206, 166, 0,   0,
    0,   0,   0,   0,   59,  245, 166, 0,   0,   0,   0,   0,   12,  235, 166,
    0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235,
    166, 0,   0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   0,
    12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,
    12,  235, 125, 0,   0,   175, 201, 0,   0,   0,   0,   175, 225, 21,  12,
    235, 125, 0,   0,   138, 232, 245, 125, 12,  235, 125, 0,   0,   0,   0,
    138, 247, 34,  12,  235, 125, 0,   0,   0,   0,   138, 247, 34,  12,  235,
    125, 0,   0,   0,   0,   138, 247, 34,  12,  235, 125, 0,   0,   0,   0,
    138, 247, 34,  12,  235, 125, 0,   0,   0,   0,   138, 247, 34,  0,   0,
    175, 225, 21,  0,   175, 225, 21,  0,   59,  245, 191, 201, 0,   0,   0,
    89,  225, 21,  12,  235, 166, 0,   0,   0,   89,  251, 89,  12,  235, 166,
    0,   0,   0,   89,  251, 89,  12,  235, 166, 0,   0,   0,   89,  251, 89,
    12,  235, 166, 0,   0,   0,   89,  251, 89,  0,   0,   0,   0,   175, 225,
    21,  0,   0,   12,  235, 255, 255, 255, 247, 34,  0,   12,  235, 125, 0,
    0,   59,  241, 89,  0,   0,   127, 138, 225, 21,  0,   0,   0,   138, 247,
    163, 225, 21,  0,   0,   0,   138, 247, 163, 225, 21,  0,   0,   0,   138,
    247, 34,  138, 225, 21,  0,   0,   0,   138, 247, 163, 225, 21,  0,   0,
    0,   138, 247, 163, 225, 21,  0,   0,   0,   138, 247, 198, 225, 21,  0,
    0,   7,   206, 166, 0,   0,   0,   0,   0,   0,   0,   138, 255, 125, 0,
    0,   7,   202, 102, 235, 166, 0,   0,   0,   0,   0,   12,  235, 166, 0,
    0,   0,   0,   0,   12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 166,
    0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   12,  235, 125, 0,
    0,   12,  235, 125, 0,   0,   12,  235, 125, 0,   0,   175, 201, 0,   0,
    7,   206, 251, 89,  0,   12,  235, 125, 0,   0,   12,  235, 255, 125, 0,
    89,  255, 125, 0,   0,   89,  251, 89,  0,   0,   89,  255, 125, 0,   0,
    89,  251, 89,  0,   0,   89,  255, 125, 0,   0,   89,  251, 89,  0,   0,
    89,  255, 125, 0,   0,   89,  251, 89,  0,   0,   89,  255, 125, 0,   0,
    89,  251, 89,  0,   0,   138, 225, 21,  0,   0,   0,   175, 201, 0,   0,
    138, 251, 89,  0,   0,   89,  251, 89,  0,   0,   138, 247, 34,  0,   7,
    206, 225, 21,  0,   138, 247, 34,  0,   7,   206, 225, 21,  0,   138, 247,
    34,  0,   7,   206, 225, 21,  0,   138, 247, 34,  0,   7,   206, 225, 21,
    0,   0,   0,   0,   175, 225, 21,  0,   0,   12,  235, 166, 0,   0,   0,
    0,   0,   12,  235, 125, 0,   0,   175, 225, 21,  0,   0,   127, 206, 166,
    0,   0,   0,   0,   59,  245, 255, 166, 0,   0,   0,   0,   59,  245, 255,
    166, 0,   0,   0,   0,   59,  245, 133, 206, 166, 0,   0,   0,   0,   59,
    245, 255, 166, 0,   0,   0,   0,   59,  245, 255, 166, 0,   0,   0,   0,
    59,  245, 255, 125, 0,   0,   0,   7,   206, 255, 255, 255, 255, 255, 125,
    0,   0,   0,   59,  245, 255, 255, 255, 201, 12,  235, 255, 255, 255, 255,
    255, 125, 12,  235, 255, 255, 255, 255, 255, 125, 12,  235, 255, 255, 255,
    255, 255, 125, 12,  235, 255, 255, 255, 255, 255, 125, 89,  255, 255, 255,
    201, 89,  255, 255, 255, 201, 89,  255, 255, 255, 201, 89,  255, 255, 255,
    201, 0,   175, 255, 255, 255, 255, 225, 21,  0,   0,   12,  235, 125, 0,
    0,   0,   138, 255, 125, 0,   0,   0,   175, 255, 255, 201, 0,   0,   0,
    0,   0,   0,   175, 255, 255, 201, 0,   0,   0,   0,   0,   0,   175, 255,
    255, 201, 0,   0,   0,   0,   0,   0,   175, 255, 255, 201, 0,   0,   0,
    0,   0,   0,   175, 255, 255, 201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   202, 97,  206, 255, 255, 201, 0,   0,   0,
    0,   0,   138, 255, 255, 255, 201, 0,   0,   0,   0,   138, 255, 255, 255,
    201, 0,   0,   0,   0,   138, 255, 255, 255, 201, 0,   0,   0,   0,   138,
    255, 255, 255, 201, 0,   0,   0,   0,   0,   0,   175, 225, 21,  0,   0,
    12,  235, 166, 0,   0,   0,   0,   0,   12,  235, 133, 206, 255, 225, 21,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   138, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   175, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   7,   206, 255, 225, 21,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 0,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   12,
    235, 225, 21,  0,   0,   89,  255, 225, 21,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 255, 247,
    34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 247, 34,  0,   0,   0,   0,
    0,   0,   0,   138, 251, 89,  0,   0,   59,  245, 247, 34,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   175, 247, 34,  0,   0,   175, 0,   255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   206, 225, 21,  0,   0,   0,   0,   0,   0,
    0,   138, 255, 125, 0,   0,   0,   12,  235, 251, 89,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   138, 251, 89,  0,   0,   0,   0,   0,   0,   7,   206, 225,
    21,  0,   0,   0,   7,   206, 251, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   59,  245, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   59,  241, 89,
    0,   0,   0,   0,   0,   89,  247, 34,  0,   0,   7,   206, 138, 235, 125,
    0,   0,   89,  255, 225, 21,  175, 125, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   138, 201, 0,   138, 201, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    235, 125, 0,   0,   0,   0,   0,   0,   12,  235, 125, 0,   0,   0,   175,
    171, 206, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 166,
    0,   59,  245, 255, 166, 238, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   7,   206, 255, 125, 59,  215, 21,  0,   0,   59,  241, 89,
    0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   138, 201,
    175, 201, 0,   0,   0,   12,  235, 251, 89,  89,  201, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,
    0,   0,   0,   89,  247, 34,  0,   0,   0,   0,   89,  225, 151, 201, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 201, 0,
    12,  235, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   138, 201, 0,   0,   0,   0,   7,   206, 125, 0,   0,
    0,   138, 201, 0,   89,  225, 21,  12,  228, 34,  138, 255, 201, 0,   0,
    0,   138, 247, 34,  175, 225, 21,  0,   138, 201, 0,   138, 201, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   89,  225, 21,  0,   0,   0,   0,   0,   89,
    225, 21,  0,   0,   89,  247, 34,  59,  241, 89,  0,   59,  241, 89,  89,
    247, 34,  0,   0,   89,  225, 21,  175, 127, 215, 21,  206, 247, 42,  206,
    0,   138, 255, 247, 42,  206, 125, 0,   0,   138, 166, 12,  235, 251, 89,
    0,   0,   0,   0,   138, 201, 0,   0,   0,   0,   0,   0,   89,  225, 21,
    0,   0,   0,   59,  241, 89,  12,  235, 125, 0,   0,   175, 125, 59,  245,
    247, 34,  0,   0,   12,  235, 125, 89,  251, 89,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   59,  238, 34,  0,   0,   0,   0,   0,   175, 166, 0,   0,   0,   0,
    12,  232, 89,  7,   206, 125, 0,   0,   12,  235, 166, 59,  245, 125, 0,
    0,   0,   59,  238, 34,  0,   12,  235, 125, 0,   0,   0,   0,   0,   0,
    89,  247, 34,  138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 255, 247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 251, 89,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   7,   199, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 125,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 12,  235,
    255, 255, 255, 166, 0,   12,  235, 255, 255, 255, 166, 0,   12,  235, 255,
    255, 255, 166, 0,   12,  235, 255, 255, 255, 166, 0,   0,   12,  235, 255,
    255, 255, 166, 0,   12,  235, 255, 255, 255, 166, 0,   12,  235, 255, 255,
    255, 166, 0,   175, 255, 255, 125, 0,   0,   12,  235, 255, 255, 125, 0,
    0,   12,  235, 255, 255, 225, 21,  0,   0,   12,  235, 255, 255, 225, 21,
    0,   12,  235, 255, 255, 225, 21,  0,   12,  235, 255, 255, 225, 21,  0,
    12,  235, 125, 12,  235, 125, 12,  235, 125, 12,  235, 125, 0,   12,  235,
    125, 89,  251, 89,  0,   12,  235, 138, 235, 255, 247, 34,  0,   0,   12,
    235, 255, 255, 201, 0,   0,   0,   12,  235, 255, 255, 201, 0,   0,   0,
    12,  235, 255, 255, 201, 0,   0,   0,   12,  235, 255, 255, 201, 0,   0,
    0,   12,  235, 255, 255, 201, 0,   0,   0,   0,   0,   0,   175, 247, 34,
    0,   0,   0,   12,  235, 255, 255, 255, 166, 0,   59,  241, 89,  0,   0,
    89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,
    0,   89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 42,  206, 201, 0,
    0,   0,   138, 232, 245, 166, 245, 255, 251, 89,  7,   206, 201, 0,   0,
    0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 12,  206, 21,  0,   59,  245, 125, 12,  206, 21,  0,
    59,  245, 125, 12,  206, 21,  0,   59,  245, 125, 12,  206, 21,  0,   59,
    245, 125, 0,   12,  206, 21,  0,   59,  245, 125, 12,  206, 21,  0,   59,
    245, 125, 12,  206, 21,  0,   12,  235, 255, 125, 0,   7,   206, 166, 12,
    235, 166, 0,   0,   172, 102, 0,   235, 125, 0,   0,   175, 201, 0,   12,
    235, 125, 0,   0,   175, 201, 12,  235, 125, 0,   0,   175, 201, 12,  235,
    125, 0,   0,   175, 201, 0,   12,  235, 125, 12,  235, 125, 12,  235, 125,
    12,  235, 125, 0,   0,   0,   0,   0,   175, 201, 0,   12,  235, 247, 34,
    0,   175, 201, 0,   12,  235, 166, 0,   7,   206, 201, 0,   12,  235, 166,
    0,   7,   206, 201, 0,   12,  235, 166, 0,   7,   206, 201, 0,   12,  235,
    166, 0,   7,   206, 201, 0,   12,  235, 166, 0,   7,   206, 201, 0,   0,
    0,   0,   0,   175, 247, 34,  0,   0,   12,  235, 166, 0,   12,  235, 201,
    0,   59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,   0,   89,
    247, 34,  59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,   0,
    89,  247, 34,  89,  247, 34,  0,   7,   206, 176, 235, 225, 21,  0,   175,
    225, 21,  89,  247, 34,  0,   7,   206, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   7,
    206, 166, 0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   7,   206,
    166, 0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   7,   206,
    166, 0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   175, 201,
    0,   0,   0,   89,  225, 138, 225, 21,  0,   0,   0,   0,   89,  225, 21,
    0,   0,   89,  247, 34,  89,  225, 21,  0,   0,   89,  247, 124, 225, 21,
    0,   0,   89,  247, 124, 225, 21,  0,   0,   89,  247, 34,  12,  235, 125,
    12,  235, 125, 12,  235, 125, 12,  235, 125, 0,   89,  255, 255, 255, 255,
    247, 34,  12,  235, 125, 0,   0,   89,  247, 34,  138, 225, 21,  0,   0,
    59,  238, 34,  138, 225, 21,  0,   0,   59,  238, 34,  138, 225, 21,  0,
    0,   59,  238, 34,  138, 225, 21,  0,   0,   59,  238, 34,  138, 225, 21,
    0,   0,   59,  238, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   138,
    225, 21,  0,   172, 132, 238, 34,  59,  241, 89,  0,   0,   89,  247, 34,
    59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,   0,   89,  247,
    34,  59,  241, 89,  0,   0,   89,  247, 34,  12,  235, 125, 0,   59,  238,
    47,  235, 125, 0,   0,   59,  241, 89,  12,  235, 125, 0,   59,  238, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   59,  245, 255, 255, 255, 166, 0,   59,  245, 255, 255, 255, 166,
    0,   59,  245, 255, 255, 255, 166, 0,   59,  245, 255, 255, 255, 166, 0,
    0,   59,  245, 255, 255, 255, 166, 0,   59,  245, 255, 255, 255, 166, 0,
    89,  255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 175, 201, 0,   0,
    0,   0,   0,   175, 255, 255, 255, 255, 255, 247, 34,  175, 255, 255, 255,
    255, 255, 247, 198, 255, 255, 255, 255, 255, 247, 198, 255, 255, 255, 255,
    255, 247, 34,  12,  235, 125, 12,  235, 125, 12,  235, 125, 12,  235, 125,
    89,  251, 89,  0,   0,   59,  241, 89,  12,  235, 125, 0,   0,   89,  247,
    34,  175, 201, 0,   0,   0,   12,  232, 89,  175, 201, 0,   0,   0,   12,
    232, 89,  175, 201, 0,   0,   0,   12,  232, 89,  175, 201, 0,   0,   0,
    12,  232, 89,  175, 201, 0,   0,   0,   12,  232, 89,  7,   206, 255, 255,
    255, 255, 255, 255, 251, 226, 201, 0,   89,  166, 12,  232, 89,  59,  241,
    89,  0,   0,   89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 34,  59,
    241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 34,
    0,   175, 201, 0,   138, 201, 12,  235, 125, 0,   0,   12,  235, 125, 0,
    175, 201, 0,   138, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 89,  255, 166, 0,   7,   206, 166, 89,
    255, 166, 0,   7,   206, 166, 89,  255, 166, 0,   7,   206, 166, 89,  255,
    166, 0,   7,   206, 166, 0,   89,  255, 166, 0,   7,   206, 166, 89,  255,
    166, 0,   7,   206, 166, 138, 255, 125, 0,   0,   175, 201, 0,   0,   0,
    0,   0,   175, 201, 0,   0,   0,   0,   0,   175, 201, 0,   0,   0,   0,
    0,   0,   175, 201, 0,   0,   0,   0,   0,   175, 201, 0,   0,   0,   0,
    0,   175, 201, 0,   0,   0,   0,   0,   0,   12,  235, 125, 12,  235, 125,
    12,  235, 125, 12,  235, 125, 175, 201, 0,   0,   0,   59,  241, 89,  12,
    235, 125, 0,   0,   89,  247, 34,  175, 201, 0,   0,   0,   12,  232, 89,
    175, 201, 0,   0,   0,   12,  232, 89,  175, 201, 0,   0,   0,   12,  232,
    89,  175, 201, 0,   0,   0,   12,  232, 89,  175, 201, 0,   0,   0,   12,
    232, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   175, 201, 7,   176,
    21,  12,  232, 89,  59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,
    0,   0,   89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 34,  59,  241,
    89,  0,   0,   89,  247, 34,  0,   89,  247, 47,  235, 125, 12,  235, 125,
    0,   0,   12,  235, 125, 0,   89,  247, 47,  235, 125, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 175, 201,
    0,   0,   7,   206, 166, 175, 201, 0,   0,   7,   206, 166, 175, 201, 0,
    0,   7,   206, 166, 175, 201, 0,   0,   7,   206, 166, 0,   175, 201, 0,
    0,   7,   206, 166, 175, 201, 0,   0,   7,   206, 166, 175, 201, 0,   0,
    0,   138, 225, 21,  0,   0,   0,   0,   138, 225, 21,  0,   0,   0,   0,
    138, 247, 34,  0,   0,   0,   0,   0,   138, 247, 34,  0,   0,   0,   0,
    138, 247, 34,  0,   0,   0,   0,   138, 247, 34,  0,   0,   0,   0,   0,
    12,  235, 125, 12,  235, 125, 12,  235, 125, 12,  235, 125, 175, 201, 0,
    0,   0,   89,  247, 34,  12,  235, 125, 0,   0,   89,  247, 34,  138, 225,
    21,  0,   0,   89,  247, 34,  138, 225, 21,  0,   0,   89,  247, 34,  138,
    225, 21,  0,   0,   89,  247, 34,  138, 225, 21,  0,   0,   89,  247, 34,
    138, 225, 21,  0,   0,   89,  247, 34,  0,   0,   0,   0,   175, 247, 34,
    0,   0,   138, 225, 151, 125, 0,   89,  247, 34,  59,  241, 89,  0,   0,
    89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 34,  59,  241, 89,  0,
    0,   89,  247, 34,  59,  241, 89,  0,   0,   89,  247, 34,  0,   12,  235,
    191, 247, 34,  12,  235, 125, 0,   0,   59,  241, 89,  0,   12,  235, 191,
    247, 34,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 138, 225, 21,  0,   138, 255, 166, 138, 225, 21,  0,
    138, 255, 166, 138, 225, 21,  0,   138, 255, 166, 138, 225, 21,  0,   138,
    255, 166, 0,   138, 225, 21,  0,   138, 255, 166, 138, 225, 21,  0,   138,
    255, 166, 89,  247, 34,  0,   89,  255, 255, 166, 0,   0,   12,  206, 12,
    235, 166, 0,   0,   127, 102, 0,   235, 201, 0,   0,   12,  206, 21,  12,
    235, 201, 0,   0,   12,  206, 34,  235, 201, 0,   0,   12,  206, 34,  235,
    201, 0,   0,   12,  206, 21,  12,  235, 125, 12,  235, 125, 12,  235, 125,
    12,  235, 125, 89,  255, 125, 0,   7,   206, 166, 0,   12,  235, 125, 0,
    0,   89,  247, 34,  12,  235, 166, 0,   7,   206, 201, 0,   12,  235, 166,
    0,   7,   206, 201, 0,   12,  235, 166, 0,   7,   206, 201, 0,   12,  235,
    166, 0,   7,   206, 201, 0,   12,  235, 166, 0,   7,   206, 201, 0,   0,
    0,   0,   0,   175, 247, 34,  0,   0,   12,  235, 201, 0,   7,   206, 201,
    0,   7,   206, 166, 0,   59,  245, 247, 34,  7,   206, 166, 0,   59,  245,
    247, 34,  7,   206, 166, 0,   59,  245, 247, 34,  7,   206, 166, 0,   59,
    245, 247, 34,  0,   0,   138, 255, 166, 0,   12,  235, 125, 0,   7,   206,
    201, 0,   0,   0,   138, 255, 166, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 7,   206, 255, 255, 171,
    206, 166, 7,   206, 255, 255, 171, 206, 166, 7,   206, 255, 255, 171, 206,
    166, 7,   206, 255, 255, 171, 206, 166, 0,   7,   206, 255, 255, 171, 206,
    166, 7,   206, 255, 255, 171, 206, 166, 0,   89,  255, 255, 201, 0,   0,
    175, 255, 255, 247, 34,  0,   12,  235, 255, 255, 166, 0,   0,   7,   206,
    255, 255, 247, 34,  0,   0,   7,   206, 255, 255, 247, 34,  0,   7,   206,
    255, 255, 247, 34,  0,   7,   206, 255, 255, 247, 34,  0,   12,  235, 125,
    12,  235, 125, 12,  235, 125, 12,  235, 125, 0,   89,  255, 255, 255, 201,
    0,   0,   12,  235, 125, 0,   0,   89,  247, 34,  0,   12,  235, 255, 255,
    201, 0,   0,   0,   12,  235, 255, 255, 201, 0,   0,   0,   12,  235, 255,
    255, 201, 0,   0,   0,   12,  235, 255, 255, 201, 0,   0,   0,   12,  235,
    255, 255, 201, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,
    206, 255, 255, 255, 201, 0,   0,   0,   59,  245, 255, 225, 111, 247, 34,
    0,   59,  245, 255, 225, 111, 247, 34,  0,   59,  245, 255, 225, 111, 247,
    34,  0,   59,  245, 255, 225, 111, 247, 34,  0,   0,   59,  241, 89,  0,
    12,  235, 255, 255, 255, 225, 21,  0,   0,   0,   59,  241, 89,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    175, 125, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   138, 125, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   138, 225, 21,  0,   12,  235, 125, 0,   0,   0,   0,   0,   0,
    0,   138, 225, 21,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   7,   202, 89,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   12,  235, 125,
    0,   0,   0,   0,   0,   0,   7,   206, 166, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   12,  235, 255, 166, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  247,
    34,  0,   0,   12,  235, 125, 0,   0,   0,   0,   0,   0,   89,  247, 34,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 0,   127, 127, 0,   127, 127, 0,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0};

const int FONTFIXED1_BM_W = 257;
const int FONTFIXED1_BM_H = 112;

static const unsigned char s_FontFixed1[] = {
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 56,  255, 4,   0,
    0,   0,   0,   0,   212, 44,  76,  180, 0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   109, 231, 218, 72,  0,   0,   0,   0,   0,   96,  227, 243,
    170, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,
    158, 104, 0,   0,   0,   0,   153, 114, 0,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   172, 128, 0,
    0,   12,  164, 241, 234, 133, 1,   0,   0,   22,  179, 237, 255, 4,   0,
    0,   0,   141, 220, 246, 236, 164, 22,  0,   0,   94,  216, 242, 243, 194,
    56,  0,   0,   0,   0,   0,   186, 255, 4,   0,   0,   52,  255, 244, 244,
    244, 91,  0,   0,   1,   120, 223, 244, 225, 62,  0,   0,   244, 244, 244,
    244, 249, 242, 0,   0,   62,  200, 245, 242, 181, 35,  0,   0,   46,  196,
    244, 232, 139, 2,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   104, 216, 246, 215, 62,  0,   0,   127, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 56,  255,
    4,   0,   0,   0,   0,   13,  239, 2,   131, 124, 0,   0,   110, 232, 255,
    238, 202, 62,  0,   29,  254, 51,  99,  231, 0,   0,   0,   0,   0,   241,
    53,  0,   34,  0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    0,   45,  225, 4,   0,   0,   0,   0,   30,  237, 15,  0,   0,   0,   0,
    99,  95,  52,  255, 11,  127, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   38,  242,
    19,  0,   0,   155, 188, 12,  29,  221, 103, 0,   0,   21,  101, 90,  255,
    4,   0,   0,   0,   127, 46,  1,   15,  165, 192, 0,   0,   34,  24,  0,
    5,   127, 233, 0,   0,   0,   0,   98,  197, 255, 4,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   121, 219, 45,  0,   22,  27,  0,   0,   0,
    0,   0,   0,   170, 151, 0,   14,  242, 119, 4,   12,  160, 207, 0,   3,
    224, 136, 5,   18,  188, 114, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   114, 38,  2,   133, 225, 0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255,
    56,  255, 4,   0,   0,   0,   0,   67,  189, 0,   187, 69,  0,   26,  254,
    100, 255, 8,   53,  44,  0,   30,  254, 49,  100, 235, 0,   1,   0,   0,
    0,   206, 47,  0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   0,   155, 133, 0,   0,   0,   0,   0,   0,   186, 110, 0,   0,
    0,   0,   3,   103, 195, 255, 177, 75,  0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    156, 144, 0,   0,   5,   244, 63,  0,   0,   112, 200, 0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   60,  251, 0,   0,   0,
    0,   0,   10,  127, 211, 0,   0,   0,   25,  215, 67,  255, 4,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   1,   235, 77,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   24,  250, 49,  0,   44,  255, 15,  0,   0,   64,  251,
    0,   41,  255, 17,  0,   0,   68,  205, 0,   0,   0,   43,  216, 3,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    47,  147, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   197, 97,  11,
    0,   0,   0,   0,   0,   0,   0,   0,   69,  248, 0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   63,  240, 247, 248, 240, 254, 241, 0,
    42,  255, 69,  255, 4,   0,   0,   0,   0,   112, 232, 221, 80,  97,  184,
    0,   0,   14,  189, 196, 7,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   1,   233, 64,  0,   0,   0,   0,   0,   0,   117, 186,
    0,   0,   0,   0,   3,   102, 194, 255, 177, 74,  0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   26,  244, 30,  0,   0,   37,  255, 37,  175, 0,   66,  244, 0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   126, 195, 0,
    0,   0,   48,  241, 255, 190, 16,  0,   0,   0,   176, 90,  52,  255, 4,
    0,   0,   52,  255, 228, 236, 162, 16,  0,   33,  255, 106, 220, 237, 172,
    21,  0,   0,   0,   0,   0,   125, 203, 0,   0,   4,   205, 120, 6,   14,
    160, 159, 0,   40,  255, 21,  0,   0,   69,  245, 0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   13,
    102, 203, 225, 132, 0,   0,   240, 240, 240, 240, 240, 240, 0,   0,   84,
    180, 237, 152, 52,  0,   0,   0,   0,   0,   11,  204, 150, 0,   0,   127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   190, 66,  54,  202,
    0,   0,   2,   196, 220, 255, 106, 32,  0,   0,   0,   0,   13,  116, 184,
    93,  4,   0,   0,   176, 114, 109, 159, 0,   52,  0,   1,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   29,  255, 22,  0,   0,   0,   0,   0,   0,
    72,  236, 0,   0,   0,   0,   99,  96,  52,  255, 11,  128, 0,   0,   240,
    240, 243, 255, 240, 240, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   140, 161, 0,   0,   0,   49,  255, 33,  216, 0,   54,  255,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   68,  236,
    45,  0,   0,   0,   0,   0,   27,  198, 133, 0,   0,   87,  189, 0,   52,
    255, 4,   0,   0,   19,  40,  0,   29,  197, 168, 0,   49,  255, 146, 9,
    15,  182, 176, 0,   0,   0,   0,   3,   226, 100, 0,   0,   0,   28,  210,
    252, 255, 182, 7,   0,   2,   223, 142, 6,   20,  186, 255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   43,  216, 3,   0,   0,   0,   0,
    158, 233, 162, 67,  1,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   23,  115, 210, 208, 0,   0,   0,   0,   176, 192, 6,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   7,   239, 9,
    118, 138, 0,   0,   0,   5,   109, 255, 153, 234, 112, 0,   0,   88,  179,
    76,  110, 231, 220, 0,   28,  255, 16,  0,   176, 110, 68,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   46,  255, 8,   0,   0,   0,   0,
    0,   0,   57,  253, 0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   244, 244, 244, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   17,  241, 43,  0,   0,   0,   37,  255, 17,  0,   0,
    66,  244, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   73,
    233, 60,  0,   0,   0,   0,   0,   0,   0,   69,  238, 0,   16,  227, 40,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   69,  246, 0,   38,  255,
    20,  0,   0,   69,  247, 0,   0,   0,   0,   79,  242, 11,  0,   0,   5,
    214, 122, 7,   16,  162, 175, 0,   0,   46,  193, 239, 210, 125, 241, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   202, 215, 114, 23,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   2,   67,  162, 236, 0,   0,   0,   34,  255, 31,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   236, 244,
    251, 240, 251, 244, 228, 0,   0,   0,   52,  255, 4,   76,  245, 0,   0,
    11,  0,   29,  254, 51,  101, 0,   40,  255, 30,  0,   18,  222, 193, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   29,  255, 23,  0,   0,
    0,   0,   0,   0,   72,  237, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   124, 178, 0,   0,   0,   0,   5,   245, 63,
    0,   0,   111, 200, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    87,  233, 54,  0,   0,   0,   0,   0,   0,   0,   0,   70,  248, 0,   52,
    255, 255, 255, 255, 255, 255, 0,   0,   0,   0,   0,   0,   68,  246, 0,
    6,   248, 19,  0,   0,   65,  247, 0,   0,   0,   0,   185, 151, 0,   0,
    0,   42,  255, 15,  0,   0,   59,  252, 0,   0,   0,   0,   0,   0,   124,
    188, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   46,  147, 234, 180, 86,  0,   0,   244, 244, 244,
    244, 244, 244, 0,   0,   39,  133, 226, 197, 97,  10,  0,   0,   0,   51,
    255, 4,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   140, 115, 12,  235, 8,   0,   0,   32,  73,  55,  255, 11,  143, 215,
    0,   0,   0,   0,   31,  255, 47,  97,  0,   1,   211, 174, 16,  6,   148,
    252, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   233, 65,
    0,   0,   0,   0,   0,   0,   116, 187, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    44,  220, 3,   0,   0,   0,   0,   9,   234, 58,  0,   0,   0,   0,   0,
    155, 186, 12,  28,  220, 103, 0,   0,   0,   8,   58,  255, 11,  8,   0,
    0,   105, 232, 47,  0,   0,   0,   0,   0,   101, 32,  0,   25,  193, 173,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   70,  25,  0,   25,  193,
    166, 0,   0,   165, 147, 7,   13,  176, 176, 0,   0,   0,   36,  254, 49,
    0,   0,   0,   15,  245, 120, 4,   12,  157, 210, 0,   0,   37,  12,  0,
    68,  239, 74,  0,   0,   0,   44,  220, 3,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   8,   92,  192, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   235, 142, 41,  0,   0,   0,   0,   0,
    0,   4,   24,  0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   49,  244, 3,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   222, 33,  87,  169, 0,   0,   0,   23,  186, 245, 255, 243,
    186, 39,  0,   0,   0,   0,   0,   122, 239, 229, 0,   0,   30,  181, 245,
    231, 147, 178, 0,   3,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    155, 133, 0,   0,   0,   0,   0,   0,   185, 111, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   62,  241, 1,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   108, 195, 0,   0,   0,   0,
    0,   0,   13,  165, 241, 235, 134, 1,   0,   0,   0,   255, 255, 255, 255,
    255, 0,   0,   255, 250, 244, 244, 244, 244, 0,   0,   143, 226, 244, 238,
    163, 17,  0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   134, 236, 245,
    236, 155, 13,  0,   0,   16,  167, 238, 242, 175, 22,  0,   0,   0,   139,
    203, 0,   0,   0,   0,   0,   68,  203, 246, 243, 185, 39,  0,   0,   104,
    235, 244, 212, 88,  0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   62,  241, 1,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   6,   0,   0,   0,   0,   0,
    0,   0,   0,   49,  244, 3,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   45,  225, 4,   0,   0,   0,   0,   29,  237, 16,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   120, 157, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   223, 75,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   120, 157, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   158, 104, 0,   0,   0,   0,   153, 115, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   181, 63,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   31,  119, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   181, 63,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   28,  160, 229, 236, 163,
    14,  0,   0,   0,   20,  252, 200, 0,   0,   0,   52,  255, 244, 243, 238,
    177, 32,  0,   0,   0,   96,  213, 244, 234, 137, 0,   52,  255, 244, 239,
    190, 68,  0,   0,   52,  255, 244, 244, 244, 244, 68,  0,   0,   52,  255,
    244, 244, 244, 244, 0,   0,   0,   104, 217, 244, 222, 100, 0,   52,  255,
    4,   0,   0,   52,  255, 0,   0,   244, 246, 255, 244, 244, 0,   0,   0,
    0,   137, 244, 246, 255, 4,   0,   52,  255, 4,   0,   0,   137, 216, 0,
    0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 163, 0,   0,   212, 255,
    0,   52,  255, 166, 0,   0,   52,  255, 0,   0,   15,  168, 241, 236, 141,
    2,   0,   52,  255, 244, 244, 239, 176, 33,  0,   0,   15,  168, 241, 236,
    141, 2,   0,   52,  255, 244, 244, 236, 171, 30,  0,   0,   47,  189, 243,
    241, 204, 73,  0,   0,   244, 244, 246, 255, 244, 244, 0,   52,  255, 4,
    0,   0,   52,  255, 0,   125, 203, 0,   0,   0,   24,  254, 0,   236, 75,
    0,   0,   0,   0,   152, 0,   44,  247, 51,  0,   0,   71,  241, 0,   220,
    112, 0,   0,   0,   161, 179, 0,   0,   244, 244, 244, 244, 249, 254, 0,
    0,   0,   52,  255, 244, 125, 0,   0,   35,  244, 23,  0,   0,   0,   0,
    0,   0,   0,   171, 246, 255, 4,   0,   0,   0,   0,   41,  242, 200, 7,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 24,  227, 130, 13,
    21,  206, 163, 0,   0,   0,   98,  207, 251, 26,  0,   0,   52,  255, 4,
    0,   8,   147, 204, 0,   0,   100, 224, 51,  0,   22,  89,  0,   52,  255,
    4,   7,   81,  238, 66,  0,   52,  255, 4,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   106, 221, 46,  0,   29,  89,  0,
    52,  255, 4,   0,   0,   52,  255, 0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   52,  255, 4,   0,   133, 218,
    26,  0,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 202, 4,   36,
    219, 255, 0,   52,  255, 242, 30,  0,   52,  255, 0,   0,   160, 175, 10,
    24,  213, 112, 0,   52,  255, 4,   0,   20,  166, 204, 0,   0,   160, 175,
    10,  24,  213, 112, 0,   52,  255, 4,   0,   17,  166, 201, 0,   8,   232,
    119, 7,   3,   56,  63,  0,   0,   0,   0,   52,  255, 4,   0,   0,   52,
    255, 4,   0,   0,   52,  255, 0,   48,  253, 19,  0,   0,   94,  226, 0,
    197, 107, 0,   0,   0,   0,   184, 0,   0,   131, 201, 2,   8,   220, 101,
    0,   74,  238, 18,  0,   52,  245, 33,  0,   0,   0,   0,   0,   1,   192,
    148, 0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   168, 135, 0,   0,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   16,  218, 106,
    178, 159, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 157, 170,
    0,   0,   0,   88,  243, 0,   0,   0,   179, 121, 200, 105, 0,   0,   52,
    255, 4,   0,   0,   63,  251, 0,   0,   227, 86,  0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   125, 186, 0,   52,  255, 4,   0,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   1,   229, 84,  0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   52,  255, 0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   52,  255, 4,   130,
    220, 27,  0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 141,
    66,  115, 140, 255, 0,   52,  255, 150, 143, 0,   52,  255, 0,   6,   246,
    54,  0,   0,   105, 204, 0,   52,  255, 4,   0,   0,   60,  253, 0,   6,
    246, 54,  0,   0,   105, 204, 0,   52,  255, 4,   0,   0,   59,  252, 0,
    46,  255, 15,  0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   52,  255, 4,   0,   0,   52,  255, 0,   0,   225, 87,  0,   0,   165,
    150, 0,   157, 139, 0,   234, 154, 0,   215, 0,   0,   7,   216, 102, 132,
    187, 0,   0,   0,   176, 146, 0,   194, 127, 0,   0,   0,   0,   0,   0,
    101, 223, 12,  0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   48,  239,
    14,  0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   3,   184,
    122, 0,   8,   193, 112, 0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    243, 67,  72,  217, 233, 162, 255, 0,   3,   12,  248, 51,  130, 186, 0,
    0,   52,  255, 4,   0,   12,  158, 197, 0,   31,  255, 22,  0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   69,  240, 0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   32,  255, 22,  0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   52,  255, 0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   52,  255,
    130, 244, 29,  0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   52,
    255, 61,  146, 188, 66,  255, 0,   52,  255, 37,  239, 16,  52,  255, 0,
    38,  255, 15,  0,   0,   64,  244, 0,   52,  255, 4,   0,   24,  169, 202,
    0,   38,  255, 15,  0,   0,   64,  244, 0,   52,  255, 4,   0,   16,  159,
    188, 0,   11,  235, 176, 72,  17,  0,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   52,  255, 4,   0,   0,   52,  255, 0,   0,   148, 157, 0,
    1,   234, 73,  0,   118, 170, 31,  225, 207, 0,   246, 0,   0,   0,   62,
    237, 237, 32,  0,   0,   0,   31,  241, 124, 221, 8,   0,   0,   0,   0,
    0,   25,  238, 68,  0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   184, 118, 0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    44,  101, 0,   0,   0,   14,  121, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 255, 32,  242, 102, 5,   147, 255, 0,   4,   86,  235, 1,   59,
    250, 16,  0,   52,  255, 240, 241, 254, 223, 32,  0,   48,  255, 6,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   55,  255, 0,   52,  255, 240,
    240, 240, 240, 22,  0,   0,   52,  255, 240, 240, 240, 202, 0,   48,  255,
    6,   0,   138, 241, 248, 0,   52,  255, 240, 240, 240, 243, 255, 0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    52,  255, 220, 235, 100, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   52,  255, 6,   209, 182, 52,  255, 0,   52,  255, 4,   169, 120, 52,
    255, 0,   49,  255, 5,   0,   0,   54,  255, 0,   52,  255, 240, 240, 231,
    169, 31,  0,   49,  255, 5,   0,   0,   54,  255, 0,   52,  255, 240, 244,
    255, 173, 11,  0,   0,   38,  160, 229, 253, 188, 37,  0,   0,   0,   0,
    52,  255, 4,   0,   0,   52,  255, 4,   0,   0,   52,  255, 0,   0,   70,
    227, 0,   51,  244, 7,   0,   78,  202, 83,  147, 220, 32,  251, 0,   0,
    0,   14,  235, 202, 0,   0,   0,   0,   0,   124, 255, 75,  0,   0,   0,
    0,   0,   0,   173, 156, 0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   64,  230, 7,   0,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 255, 54,  255, 13,  0,   62,  255, 0,   4,   167, 167,
    0,   4,   240, 92,  0,   52,  255, 4,   0,   18,  164, 181, 0,   31,  255,
    22,  0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   69,  240, 0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    32,  255, 20,  0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   52,  255,
    3,   0,   52,  255, 33,  86,  243, 35,  0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   52,  255, 4,   127, 89,  52,  255, 0,   52,  255, 4,   52,
    230, 58,  255, 0,   38,  255, 15,  0,   0,   64,  244, 0,   52,  255, 4,
    0,   0,   0,   0,   0,   38,  255, 15,  0,   0,   64,  247, 0,   52,  255,
    4,   1,   87,  251, 67,  0,   0,   0,   0,   0,   32,  176, 207, 0,   0,
    0,   0,   52,  255, 4,   0,   0,   51,  255, 4,   0,   0,   52,  255, 0,
    0,   6,   242, 42,  121, 174, 0,   0,   39,  234, 136, 91,  168, 112, 218,
    0,   0,   0,   153, 180, 211, 98,  0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   80,  228, 15,  0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   200, 101, 0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 255, 37,  243, 102, 5,   147, 255, 0,   11,
    241, 250, 248, 248, 253, 173, 0,   52,  255, 4,   0,   0,   58,  249, 0,
    0,   228, 84,  0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   126, 186,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   1,   230, 78,  0,   0,   52,  255, 0,   52,  255, 4,   0,   0,
    52,  255, 0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,
    61,  250, 0,   0,   52,  255, 4,   0,   170, 199, 3,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   52,  255, 4,   0,   0,   52,  255, 0,   52,  255,
    4,   0,   192, 149, 255, 0,   6,   247, 54,  0,   0,   104, 204, 0,   52,
    255, 4,   0,   0,   0,   0,   0,   6,   247, 54,  0,   0,   104, 209, 0,
    52,  255, 4,   0,   0,   151, 207, 0,   0,   0,   0,   0,   0,   61,  253,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   40,  255, 8,   0,   0,   56,
    248, 0,   0,   0,   171, 112, 192, 97,  0,   0,   4,   250, 199, 35,  111,
    196, 178, 0,   0,   66,  241, 28,  64,  236, 21,  0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   14,  228, 75,  0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   80,  218, 2,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 238, 80,  75,  219, 234, 164, 248,
    0,   77,  249, 16,  0,   0,   87,  245, 0,   52,  255, 4,   0,   9,   145,
    216, 0,   0,   102, 222, 50,  0,   22,  89,  0,   52,  255, 4,   7,   82,
    239, 67,  0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   109, 215, 40,  0,   81,  255, 0,   52,  255, 4,
    0,   0,   52,  255, 0,   0,   0,   52,  255, 4,   0,   0,   0,   43,  112,
    13,  3,   145, 201, 0,   0,   52,  255, 4,   0,   20,  233, 126, 0,   0,
    52,  255, 11,  8,   8,   8,   0,   52,  255, 4,   0,   0,   52,  255, 0,
    52,  255, 4,   0,   75,  246, 255, 0,   0,   163, 173, 9,   23,  212, 114,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   162, 173, 9,   23,  212,
    123, 0,   52,  255, 4,   0,   0,   25,  246, 0,   14,  123, 29,  0,   18,
    163, 202, 0,   0,   0,   0,   52,  255, 4,   0,   0,   5,   232, 107, 3,
    9,   151, 190, 0,   0,   0,   93,  195, 248, 21,  0,   0,   0,   216, 233,
    0,   54,  254, 139, 0,   10,  221, 108, 0,   0,   168, 163, 0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   151, 163, 0,   0,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   1,   214, 84,  0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 146, 193, 1,   0,   0,
    0,   0,   0,   155, 179, 0,   0,   0,   11,  245, 0,   52,  255, 244, 243,
    239, 184, 44,  0,   0,   0,   99,  214, 244, 237, 140, 0,   52,  255, 244,
    240, 191, 69,  0,   0,   52,  255, 244, 244, 244, 244, 99,  0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   109, 219, 243, 223, 116, 0,   52,
    255, 4,   0,   0,   52,  255, 0,   0,   244, 246, 255, 244, 244, 0,   0,
    27,  186, 240, 244, 204, 50,  0,   0,   52,  255, 4,   0,   0,   82,  251,
    0,   0,   52,  255, 255, 255, 255, 255, 0,   52,  255, 4,   0,   0,   52,
    255, 0,   52,  255, 4,   0,   1,   213, 255, 0,   0,   16,  171, 241, 237,
    143, 3,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   16,  170, 241,
    254, 187, 6,   0,   52,  255, 4,   0,   0,   0,   138, 0,   9,   171, 229,
    246, 240, 177, 33,  0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   51,
    197, 244, 241, 177, 28,  0,   0,   0,   18,  252, 198, 0,   0,   0,   0,
    176, 179, 0,   5,   247, 99,  0,   144, 199, 2,   0,   0,   27,  243, 0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   254, 247, 244, 244, 244, 244,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   96,  203,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 16,  216, 165,
    31,  2,   24,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   87,  237, 57,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,
    5,   226, 68,  0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    19,  144, 222, 242, 159, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   58,  12,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   50,  248, 240, 123, 0,   0,   0,   0,
    0,   0,   0,   71,  79,  0,   0,   0,   168, 241, 248, 3,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   244, 244, 244, 244, 244, 244,
    99,  0,   0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   9,   124, 20,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   140, 0,   127, 0,   0,   112, 174, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   4,   221, 0,   127, 0,   0,   0,   159,
    98,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   52,  255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   2,   166, 243, 244, 102, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   49,  244, 3,   0,
    0,   0,   0,   0,   49,  244, 3,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   11,  244, 246, 255, 4,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   158, 239, 209, 0,   0,   0,   0,   52,  255,
    4,   0,   0,   11,  244, 233, 119, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   68,  162, 0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   46,  255, 24,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   37,  255, 46,  0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   96,  244, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   149, 81,  0,   127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   103, 213, 244, 239, 180, 36,
    0,   52,  255, 122, 232, 241, 159, 9,   0,   0,   12,  151, 233, 244, 202,
    40,  0,   0,   26,  183, 244, 223, 133, 255, 0,   0,   13,  156, 236, 241,
    165, 12,  0,   19,  244, 246, 255, 244, 244, 102, 0,   0,   25,  182, 244,
    221, 127, 255, 0,   52,  255, 111, 233, 235, 80,  0,   0,   0,   0,   244,
    246, 255, 4,   0,   0,   0,   45,  244, 246, 255, 4,   0,   0,   0,   52,
    255, 4,   0,   122, 210, 0,   0,   0,   52,  255, 4,   0,   0,   0,   255,
    173, 246, 137, 195, 245, 114, 0,   52,  255, 111, 233, 235, 80,  0,   0,
    0,   29,  182, 242, 237, 157, 11,  0,   52,  255, 121, 231, 241, 161, 10,
    0,   0,   26,  183, 244, 222, 130, 255, 0,   0,   52,  255, 78,  217, 244,
    175, 0,   0,   0,   109, 226, 245, 229, 114, 0,   19,  244, 246, 255, 244,
    244, 152, 0,   52,  255, 4,   0,   52,  255, 4,   0,   51,  242, 9,   0,
    0,   73,  228, 0,   226, 63,  0,   0,   0,   0,   140, 0,   13,  219, 93,
    0,   0,   173, 156, 0,   40,  250, 19,  0,   0,   50,  245, 0,   0,   209,
    244, 244, 246, 255, 0,   0,   0,   0,   0,   51,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 3,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   5,   8,   0,   1,   219, 10,
    0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   93,  52,  2,   9,
    162, 204, 0,   52,  255, 163, 9,   19,  198, 144, 0,   0,   174, 188, 24,
    0,   53,  43,  0,   0,   193, 157, 7,   20,  200, 255, 0,   0,   173, 186,
    16,  11,  179, 152, 0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   189,
    167, 10,  19,  199, 255, 0,   52,  255, 121, 2,   133, 219, 0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   52,  255, 4,   133, 204, 19,  0,   0,   0,   52,  255, 4,   0,   0,
    0,   255, 51,  99,  255, 55,  96,  227, 0,   52,  255, 121, 2,   133, 219,
    0,   0,   0,   198, 158, 8,   19,  198, 149, 0,   52,  255, 163, 9,   19,
    198, 145, 0,   0,   192, 158, 8,   20,  198, 255, 0,   0,   52,  255, 169,
    20,  2,   52,  0,   0,   31,  255, 62,  0,   20,  57,  0,   0,   0,   52,
    255, 4,   0,   0,   0,   52,  255, 4,   0,   52,  255, 4,   0,   0,   216,
    84,  0,   0,   162, 139, 0,   166, 118, 0,   0,   0,   0,   195, 0,   0,
    49,  235, 36,  104, 211, 9,   0,   0,   200, 106, 0,   0,   142, 163, 0,
    0,   0,   0,   0,   149, 171, 0,   0,   0,   0,   0,   54,  255, 2,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   50,  255, 5,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   95,  232, 100, 0,   55,
    175, 0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   61,  252, 0,   52,  255, 33,  0,   0,   81,  233, 0,   22,  255,
    41,  0,   0,   0,   0,   0,   25,  255, 32,  0,   0,   82,  255, 0,   21,
    255, 49,  0,   0,   70,  236, 0,   0,   0,   52,  255, 4,   0,   0,   0,
    24,  255, 35,  0,   0,   82,  255, 0,   52,  255, 18,  0,   56,  254, 1,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   52,  255, 146, 227, 15,  0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   255, 8,   56,  255, 8,   56,  253, 0,   52,  255, 18,  0,
    56,  254, 1,   0,   27,  255, 32,  0,   0,   81,  234, 0,   52,  255, 33,
    0,   0,   81,  232, 0,   25,  255, 32,  0,   0,   81,  255, 0,   0,   52,
    255, 41,  0,   0,   0,   0,   0,   26,  252, 96,  11,  0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   52,  255, 4,   0,   52,  255, 4,   0,
    0,   125, 173, 0,   8,   242, 48,  0,   106, 173, 0,   192, 118, 4,   246,
    0,   0,   0,   106, 212, 234, 40,  0,   0,   0,   103, 199, 0,   3,   230,
    67,  0,   0,   0,   0,   90,  216, 13,  0,   0,   0,   0,   1,   121, 226,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   22,  252,
    74,  0,   0,   0,   0,   115, 225, 233, 149, 42,  25,  0,   15,  54,  177,
    0,   136, 94,  0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    78,  197, 237, 240, 243, 255, 0,   52,  255, 6,   0,   0,   55,  254, 0,
    47,  255, 7,   0,   0,   0,   0,   0,   48,  255, 6,   0,   0,   55,  255,
    0,   47,  255, 240, 240, 240, 241, 247, 0,   0,   0,   52,  255, 4,   0,
    0,   0,   48,  255, 7,   0,   0,   55,  255, 0,   52,  255, 4,   0,   52,
    255, 4,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   52,  255, 187, 238, 80,  0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   255, 4,   52,  255, 4,   52,  255, 0,   55,  255,
    4,   0,   52,  255, 4,   0,   48,  255, 6,   0,   0,   55,  254, 0,   52,
    255, 6,   0,   0,   55,  254, 0,   48,  255, 6,   0,   0,   55,  255, 0,
    0,   52,  255, 6,   0,   0,   0,   0,   0,   0,   84,  186, 239, 208, 75,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   52,  255, 4,   0,   52,  255,
    4,   0,   0,   35,  246, 13,  84,  213, 0,   0,   46,  228, 16,  202, 187,
    49,  226, 0,   0,   0,   13,  239, 176, 0,   0,   0,   0,   15,  246, 36,
    70,  225, 1,   0,   0,   0,   43,  233, 43,  0,   0,   0,   0,   33,  247,
    247, 73,  0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   112, 254, 237, 0,   0,   0,   139, 14,  17,  117, 219, 229, 0,   0,
    1,   219, 8,   211, 18,  0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   24,  251, 86,  5,   0,   65,  255, 0,   52,  255, 32,  0,   0,   80,
    232, 0,   22,  255, 41,  0,   0,   0,   0,   0,   25,  255, 32,  0,   0,
    81,  255, 0,   22,  255, 21,  0,   0,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   25,  255, 34,  0,   0,   81,  255, 0,   52,  255, 4,
    0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   52,  255, 8,   84,  237, 29,  0,   0,
    0,   51,  255, 5,   0,   0,   0,   255, 4,   52,  255, 4,   52,  255, 0,
    56,  255, 4,   0,   52,  255, 4,   0,   27,  255, 32,  0,   0,   81,  234,
    0,   52,  255, 32,  0,   0,   80,  233, 0,   26,  255, 32,  0,   0,   81,
    255, 0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   4,
    126, 237, 0,   0,   0,   51,  255, 4,   0,   0,   0,   48,  255, 8,   0,
    66,  255, 4,   0,   0,   0,   199, 94,  173, 122, 0,   0,   2,   239, 110,
    116, 189, 116, 166, 0,   0,   0,   166, 166, 222, 90,  0,   0,   0,   0,
    166, 128, 162, 131, 0,   0,   0,   13,  216, 90,  0,   0,   0,   0,   0,
    0,   3,   137, 210, 0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   14,  244, 89,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   154, 117, 188, 0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   33,  255, 68,  0,   26,  189, 255, 0,   52,  255, 160, 8,
    17,  195, 143, 0,   0,   177, 185, 23,  0,   50,  42,  0,   0,   194, 158,
    7,   19,  198, 255, 0,   0,   176, 162, 17,  0,   32,  76,  0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   191, 164, 10,  17,  195, 255, 0,   52,
    255, 4,   0,   52,  255, 4,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   52,  255, 4,   0,   156, 194,
    0,   0,   0,   24,  254, 67,  0,   0,   0,   255, 4,   52,  255, 4,   52,
    255, 0,   56,  255, 4,   0,   52,  255, 4,   0,   0,   200, 158, 7,   19,
    198, 151, 0,   52,  255, 160, 8,   17,  195, 146, 0,   0,   194, 158, 7,
    19,  198, 255, 0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   31,  72,
    8,   2,   124, 230, 0,   0,   0,   30,  255, 51,  0,   0,   0,   15,  251,
    62,  0,   147, 255, 4,   0,   0,   0,   108, 195, 246, 32,  0,   0,   0,
    182, 228, 42,  120, 228, 106, 0,   0,   103, 217, 11,  55,  238, 39,  0,
    0,   0,   70,  222, 242, 36,  0,   0,   0,   172, 148, 0,   0,   0,   0,
    0,   0,   0,   0,   57,  254, 1,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   49,  255, 8,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   78,  244, 107, 0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   114, 231, 243, 204, 108, 255, 0,   52,  255,
    120, 232, 240, 159, 9,   0,   0,   14,  153, 234, 244, 206, 41,  0,   0,
    28,  184, 244, 223, 132, 255, 0,   0,   14,  155, 234, 244, 218, 111, 0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   27,  184, 245, 221, 123, 255,
    0,   52,  255, 4,   0,   52,  255, 4,   0,   0,   118, 244, 246, 255, 244,
    244, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   52,  255, 4,   0,
    11,  217, 0,   0,   0,   0,   121, 240, 244, 114, 0,   255, 4,   52,  255,
    4,   52,  255, 0,   56,  255, 4,   0,   52,  255, 4,   0,   0,   31,  184,
    243, 238, 160, 12,  0,   52,  255, 120, 232, 241, 160, 10,  0,   0,   28,
    185, 244, 223, 130, 255, 0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    35,  199, 242, 246, 204, 61,  0,   0,   0,   0,   141, 236, 229, 142, 0,
    0,   120, 234, 191, 137, 255, 4,   0,   0,   0,   21,  251, 197, 0,   0,
    0,   0,   122, 224, 0,   45,  255, 46,  0,   48,  241, 50,  0,   0,   118,
    210, 0,   0,   0,   2,   227, 198, 0,   0,   0,   0,   255, 246, 244, 244,
    244, 0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   10,  245, 27,  0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    72,  240, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   56,  254, 1,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   199, 105, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   50,  255, 5,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   53,  255, 2,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   6,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   53,  55,
    2,   15,  182, 162, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   121, 215, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   64,  244, 16,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   25,  255, 56,  0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   106, 232, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    75,  220, 246, 237, 162, 16,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   194, 246, 227, 68,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   203, 237, 92,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   133, 229,
    206, 0,   0,   0,   0,   52,  255, 4,   0,   0,   11,  240, 221, 96,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   2,   187,
    58,  108, 143, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   102, 10,  58,  54,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   97,  59,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   33,  222, 208, 8,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   60,  194, 183, 6,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   48,  240, 52,  240, 3,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   58,  199, 211, 11,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   184, 184, 143, 119, 77,  39,  156, 0,   0,   0,   73,  164, 22,  196,
    26,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   42,  179, 3,
    188, 30,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   4,
    134, 230, 245, 210, 44,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   57,  219, 243, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   4,   195, 31,  94,  137, 0,   0,   109, 231, 221, 73,  0,
    0,   0,   0,   0,   47,  189, 243, 241, 204, 73,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   26,  175, 241, 247, 255, 244, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   244, 244, 244, 244, 249, 254, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   111, 129, 0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   111, 178, 0,   111, 129, 0,   0,   52,  255, 4,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   156, 224, 40,
    191, 0,   0,   0,   196, 0,   196, 159, 147, 211, 0,   0,   0,   0,   125,
    231, 62,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    143, 189, 126, 0,   0,   0,   171, 161, 0,   0,   1,   208, 134, 0,   127,
    0,   137, 197, 27,  0,   51,  39,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   218, 103,
    1,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   254, 51,  101,
    235, 0,   0,   0,   0,   8,   233, 119, 7,   3,   56,  63,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   181, 177, 20,  52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   192, 148, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   206, 69,  0,   0,   0,   0,   0,   0,   62,  241, 1,
    0,   0,   0,   0,   206, 94,  0,   206, 69,  0,   0,   62,  241, 1,   66,
    241, 1,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   174,
    39,  188, 126, 0,   0,   0,   196, 0,   196, 55,  118, 196, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   15,  125, 8,   0,   0,   0,   29,  241, 39,  0,   85,  240, 17,
    0,   127, 8,   244, 57,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   31,
    249, 5,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   118, 244, 246, 255, 244, 244, 72,  0,   118, 244,
    246, 255, 244, 244, 72,  0,   0,   0,   0,   0,   0,   0,   0,   0,   254,
    50,  100, 234, 1,   57,  97,  0,   46,  255, 15,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   31,  0,   0,   0,   10,  252, 51,  0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   101, 223,
    12,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   35,  255, 14,  0,   0,   0,   0,   0,   0,   118,
    157, 0,   0,   0,   0,   35,  255, 18,  35,  255, 14,  0,   0,   120, 157,
    0,   143, 157, 0,   0,   0,   0,   84,  234, 231, 76,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   196, 0,   196, 0,   0,   196, 0,
    0,   0,   109, 226, 245, 229, 114, 0,   0,   0,   0,   30,  0,   0,   0,
    0,   0,   103, 239, 220, 122, 231, 236, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   209, 244, 244, 246, 255, 0,   0,   0,   120, 167, 2,   213,
    127, 0,   0,   127, 240, 255, 241, 240, 236, 18,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   74,  210, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   111, 234, 255, 210, 148, 92,  14,  0,   10,  232, 176, 72,  17,  0,
    0,   0,   0,   0,   0,   55,  219, 0,   0,   0,   40,  255, 14,  0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   25,
    238, 68,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    0,   178, 63,  0,   0,   0,   0,   52,  255, 4,   52,  255, 4,   0,   0,
    181, 63,  0,   225, 63,  0,   0,   0,   0,   234, 255, 255, 232, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   31,  255, 62,  0,   20,  57,  0,   0,   0,   0,   218, 54,
    0,   0,   0,   5,   243, 62,  113, 255, 57,  106, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   149, 171, 0,   0,   0,   6,   214,
    135, 237, 14,  0,   0,   127, 51,  255, 4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   22,  240, 247, 250, 240, 150, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   129, 149, 89,  13,  0,   0,   0,   0,   0,   33,  152, 222,
    253, 188, 37,  0,   0,   0,   85,  217, 46,  0,   0,   0,   49,  255, 6,
    0,   52,  255, 240, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   173, 156, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   234, 255, 255, 232,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   26,  252, 96,  11,  0,   0,   0,   0,   0,   0,
    46,  217, 85,  0,   0,   39,  255, 11,  62,  255, 6,   54,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   90,  216, 13,  0,   0,   0,
    0,   68,  255, 121, 0,   0,   0,   127, 240, 255, 240, 240, 50,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   156, 126, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   113, 231, 221, 73,  73,  223, 225, 0,   0,   0,
    0,   0,   27,  170, 207, 0,   0,   0,   229, 102, 0,   0,   0,   0,   40,
    255, 14,  0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   80,  228, 15,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   84,  235,
    233, 79,  0,   0,   244, 244, 244, 244, 244, 244, 244, 0,   244, 244, 244,
    244, 244, 244, 244, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   84,  186, 239, 208, 75,  0,   0,
    0,   0,   0,   103, 229, 0,   0,   50,  255, 5,   53,  255, 240, 241, 0,
    3,   0,   0,   0,   0,   0,   0,   0,   0,   0,   43,  233, 43,  0,   0,
    0,   0,   0,   0,   252, 60,  0,   0,   0,   127, 9,   247, 62,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   200, 85,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   116, 240, 243, 255, 240, 240, 71,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   254, 51,  102, 235, 229, 51,  101, 0,
    0,   0,   0,   0,   0,   61,  253, 0,   0,   0,   29,  206, 115, 0,   0,
    0,   10,  252, 51,  0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   14,  228, 75,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   4,   126, 237,
    0,   0,   0,   0,   116, 206, 29,  0,   0,   39,  255, 11,  60,  255, 8,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   13,  216, 90,  0,
    0,   0,   0,   0,   0,   0,   252, 60,  0,   0,   0,   127, 0,   146, 201,
    26,  0,   50,  39,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   1,   243, 45,  0,   0,   0,   0,
    52,  255, 4,   52,  255, 4,   0,   44,  220, 3,   44,  220, 3,   44,  0,
    3,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   254, 47,  99,  235, 230, 47,
    97,  0,   15,  135, 41,  5,   30,  171, 202, 0,   0,   0,   0,   12,  180,
    0,   0,   0,   0,   181, 177, 19,  52,  255, 4,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   151, 163, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   31,  72,  8,   2,
    124, 230, 0,   0,   0,   0,   179, 12,  0,   0,   0,   6,   244, 62,  114,
    255, 80,  0,   0,   1,   0,   0,   0,   0,   0,   0,   0,   0,   172, 148,
    0,   0,   0,   0,   0,   0,   0,   0,   252, 60,  0,   0,   0,   127, 0,
    6,   140, 231, 245, 213, 44,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   62,  241, 1,   0,   0,   0,   0,   0,   33,  253, 7,   0,   0,
    0,   0,   62,  241, 1,   66,  241, 1,   0,   52,  255, 4,   52,  255, 4,
    52,  0,   4,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   116, 239, 229, 79,
    79,  231, 233, 0,   9,   166, 229, 252, 240, 177, 33,  0,   0,   0,   0,
    0,   3,   0,   0,   0,   0,   26,  177, 242, 247, 255, 244, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   254, 247, 244, 244, 244, 244, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   35,  199,
    242, 246, 204, 61,  0,   0,   0,   0,   2,   0,   0,   0,   0,   0,   105,
    240, 221, 111, 222, 244, 0,   1,   0,   0,   0,   0,   0,   0,   0,   0,
    255, 246, 244, 244, 244, 0,   0,   0,   0,   0,   252, 60,  0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   120, 157, 0,   0,   0,   0,   0,   0,   85,  212, 0,
    0,   0,   0,   0,   120, 157, 0,   143, 157, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   181, 63,  0,   0,   0,   0,   30,  11,  195,
    131, 0,   0,   0,   0,   0,   181, 63,  0,   225, 63,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   227,
    245, 177, 12,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   58,  98,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   2,   8,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   20,  221, 50,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   103, 185, 255, 4,   0,   0,   0,   0,   106, 186, 255,
    4,   0,   0,   0,   0,   0,   144, 235, 229, 126, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   49,  244, 3,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   86,  224, 245,
    206, 0,   0,   0,   0,   0,   0,   0,   0,   0,   215, 116, 0,   0,   0,
    164, 172, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   110, 227, 245,
    223, 102, 0,   0,   0,   49,  244, 53,  244, 3,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   122, 237, 237, 102, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   3,   244, 244, 244, 175, 0,   0,
    0,   0,   116, 239, 227, 77,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   116, 235, 233, 108, 0,   0,   0,   0,   165, 239, 235, 132,
    0,   0,   0,   0,   0,   173, 88,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   40,  184, 245, 255, 246, 255, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   101,
    185, 255, 4,   0,   0,   0,   0,   77,  219, 247, 201, 45,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   17,  0,   101, 238, 0,   0,   0,
    0,   0,   49,  244, 3,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   11,  245,
    105, 1,   43,  0,   0,   0,   0,   0,   0,   0,   0,   0,   62,  241, 25,
    0,   61,  236, 23,  0,   0,   0,   0,   52,  255, 4,   0,   0,   33,  255,
    60,  0,   26,  51,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    53,  181, 236, 230, 161, 31,  0,   0,   0,   33,  1,   102, 238, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   53,  181, 236, 230, 161, 31,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   30,  254, 51,  99,  236, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   42,  3,   96,  248, 0,   0,   0,   0,   15,  2,
    107, 236, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   5,   226, 255, 255, 255, 56,  255, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   14,  244, 93,  2,   142, 209, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   222, 242, 72,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   14,  161, 243, 255, 236, 0,   0,   0,
    47,  255, 11,  0,   0,   0,   0,   82,  25,  0,   0,   44,  64,  0,   0,
    155, 165, 3,   210, 93,  0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    15,  230, 104, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   67,  255, 255, 253, 224, 140, 233, 0,   0,   3,   160, 237, 243, 255,
    3,   0,   0,   0,   0,   31,  0,   0,   31,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   67,  255, 255, 251, 251, 255, 233, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   30,  254, 50,  100, 234, 0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   0,   113, 173, 0,   0,   0,   0,
    0,   244, 239, 67,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   52,  255, 4,   0,   45,  255, 255, 255, 255, 56,  255, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   47,  255, 12,  0,   61,
    253, 0,   0,   30,  0,   0,   30,  0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   108,
    224, 0,   0,   0,   0,   0,   44,  255, 3,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   174, 182, 65,  255, 13,  0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   79,  222, 204, 189, 221, 47,
    0,   123, 185, 244, 166, 235, 176, 90,  0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   100, 237, 188, 46,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   222, 255, 110, 2,   21,  6,   156, 0,   0,   42,  255, 40,
    100, 255, 4,   0,   0,   0,   55,  219, 0,   55,  218, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   222, 161, 255, 4,   86,  247, 157, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   114, 233, 220, 71,  0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   65,  191, 12,  0,   0,
    0,   0,   0,   0,   107, 224, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   52,  255, 4,   0,   19,  250, 255, 255, 255, 56,
    255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   12,  243, 94,
    4,   144, 206, 0,   0,   218, 54,  0,   219, 54,  0,   0,   81,  176, 212,
    160, 0,   28,  70,  0,   81,  176, 212, 168, 31,  8,   78,  0,   0,   35,
    1,   114, 240, 0,   0,   0,   0,   0,   63,  242, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   22,  255, 48,  52,  255,
    4,   0,   0,   172, 243, 255, 240, 240, 37,  0,   0,   17,  230, 49,  97,
    201, 0,   0,   0,   0,   101, 255, 50,  0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   21,  249, 20,  108, 240, 103, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   255, 255, 14,  0,   0,   0,   67,  0,   0,   5,
    182, 234, 147, 248, 3,   0,   0,   85,  219, 47,  85,  212, 43,  0,   186,
    244, 244, 244, 244, 245, 252, 0,   255, 71,  255, 244, 232, 84,  67,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   240, 240, 243, 255, 240, 240, 0,   0,   0,   56,  182, 13,  0,
    0,   0,   0,   0,   42,  1,   116, 239, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   52,  255, 4,   0,   0,   105, 248, 255,
    255, 56,  255, 0,   0,   0,   44,  220, 3,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   81,  176, 212, 168, 49,  0,   0,   0,
    72,  213, 242, 195, 40,  0,   0,   44,  213, 85,  48,  220, 85,  0,   0,
    2,   56,  132, 204, 195, 104, 0,   0,   0,   12,  86,  169, 231, 170, 0,
    0,   180, 240, 219, 87,  42,  91,  0,   0,   15,  210, 132, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   48,  255, 11,
    52,  255, 4,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   16,  229,
    53,  101, 198, 0,   0,   168, 240, 243, 255, 240, 240, 123, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   30,  254, 67,  0,   93,  244, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   255, 222, 111, 4,   17,  5,   67,  0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   229, 109, 0,   229, 89,  0,
    0,   0,   0,   0,   0,   0,   52,  255, 0,   255, 71,  255, 9,   178, 76,
    67,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   173, 242,
    240, 240, 11,  0,   0,   0,   200, 241, 217, 82,  0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   52,  255, 4,   0,   0,   0,
    18,  107, 255, 56,  255, 0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   89,  229, 0,   109, 229,
    0,   150, 216, 167, 92,  18,  0,   0,   0,   92,  177, 230, 162, 79,  8,
    0,   0,   21,  98,  143, 151, 148, 110, 60,  0,   0,   186, 168, 2,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   21,
    255, 43,  52,  255, 4,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    76,  223, 204, 189, 222, 49,  0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   117, 241, 119, 83,  210, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   224, 153, 197, 240, 202, 13,
    157, 0,   0,   3,   240, 240, 240, 240, 11,  0,   0,   29,  207, 117, 29,
    205, 112, 0,   0,   0,   0,   0,   0,   50,  248, 0,   224, 156, 228, 3,
    22,  188, 166, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 11,  0,   62,  255, 4,   0,
    0,   0,   0,   52,  255, 56,  255, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   228, 240, 240, 240, 172, 0,   0,   113, 205, 29,  118,
    207, 29,  0,   49,  4,   0,   0,   146, 255, 4,   0,   130, 71,  5,   87,
    225, 232, 129, 0,   21,  53,  7,   0,   2,   174, 255, 0,   40,  255, 20,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   175, 172, 59,  255, 8,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   85,  25,  0,   0,   46,  62,  0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   43,  182, 246,
    51,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   68,  240, 97,  7,
    14,  129, 232, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   12,
    180, 0,   12,  180, 0,   0,   0,   0,   0,   0,   0,   0,   0,   68,  240,
    97,  7,   14,  129, 232, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 99,  3,   157, 255,
    18,  0,   0,   0,   0,   52,  255, 56,  255, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   179, 12,
    0,   179, 12,  0,   0,   0,   0,   0,   44,  141, 255, 4,   0,   0,   0,
    0,   28,  5,   88,  239, 0,   0,   0,   0,   0,   126, 102, 255, 0,   23,
    252, 95,  8,   66,  99,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   13,  156, 235, 255, 231, 0,   0,   244, 246, 255, 244,
    244, 244, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    0,   127, 213, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   53,
    184, 236, 232, 164, 31,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   3,   0,   0,   3,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   53,  184, 236, 232, 164, 31,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   244, 244, 244, 244, 244,
    244, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 163, 243,
    148, 208, 191, 0,   0,   0,   0,   52,  255, 56,  255, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    2,   0,   0,   2,   0,   0,   0,   0,   0,   0,   136, 53,  255, 4,   0,
    0,   0,   0,   0,   9,   168, 67,  0,   0,   0,   0,   84,  94,  52,  255,
    0,   0,   101, 231, 249, 202, 68,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   1,
    72,  10,  3,   120, 233, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 56,  255, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   33,  65,  0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   240, 243, 255,
    176, 0,   0,   0,   0,   28,  162, 38,  0,   0,   0,   0,   0,   174, 240,
    243, 255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   2,   188, 234, 241, 201, 61,  0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   23,  116, 25,  116,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   2,   214, 0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   143, 245, 244, 244, 0,   0,   0,   0,
    0,   0,   52,  255, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   116, 199,
    178, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 0,   0,   51,  90,
    0,   0,   0,   0,   0,   0,   0,   6,   119, 17,  0,   0,   0,   0,   4,
    117, 84,  0,   0,   0,   0,   2,   196, 190, 20,  191, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   145, 237, 102, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   99,  44,  0,   0,   0,   0,   0,   0,   0,   48,  95,  0,   0,
    0,   0,   0,   41,  128, 38,  0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   97,  35,  0,   0,   0,   0,   0,   0,   0,   57,  85,
    0,   0,   0,   0,   0,   46,  127, 26,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   18,  217,
    181, 17,  203, 0,   0,   0,   0,   74,  213, 6,   0,   0,   0,   0,   0,
    0,   27,  229, 40,  0,   0,   0,   0,   29,  222, 209, 8,   0,   0,   0,
    2,   195, 200, 27,  193, 1,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   101, 212, 8,   0,   0,   0,   0,   0,   0,   29,  234, 61,
    0,   0,   0,   0,   42,  226, 218, 16,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   143, 135, 0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,
    3,   172, 83,  0,   0,   0,   0,   0,   0,   158, 105, 0,   0,   0,   0,
    0,   154, 91,  157, 84,  0,   0,   0,   34,  144, 63,  189, 101, 0,   0,
    0,   48,  240, 52,  240, 3,   0,   0,   0,   32,  255, 96,  239, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   39,  201, 20,  0,   0,   0,   0,   0,   25,  203, 35,
    0,   0,   0,   0,   24,  188, 62,  188, 22,  0,   0,   0,   48,  240, 52,
    240, 3,   0,   0,   0,   0,   42,  195, 12,  0,   0,   0,   0,   0,   29,
    198, 25,  0,   0,   0,   0,   22,  184, 66,  185, 14,  0,   0,   0,   48,
    240, 52,  240, 3,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    74,  110, 77,  192, 109, 0,   0,   0,   0,   0,   139, 130, 0,   0,   0,
    0,   0,   0,   177, 92,  0,   0,   0,   0,   1,   183, 63,  107, 141, 0,
    0,   0,   37,  141, 60,  187, 123, 0,   0,   0,   48,  240, 3,   48,  240,
    3,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   151, 143, 0,   0,   0,   0,   0,   1,   189,
    104, 0,   0,   0,   0,   8,   200, 53,  93,  169, 0,   0,   0,   48,  240,
    3,   48,  240, 3,   0,   0,   0,   42,  192, 5,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   20,  246, 100, 217,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   9,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   0,   20,  252, 200, 0,   0,   0,   0,   0,   20,  252, 200,
    0,   0,   0,   0,   0,   20,  252, 200, 0,   0,   0,   0,   0,   20,  252,
    200, 0,   0,   0,   0,   0,   80,  255, 32,  0,   0,   0,   0,   0,   62,
    238, 30,  0,   0,   0,   0,   0,   196, 247, 255, 244, 244, 0,   0,   0,
    96,  213, 244, 234, 137, 0,   52,  255, 244, 244, 244, 244, 68,  0,   52,
    255, 244, 244, 244, 244, 68,  0,   52,  255, 244, 244, 244, 244, 68,  0,
    52,  255, 244, 244, 244, 236, 0,   0,   0,   244, 246, 255, 244, 244, 0,
    0,   0,   244, 246, 255, 244, 244, 0,   0,   0,   244, 246, 255, 244, 244,
    0,   0,   0,   244, 246, 255, 244, 244, 0,   0,   52,  255, 244, 239, 190,
    68,  0,   0,   52,  255, 166, 0,   0,   52,  255, 0,   0,   15,  168, 241,
    236, 141, 2,   0,   0,   15,  168, 241, 236, 141, 2,   0,   0,   15,  168,
    241, 236, 141, 2,   0,   0,   15,  168, 241, 236, 141, 2,   0,   0,   15,
    168, 241, 236, 141, 2,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    13,  166, 240, 239, 148, 152, 0,   52,  255, 4,   0,   0,   52,  255, 0,
    52,  255, 4,   0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255,
    0,   52,  255, 4,   0,   0,   52,  255, 0,   220, 112, 0,   0,   0,   161,
    179, 0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   100, 226, 247, 212,
    58,  0,   0,   127, 0,   0,   98,  207, 251, 26,  0,   0,   0,   0,   98,
    207, 251, 26,  0,   0,   0,   0,   98,  207, 251, 26,  0,   0,   0,   0,
    98,  207, 251, 26,  0,   0,   0,   0,   160, 212, 112, 0,   0,   0,   0,
    0,   91,  210, 42,  0,   0,   0,   0,   17,  242, 58,  255, 4,   0,   0,
    0,   100, 223, 50,  0,   22,  89,  0,   52,  255, 4,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   52,  255, 4,
    7,   81,  238, 66,  0,   52,  255, 242, 30,  0,   52,  255, 0,   0,   160,
    175, 10,  24,  213, 112, 0,   0,   160, 175, 10,  24,  213, 112, 0,   0,
    160, 175, 10,  24,  213, 112, 0,   0,   160, 175, 10,  24,  213, 112, 0,
    0,   160, 175, 10,  24,  213, 112, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   158, 169, 7,   42,  243, 176, 0,   52,  255, 4,   0,   0,   52,
    255, 0,   52,  255, 4,   0,   0,   52,  255, 0,   52,  255, 4,   0,   0,
    52,  255, 0,   52,  255, 4,   0,   0,   52,  255, 0,   74,  238, 18,  0,
    52,  245, 33,  0,   52,  255, 4,   0,   0,   0,   0,   0,   20,  251, 87,
    1,   120, 216, 0,   0,   127, 0,   0,   179, 121, 200, 105, 0,   0,   0,
    0,   179, 121, 200, 105, 0,   0,   0,   0,   179, 121, 200, 105, 0,   0,
    0,   0,   179, 121, 200, 105, 0,   0,   0,   3,   236, 97,  192, 0,   0,
    0,   0,   0,   185, 107, 136, 0,   0,   0,   0,   86,  189, 52,  255, 4,
    0,   0,   0,   227, 84,  0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   52,
    255, 4,   0,   0,   125, 186, 0,   52,  255, 150, 143, 0,   52,  255, 0,
    6,   246, 54,  0,   0,   105, 204, 0,   6,   246, 54,  0,   0,   105, 204,
    0,   6,   246, 54,  0,   0,   105, 204, 0,   6,   246, 54,  0,   0,   105,
    204, 0,   6,   246, 54,  0,   0,   105, 204, 0,   0,   123, 22,  0,   0,
    66,  81,  0,   5,   246, 57,  0,   52,  245, 208, 0,   52,  255, 4,   0,
    0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255, 0,   52,  255, 4,
    0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255, 0,   0,   176,
    146, 0,   194, 127, 0,   0,   52,  255, 240, 240, 233, 177, 38,  0,   50,
    255, 8,   69,  180, 160, 0,   0,   127, 0,   12,  248, 51,  130, 186, 0,
    0,   0,   12,  248, 51,  130, 186, 0,   0,   0,   12,  248, 51,  130, 186,
    0,   0,   0,   12,  248, 51,  130, 186, 0,   0,   0,   64,  222, 16,  250,
    19,  0,   0,   0,   25,  230, 26,  227, 2,   0,   0,   0,   159, 125, 52,
    255, 4,   0,   0,   31,  255, 21,  0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   52,  255, 4,   0,   0,   69,  240, 0,   52,  255, 37,  239, 16,  52,
    255, 0,   38,  255, 15,  0,   0,   64,  244, 0,   38,  255, 15,  0,   0,
    64,  244, 0,   38,  255, 15,  0,   0,   64,  244, 0,   38,  255, 15,  0,
    0,   64,  244, 0,   38,  255, 15,  0,   0,   64,  244, 0,   0,   130, 214,
    25,  73,  235, 64,  0,   38,  255, 20,  12,  210, 108, 246, 0,   52,  255,
    4,   0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255, 0,   52,
    255, 4,   0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255, 0,
    0,   31,  241, 124, 221, 8,   0,   0,   52,  255, 4,   0,   18,  164, 208,
    0,   52,  255, 30,  251, 36,  0,   0,   0,   127, 0,   86,  235, 1,   59,
    250, 16,  0,   0,   86,  235, 1,   59,  250, 16,  0,   0,   86,  235, 1,
    59,  250, 16,  0,   0,   86,  235, 1,   59,  250, 16,  0,   0,   144, 164,
    0,   213, 96,  0,   0,   0,   117, 182, 0,   230, 68,  0,   0,   1,   231,
    61,  52,  255, 240, 240, 0,   48,  255, 6,   0,   0,   0,   0,   0,   52,
    255, 240, 240, 240, 240, 22,  0,   52,  255, 240, 240, 240, 240, 22,  0,
    52,  255, 240, 240, 240, 240, 22,  0,   52,  255, 240, 240, 240, 187, 0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   239, 255, 240, 198, 0,   55,  255, 0,   52,  255, 4,   169,
    120, 52,  255, 0,   49,  255, 5,   0,   0,   54,  255, 0,   49,  255, 5,
    0,   0,   54,  255, 0,   49,  255, 5,   0,   0,   54,  255, 0,   49,  255,
    5,   0,   0,   54,  255, 0,   49,  255, 5,   0,   0,   54,  255, 0,   0,
    0,   124, 225, 235, 60,  0,   0,   49,  255, 6,   160, 101, 53,  255, 0,
    52,  255, 4,   0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,  255,
    0,   52,  255, 4,   0,   0,   52,  255, 0,   52,  255, 4,   0,   0,   52,
    255, 0,   0,   0,   124, 255, 75,  0,   0,   0,   52,  255, 4,   0,   0,
    59,  253, 0,   52,  255, 26,  245, 109, 2,   0,   0,   127, 0,   167, 167,
    0,   4,   240, 92,  0,   0,   167, 167, 0,   4,   240, 92,  0,   0,   167,
    167, 0,   4,   240, 92,  0,   0,   167, 167, 0,   4,   240, 92,  0,   0,
    223, 105, 0,   154, 176, 0,   0,   0,   210, 130, 0,   179, 162, 0,   0,
    49,  246, 6,   52,  255, 4,   0,   0,   31,  255, 23,  0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   52,  255, 4,   0,   0,   69,  240, 0,   52,  255,
    4,   52,  230, 58,  255, 0,   38,  255, 15,  0,   0,   64,  244, 0,   38,
    255, 15,  0,   0,   64,  244, 0,   38,  255, 15,  0,   0,   64,  244, 0,
    38,  255, 15,  0,   0,   64,  244, 0,   38,  255, 15,  0,   0,   64,  244,
    0,   0,   0,   80,  241, 224, 31,  0,   0,   40,  255, 92,  164, 0,   63,
    245, 0,   51,  255, 4,   0,   0,   52,  255, 0,   51,  255, 4,   0,   0,
    52,  255, 0,   51,  255, 4,   0,   0,   52,  255, 0,   51,  255, 4,   0,
    0,   52,  255, 0,   0,   0,   52,  255, 4,   0,   0,   0,   52,  255, 4,
    0,   17,  161, 208, 0,   52,  255, 4,   47,  187, 209, 46,  0,   127, 7,
    241, 250, 248, 248, 253, 173, 0,   7,   241, 250, 248, 248, 253, 173, 0,
    7,   241, 250, 248, 248, 253, 173, 0,   7,   241, 250, 248, 248, 253, 173,
    0,   48,  255, 249, 248, 250, 246, 10,  0,   49,  255, 249, 248, 250, 244,
    11,  0,   122, 251, 240, 243, 255, 4,   0,   0,   0,   228, 88,  0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   52,  255, 4,   0,   0,   126, 186, 0,
    52,  255, 4,   0,   192, 149, 255, 0,   6,   247, 54,  0,   0,   104, 204,
    0,   6,   247, 54,  0,   0,   104, 204, 0,   6,   247, 54,  0,   0,   104,
    204, 0,   6,   247, 54,  0,   0,   104, 204, 0,   6,   247, 54,  0,   0,
    104, 204, 0,   0,   87,  233, 51,  113, 224, 34,  0,   10,  253, 203, 12,
    0,   100, 203, 0,   40,  255, 9,   0,   0,   57,  248, 0,   40,  255, 9,
    0,   0,   57,  248, 0,   40,  255, 9,   0,   0,   57,  248, 0,   40,  255,
    8,   0,   0,   56,  248, 0,   0,   0,   52,  255, 4,   0,   0,   0,   52,
    255, 240, 240, 234, 178, 38,  0,   52,  255, 4,   0,   0,   116, 220, 0,
    127, 74,  249, 16,  0,   0,   87,  245, 0,   74,  249, 16,  0,   0,   87,
    245, 0,   74,  249, 16,  0,   0,   87,  245, 0,   74,  249, 16,  0,   0,
    87,  245, 0,   128, 211, 0,   0,   15,  248, 80,  0,   143, 187, 0,   0,
    5,   234, 94,  0,   195, 109, 0,   52,  255, 4,   0,   0,   0,   102, 228,
    62,  5,   34,  99,  0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   52,  255, 4,   0,   0,   0,   0,   8,   58,  255, 11,  8,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   52,  255, 4,   7,   82,  239,
    67,  0,   52,  255, 4,   0,   75,  246, 255, 0,   0,   163, 173, 9,   23,
    212, 114, 0,   0,   163, 173, 9,   23,  212, 114, 0,   0,   163, 173, 9,
    23,  212, 114, 0,   0,   163, 173, 9,   23,  212, 114, 0,   0,   163, 173,
    9,   23,  212, 114, 0,   0,   166, 49,  0,   0,   108, 107, 0,   13,  237,
    190, 15,  18,  206, 111, 0,   5,   232, 117, 10,  19,  158, 190, 0,   5,
    232, 117, 10,  19,  158, 190, 0,   5,   232, 117, 10,  19,  158, 190, 0,
    5,   232, 107, 3,   9,   151, 190, 0,   0,   0,   52,  255, 4,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 34,  8,   2,   118,
    240, 0,   127, 155, 179, 0,   0,   0,   11,  245, 0,   155, 179, 0,   0,
    0,   11,  245, 0,   155, 179, 0,   0,   0,   11,  245, 0,   155, 179, 0,
    0,   0,   11,  245, 0,   208, 126, 0,   0,   0,   176, 160, 0,   232, 94,
    0,   0,   0,   144, 188, 0,   251, 31,  0,   52,  255, 244, 244, 0,   0,
    0,   99,  214, 255, 240, 135, 0,   52,  255, 244, 244, 244, 244, 99,  0,
    52,  255, 244, 244, 244, 244, 99,  0,   52,  255, 244, 244, 244, 244, 99,
    0,   52,  255, 244, 244, 244, 244, 22,  0,   0,   244, 246, 255, 244, 244,
    0,   0,   0,   244, 246, 255, 244, 244, 0,   0,   0,   255, 255, 255, 255,
    255, 0,   0,   0,   244, 246, 255, 244, 244, 0,   0,   52,  255, 244, 240,
    191, 69,  0,   0,   52,  255, 4,   0,   1,   213, 255, 0,   0,   16,  171,
    241, 237, 143, 3,   0,   0,   16,  171, 241, 237, 143, 3,   0,   0,   16,
    171, 241, 237, 143, 3,   0,   0,   16,  171, 241, 237, 143, 3,   0,   0,
    16,  171, 241, 237, 143, 3,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    157, 99,  185, 245, 234, 140, 2,   0,   0,   51,  197, 249, 244, 177, 28,
    0,   0,   51,  197, 249, 244, 177, 28,  0,   0,   51,  197, 249, 244, 177,
    28,  0,   0,   51,  197, 244, 241, 177, 28,  0,   0,   0,   52,  255, 4,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 116, 240,
    246, 211, 74,  0,   127, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   107, 120, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   17,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   13,  69,  194, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   233, 241, 107, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   104, 226, 228,
    110, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   127, 0,   8,   195, 125, 0,   0,   0,   0,   0,   0,   0,   0,   98,
    212, 15,  0,   0,   0,   2,   197, 213, 8,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   241,
    45,  96,  248, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   2,   157, 139, 0,   0,   0,   0,   0,
    0,   0,   0,   154, 142, 0,   0,   0,   0,   10,  199, 192, 5,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   111, 190, 2,   0,   0,
    0,   0,   0,   0,   0,   50,  225, 29,  0,   0,   0,   0,   130, 241, 30,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   11,  187, 114,
    0,   0,   0,   0,   0,   0,   0,   1,   161, 149, 1,   0,   0,   0,   24,
    211, 193, 6,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   68,  217, 9,   0,   0,   0,   0,
    0,   0,   0,   44,  223, 23,  0,   0,   0,   0,   88,  244, 31,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   64,  91,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   127, 0,   0,   18,  213, 67,  0,   0,   0,   0,   0,   0,
    46,  220, 30,  0,   0,   0,   0,   115, 148, 123, 138, 0,   0,   0,   14,
    196, 237, 203, 189, 27,  0,   0,   0,   240, 3,   49,  244, 3,   0,   0,
    0,   105, 226, 228, 110, 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   1,   155, 105, 0,   0,
    0,   0,   0,   0,   121, 140, 0,   0,   0,   0,   0,   158, 74,  90,  145,
    0,   0,   0,   0,   240, 3,   49,  244, 3,   0,   0,   0,   0,   158, 122,
    0,   0,   0,   0,   0,   0,   12,  210, 56,  0,   0,   0,   0,   38,  200,
    77,  175, 0,   0,   0,   49,  244, 53,  244, 3,   0,   0,   0,   3,   138,
    202, 114, 121, 12,  0,   0,   86,  232, 214, 194, 37,  0,   0,   0,   0,
    7,   172, 97,  0,   0,   0,   0,   0,   0,   144, 133, 0,   0,   0,   0,
    6,   185, 54,  92,  153, 0,   0,   0,   25,  206, 234, 199, 174, 3,   0,
    0,   49,  244, 3,   49,  244, 3,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   123, 138, 0,   0,
    0,   0,   0,   0,   2,   195, 59,  0,   0,   0,   0,   11,  204, 78,  171,
    0,   0,   0,   0,   49,  244, 53,  244, 3,   0,   0,   0,   0,   0,   51,
    205, 26,  0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,   49,  244,
    53,  244, 3,   0,   0,   127, 0,   0,   0,   32,  105, 0,   0,   0,   0,
    0,   0,   91,  46,  0,   0,   0,   0,   2,   113, 7,   1,   111, 7,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    6,   117, 7,   0,   0,   0,   0,   0,   57,  72,  0,   0,   0,   0,   0,
    74,  38,  0,   93,  20,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    51,  110, 93,  212, 36,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   24,  0,   0,   0,   1,   109,
    6,   0,   0,   0,   0,   0,   35,  80,  0,   0,   0,   0,   0,   49,  64,
    0,   96,  15,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   103, 25,  0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   127, 0,   103, 213, 244, 239, 180, 36,
    0,   0,   103, 213, 244, 240, 180, 36,  0,   0,   103, 213, 244, 239, 180,
    36,  0,   0,   103, 213, 244, 239, 180, 36,  0,   0,   103, 213, 244, 239,
    180, 36,  0,   0,   103, 213, 244, 239, 180, 36,  0,   0,   132, 239, 226,
    131, 234, 236, 0,   0,   12,  151, 233, 244, 202, 40,  0,   0,   13,  156,
    236, 241, 165, 12,  0,   0,   13,  156, 236, 241, 165, 12,  0,   0,   13,
    156, 236, 241, 165, 12,  0,   0,   13,  156, 235, 241, 165, 12,  0,   0,
    0,   244, 246, 255, 4,   0,   0,   0,   0,   244, 246, 255, 4,   0,   0,
    0,   0,   244, 246, 255, 4,   0,   0,   0,   148, 244, 255, 4,   0,   0,
    0,   0,   25,  178, 243, 251, 214, 9,   0,   52,  255, 111, 233, 235, 80,
    0,   0,   0,   29,  182, 242, 237, 157, 11,  0,   0,   29,  182, 242, 237,
    157, 11,  0,   0,   29,  182, 242, 237, 157, 11,  0,   0,   29,  182, 242,
    237, 157, 11,  0,   0,   29,  182, 242, 237, 157, 11,  0,   0,   0,   0,
    49,  244, 3,   0,   0,   0,   29,  181, 242, 238, 160, 192, 0,   52,  255,
    4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,
    255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,
    40,  250, 19,  0,   0,   50,  245, 0,   52,  255, 121, 231, 241, 161, 10,
    0,   93,  212, 0,   0,   0,   221, 87,  0,   127, 0,   93,  52,  2,   9,
    162, 204, 0,   0,   93,  52,  2,   10,  165, 204, 0,   0,   93,  52,  2,
    9,   162, 204, 0,   0,   93,  52,  2,   9,   162, 204, 0,   0,   93,  52,
    2,   9,   162, 204, 0,   0,   93,  52,  2,   9,   159, 204, 0,   0,   48,
    4,   120, 255, 57,  104, 0,   0,   174, 188, 24,  0,   53,  43,  0,   0,
    173, 187, 16,  11,  178, 153, 0,   0,   173, 187, 16,  11,  178, 153, 0,
    0,   173, 186, 16,  11,  179, 152, 0,   0,   173, 177, 14,  11,  179, 152,
    0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   252, 4,
    0,   0,   0,   0,   192, 167, 11,  4,   181, 123, 0,   52,  255, 121, 2,
    133, 219, 0,   0,   0,   198, 158, 8,   19,  198, 149, 0,   0,   198, 158,
    8,   19,  198, 149, 0,   0,   198, 158, 8,   19,  198, 149, 0,   0,   198,
    158, 8,   19,  198, 149, 0,   0,   198, 158, 8,   19,  198, 149, 0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   198, 161, 9,   24,  239, 165, 0,
    52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,
    0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255,
    4,   0,   0,   200, 106, 0,   0,   142, 163, 0,   52,  255, 163, 9,   19,
    198, 145, 0,   11,  244, 36,  0,   48,  244, 10,  0,   127, 0,   0,   0,
    0,   0,   61,  252, 0,   0,   0,   0,   0,   0,   62,  252, 0,   0,   0,
    0,   0,   0,   61,  252, 0,   0,   0,   0,   0,   0,   61,  252, 0,   0,
    0,   0,   0,   0,   61,  252, 0,   0,   0,   0,   0,   0,   57,  252, 0,
    0,   0,   0,   55,  255, 6,   54,  0,   22,  255, 41,  0,   0,   0,   0,
    0,   21,  255, 50,  0,   0,   67,  236, 0,   21,  255, 50,  0,   0,   67,
    236, 0,   21,  255, 49,  0,   0,   70,  236, 0,   21,  255, 37,  0,   0,
    70,  236, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,
    255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    252, 4,   0,   0,   0,   25,  255, 35,  0,   0,   86,  217, 0,   52,  255,
    18,  0,   56,  254, 1,   0,   27,  255, 32,  0,   0,   81,  234, 0,   27,
    255, 32,  0,   0,   81,  234, 0,   27,  255, 32,  0,   0,   81,  234, 0,
    27,  255, 32,  0,   0,   81,  234, 0,   27,  255, 32,  0,   0,   81,  234,
    0,   0,   240, 240, 240, 240, 240, 240, 0,   27,  255, 31,  2,   168, 165,
    235, 0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,
    255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,
    52,  255, 4,   0,   0,   103, 199, 0,   3,   230, 67,  0,   52,  255, 33,
    0,   0,   81,  232, 0,   0,   162, 117, 0,   130, 165, 0,   0,   127, 0,
    78,  197, 237, 240, 243, 255, 0,   0,   71,  189, 224, 224, 230, 255, 0,
    0,   78,  197, 237, 240, 243, 255, 0,   0,   78,  197, 237, 240, 243, 255,
    0,   0,   78,  197, 237, 240, 243, 255, 0,   0,   78,  197, 237, 240, 243,
    255, 0,   0,   125, 231, 244, 255, 240, 241, 0,   50,  255, 7,   0,   0,
    0,   0,   0,   47,  255, 225, 224, 224, 227, 239, 0,   47,  255, 225, 224,
    224, 227, 239, 0,   47,  255, 240, 240, 240, 241, 247, 0,   47,  255, 240,
    240, 240, 241, 247, 0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,
    0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   252, 4,   0,   0,   0,   48,  255, 7,   0,   0,   56,  254, 0,
    52,  255, 4,   0,   52,  255, 4,   0,   48,  255, 6,   0,   0,   55,  254,
    0,   48,  255, 6,   0,   0,   55,  254, 0,   48,  255, 6,   0,   0,   55,
    254, 0,   48,  255, 6,   0,   0,   55,  254, 0,   48,  255, 6,   0,   0,
    55,  254, 0,   0,   0,   0,   0,   0,   0,   0,   0,   48,  255, 7,   152,
    111, 54,  255, 0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,
    0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255, 4,   0,   52,  255,
    4,   0,   52,  255, 4,   0,   0,   15,  246, 36,  70,  225, 1,   0,   52,
    255, 6,   0,   0,   55,  254, 0,   0,   68,  197, 0,   212, 76,  0,   0,
    127, 24,  251, 95,  6,   0,   62,  255, 0,   23,  250, 82,  2,   0,   65,
    255, 0,   24,  251, 86,  5,   0,   65,  255, 0,   24,  251, 86,  5,   0,
    65,  255, 0,   24,  251, 86,  5,   0,   65,  255, 0,   24,  251, 86,  5,
    0,   65,  255, 0,   32,  255, 57,  53,  255, 8,   0,   0,   22,  255, 41,
    0,   0,   0,   0,   0,   22,  255, 23,  0,   0,   0,   0,   0,   22,  255,
    23,  0,   0,   0,   0,   0,   22,  255, 21,  0,   0,   0,   0,   0,   22,
    255, 21,  0,   0,   0,   0,   0,   0,   0,   0,   52,  255, 4,   0,   0,
    0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   252, 4,   0,   0,   0,   25,  255, 34,  0,   0,   83,
    238, 0,   52,  255, 4,   0,   52,  255, 4,   0,   27,  255, 32,  0,   0,
    81,  234, 0,   27,  255, 32,  0,   0,   81,  234, 0,   27,  255, 32,  0,
    0,   81,  234, 0,   27,  255, 32,  0,   0,   81,  234, 0,   27,  255, 32,
    0,   0,   81,  234, 0,   0,   0,   0,   48,  240, 3,   0,   0,   27,  255,
    165, 129, 0,   81,  234, 0,   48,  255, 8,   0,   66,  255, 4,   0,   48,
    255, 8,   0,   66,  255, 4,   0,   48,  255, 8,   0,   66,  255, 4,   0,
    48,  255, 8,   0,   66,  255, 4,   0,   0,   0,   166, 128, 162, 131, 0,
    0,   52,  255, 32,  0,   0,   80,  233, 0,   0,   2,   226, 62,  238, 5,
    0,   0,   127, 33,  255, 38,  0,   4,   170, 255, 0,   33,  255, 69,  0,
    26,  189, 255, 0,   33,  255, 68,  0,   26,  189, 255, 0,   33,  255, 68,
    0,   26,  189, 255, 0,   33,  255, 68,  0,   26,  189, 255, 0,   33,  255,
    68,  0,   26,  189, 255, 0,   38,  255, 39,  95,  255, 76,  1,   0,   0,
    177, 185, 23,  0,   50,  42,  0,   0,   176, 165, 17,  0,   32,  76,  0,
    0,   176, 165, 17,  0,   32,  76,  0,   0,   176, 162, 17,  0,   32,  76,
    0,   0,   176, 162, 17,  0,   29,  79,  0,   0,   0,   0,   52,  255, 4,
    0,   0,   0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,   52,  255,
    4,   0,   0,   0,   0,   0,   252, 4,   0,   0,   0,   0,   194, 163, 8,
    21,  202, 156, 0,   52,  255, 4,   0,   52,  255, 4,   0,   0,   200, 158,
    7,   19,  198, 151, 0,   0,   200, 158, 7,   19,  198, 151, 0,   0,   200,
    158, 7,   19,  198, 151, 0,   0,   200, 158, 7,   19,  198, 151, 0,   0,
    200, 158, 7,   19,  198, 151, 0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   216, 210, 7,   19,  198, 151, 0,   15,  251, 62,  0,   147, 255, 4,
    0,   15,  251, 62,  0,   147, 255, 4,   0,   15,  251, 62,  0,   147, 255,
    4,   0,   15,  251, 62,  0,   147, 255, 4,   0,   0,   0,   70,  222, 242,
    36,  0,   0,   52,  255, 160, 8,   17,  195, 146, 0,   0,   0,   137, 215,
    157, 0,   0,   0,   127, 0,   114, 219, 196, 189, 127, 255, 0,   0,   114,
    231, 243, 204, 107, 255, 0,   0,   114, 231, 243, 204, 108, 255, 0,   0,
    114, 231, 243, 204, 108, 255, 0,   0,   114, 231, 243, 204, 108, 255, 0,
    0,   113, 230, 243, 205, 106, 255, 0,   0,   154, 245, 215, 95,  224, 242,
    0,   0,   14,  153, 237, 255, 207, 42,  0,   0,   14,  155, 234, 244, 218,
    111, 0,   0,   14,  155, 234, 244, 218, 111, 0,   0,   14,  155, 234, 244,
    218, 111, 0,   0,   14,  153, 233, 244, 220, 120, 0,   0,   118, 244, 246,
    255, 244, 244, 0,   0,   108, 224, 230, 255, 224, 224, 0,   0,   108, 224,
    230, 255, 224, 224, 0,   22,  244, 244, 255, 244, 244, 34,  0,   0,   27,
    180, 242, 238, 161, 13,  0,   52,  255, 4,   0,   52,  255, 4,   0,   0,
    31,  184, 243, 238, 160, 12,  0,   0,   31,  184, 243, 238, 160, 12,  0,
    0,   31,  184, 243, 238, 160, 12,  0,   0,   31,  184, 243, 238, 160, 12,
    0,   0,   31,  184, 243, 238, 160, 12,  0,   0,   0,   0,   0,   0,   0,
    0,   0,   95,  164, 178, 242, 238, 160, 12,  0,   0,   120, 234, 191, 137,
    255, 4,   0,   0,   120, 234, 191, 137, 255, 4,   0,   0,   120, 234, 191,
    137, 255, 4,   0,   0,   120, 234, 191, 137, 255, 4,   0,   0,   0,   2,
    227, 198, 0,   0,   0,   52,  255, 120, 232, 241, 160, 10,  0,   0,   0,
    43,  255, 72,  0,   0,   0,   127, 0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   187, 14,  0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   40,  4,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   199, 105, 0,   0,   0,   52,  255, 4,   0,   0,   0,   0,   0,
    0,   0,   18,  232, 4,   0,   0,   0,   127, 0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   10,  1,   167, 67,  0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   64,  244, 16,  0,   0,   0,   52,  255, 4,   0,   0,   0,
    0,   0,   0,   0,   132, 148, 0,   0,   0,   0,   127, 0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   64,  242, 213, 19,  0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   203, 237, 92,  0,   0,   0,   0,   52,  255, 4,   0,
    0,   0,   0,   0,   11,  243, 209, 25,  0,   0,   0,   0,   0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127,
    127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,
    127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127,
    0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127,
    127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127,
    127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127, 127,
    127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127, 127,
    127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0,   127, 127,
    127, 127, 127, 127, 127, 0,   127, 127, 127, 127, 127, 127, 127, 0};

void TwGenerateDefaultFonts() {
  if (g_DefaultSmallFont == 0) {
    g_DefaultSmallFont = TwGenerateFont(s_Font0, FONT0_BM_W, FONT0_BM_H);
    assert(g_DefaultSmallFont && g_DefaultSmallFont->m_NbCharRead == 224);
  }

  if (g_DefaultNormalFont == 0) {
    g_DefaultNormalFont = TwGenerateFont(s_Font1, FONT1_BM_W, FONT1_BM_H);
    assert(g_DefaultNormalFont && g_DefaultNormalFont->m_NbCharRead == 224);
  }

  if (g_DefaultNormalFontAA == 0) {
    g_DefaultNormalFontAA =
        TwGenerateFont(s_Font1AA, FONT1AA_BM_W, FONT1AA_BM_H);
    assert(g_DefaultNormalFontAA && g_DefaultNormalFontAA->m_NbCharRead == 224);
  }
  if (g_DefaultLargeFont == 0) {
    g_DefaultLargeFont = TwGenerateFont(s_Font2AA, FONT2AA_BM_W, FONT2AA_BM_H);
    assert(g_DefaultLargeFont && g_DefaultLargeFont->m_NbCharRead == 224);
  }
  if (g_DefaultFixed1Font == 0) {
    g_DefaultFixed1Font =
        TwGenerateFont(s_FontFixed1, FONTFIXED1_BM_W, FONTFIXED1_BM_H);
    assert(g_DefaultFixed1Font && g_DefaultFixed1Font->m_NbCharRead == 224);
  }
}

//  ---------------------------------------------------------------------------

void TwDeleteDefaultFonts() {
  delete g_DefaultSmallFont;
  g_DefaultSmallFont = NULL;
  delete g_DefaultNormalFont;
  g_DefaultNormalFont = NULL;
  delete g_DefaultNormalFontAA;
  g_DefaultNormalFontAA = NULL;
  delete g_DefaultLargeFont;
  g_DefaultLargeFont = NULL;
  delete g_DefaultFixed1Font;
  g_DefaultFixed1Font = NULL;
}

//  ---------------------------------------------------------------------------
