""" File containing class to handle connections to datacore """

import json

import requests

from ..atrium_response import AtriumResponse
from ..endpoints import get_endpoint_url
from .openapi import ApiClient, Configuration


class AuthException(Exception):
    """Exception to be raised when authentication fails"""

    pass


class DatacoreAPI:
    """A class to handle the connection to AtriumSports Datacore"""

    DEFAULT_LIMIT = 10
    MAX_LIMIT = 200

    def __init__(self, options):
        """initialise the class"""
        self._sport = options.get("sport", "basketball")
        self._credential_id = options.get("credential_id", "")
        self._credential_secret = options.get("credential_secret", "")
        self._org_group = options.get("org_group")
        self._organizations = options.get("organizations", [])
        self._environment = options.get("environment", "production")
        self._version = 1
        self._headers = options.get("headers", {})
        self._api_endpoint_url = get_endpoint_url(self._environment, "api", version=self._version)
        self._auth_token = options.get("token")
        self._openapi_api_client = None

    def _create_openapi_configuration(self):
        openapi_configuration = Configuration(
            host=self._api_endpoint_url,
        )
        openapi_configuration.access_token = self.auth_token
        return openapi_configuration

    def __enter__(self):
        """We're creating a new ApiClient class with connection pool to backend
        that is closed when we exit the context manager

        example of creation and usage:

        atrium = AtriumSports(
            {
                "sport": "basketball",
                "credential_id": "XXXXX",
                "credential_secret": "YYYY",
                "organizations": ["b1e34"],
            }
        )
        datacore = atrium.client("datacore")
        # prepare api client with access token and connection pool
        with datacore as api_client:
            # create api instance object for handling input and output of chosen endpoint
            api_instance = CompetitionsApi(api_client)
        """
        openapi_configuration = self._create_openapi_configuration()
        self._openapi_api_client = ApiClient(openapi_configuration)
        return self._openapi_api_client

    def __exit__(self, exc_type, exc_value, traceback):
        # closing connection pool
        self._openapi_api_client.close()
        self._openapi_api_client = None

    def _get_api_url(self, url_path):
        return f"{self._api_endpoint_url}/{self._sport}{url_path}"

    def post(self, url, **kwargs):
        """POST method"""
        if not kwargs.get("body"):
            return self._return_error("POST method requires a body parameter")
        return self.call("POST", url, **kwargs)

    def put(self, url, **kwargs):
        """PUT method"""
        if not kwargs.get("body"):
            return self._return_error("PUT method requires a body parameter")
        return self.call("PUT", url, **kwargs)

    def get(self, url, **kwargs):
        """GET method"""
        return self.call("GET", url, **kwargs)

    def delete(self, url, **kwargs):
        """DELETE method"""
        return self.call("DELETE", url, **kwargs)

    def _generate_token(self):
        """generate an auth token"""
        auth_endpoint_url = get_endpoint_url(self._environment, "auth")

        auth_data = {
            "credentialId": self._credential_id,
            "credentialSecret": self._credential_secret,
            "sport": self._sport,
            "organization": {},
        }
        if self._org_group:
            auth_data["organization"]["group"] = self._org_group
        else:
            auth_data["organization"]["id"] = self._organizations

        response = self._api_call_internal("POST", auth_endpoint_url, body=auth_data)
        if response.success():
            return response.data().get("token")
        raise AuthException(response.error_string())

    @property
    def auth_token(self):
        if not self._auth_token:
            self._auth_token = self._generate_token()
        return self._auth_token

    def call(self, method, url, **kwargs):
        url = self._get_api_url(url)

        limit = kwargs.get("limit", self.DEFAULT_LIMIT)
        kwargs["limit"] = self._get_limit(limit)
        kwargs["headers"] = self._get_headers(kwargs.get("headers"))

        done = False
        response = AtriumResponse()
        while not done:
            resp = self._api_call_internal(method, url, **kwargs)
            response.merge(resp)
            if not resp.success():
                done = True
            else:
                if resp.links("next"):
                    url = resp.links("next")
                    call_limit = self._get_limit(limit - response.data_count())
                    if call_limit <= 0:
                        done = True
                else:
                    done = True
        return response

    def _get_limit(self, limit):
        """make sure limit doesn't exceed MAX_LIMIT"""
        return min(limit, self.MAX_LIMIT)

    def _get_headers(self, headers=None):
        result_headers = self._headers.copy()
        result_headers.update(headers or {})
        result_headers["Authorization"] = "Bearer {}".format(self.auth_token)
        result_headers["Content-Type"] = "application/json"
        return result_headers

    def _api_call_internal(self, method, url, **kwargs):
        """make the api call"""
        try:
            response = self._make_request(method, url, **kwargs)
            return AtriumResponse.create_from_str(response.status_code, response.text)
        except requests.exceptions.RequestException as err:
            return self._return_error(str(err))

    @staticmethod
    def _make_request(method, url, **kwargs):
        """Lets seperate the actual request code"""
        headers = kwargs.pop("headers", {})
        body = json.dumps(kwargs.pop("body", {}))

        response = None
        if method == "GET":
            response = requests.get(url, headers=headers, params=kwargs)
        elif method == "POST":
            response = requests.post(url, headers=headers, data=body)
        elif method == "PUT":
            response = requests.put(url, headers=headers, data=body)
        elif method == "DELETE":
            response = requests.delete(url, headers=headers, data=body)
        return response

    @staticmethod
    def _return_error(error):
        """Return an error response"""
        response = AtriumResponse()
        response.set_error(error)
        return response
