# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['loggate', 'loggate.http', 'loggate.loki']

package_data = \
{'': ['*']}

extras_require = \
{'asyncio': ['aiohttp']}

setup_kwargs = {
    'name': 'loggate',
    'version': '1.5.0',
    'description': 'The complex logging system.',
    'long_description': '# Loggate\n[![PyPI](https://img.shields.io/pypi/v/loggate?color=green&style=plastic)](https://pypi.org/project/loggate/)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/loggate?style=plastic)\n![License](https://img.shields.io/github/license/calcite/loggate?style=plastic)\n\nThe complex logging system with support of log metadata and delivery to [Grafana Loki](https://grafana.com/oss/loki/). \nThis library supports threading & asyncio modules.  \n\n## Simple stdout/stderr colorized output\nOne example is more than a thousand words.\n\n```python\nfrom loggate import setup_logging, get_logger\n\nsetup_logging(level=\'DEBUG\')\nlogger = get_logger(\'component\', meta={\'version\': \'1.0.0\'})\n\nlogger.debug(\'Loading resources for the component\')\nlogger.info(\'Initialize of the component\')\nlogger.warning(\'The component is not ready\')\nlogger.error(\'The component failed.\',\n             meta={\'inputs\': {\'A\': 1, \'B\': 2}})\nlogger.critical(\'The component unexpected failed.\',\n                meta={\'attrs\': {\'A\': 1, \'B\': 2}})\n```\n*Console output:*\n\n![Console output](https://github.com/calcite/loggate/raw/master/img/console.png)\n\n\n### Exceptions\n\n```python\nfrom loggate import setup_logging, get_logger\n\nsetup_logging()\nlogger = get_logger(\'component\', meta={\'version\': \'1.0.0\'})\n\ntry:\n    raise Exception(\'Some error\')\nexcept Exception as ex:\n    logger.error(\'The component failed.\', exc_info=True)\n```\n*Console output:*\n\n![Console output](https://github.com/calcite/loggate/raw/master/img/exception.png)\n\n\n## Advanced configuration\nThe Loggate supports a declarative configuration alike as [logging.config](https://docs.python.org/3/library/logging.config.html).\nBut this support profiles as well. It\'s mean we can declare many logging profiles and switch between them. Default profile is called `default`.\n\nWe can use yaml file as configuration file (see next):\n```yaml\nprofiles:\n  default:                             \n    # Default profile\n    filters:\n      warning:                          \n        # This is a filter for stdout logger, that enable only logs with level lower than WARNING. \n        # For logs with WARNING and higher we use stderr logger. \n        class: loggate.LowerLogLevelFilter\n        level: WARNING\n\n    formatters:\n      colored:\n        # This is stdout/sterr formatter. \n        class: loggate.LogColorFormatter\n      loki:\n        # This is formatter of loki messages.\n        class: loggate.loki.LokiLogFormatter\n\n    handlers:\n      stdout:\n        # This is a stdout handler\n        class: logging.StreamHandler\n        stream: ext://sys.stdout\n        formatter: colored\n        filters:\n          - warning\n      stderr:\n        # This is a stderr handler\n        class: logging.StreamHandler\n        stream: ext://sys.stderr\n        formatter: colored\n        level: WARNING        \n      loki:\n        # This is a loki handler\n        class: loggate.loki.LokiThreadHandler  # for asyncio use loggate.loki.LokiHandler       \n        formatter: loki\n        urls:\n          - "http://loki1:3100/loki/api/v1/push"\n          - "http://loki2:3100/loki/api/v1/push"\n          - "http://loki3:3100/loki/api/v1/push"\n        meta:\n          # loggate handlers accept metadata as well. Standard logging handlers do not!\n          stage: dev\n          ip: 192.168.1.10                \n                  \n\n    loggers:\n        root:          \n          level: INFO\n          handlers:\n            - stdout\n            - stderr        \n            - loki\n        \'urllib3.connectionpool\': \n          level: WARNING\n\n```\n\n```python\nimport yaml\nfrom loggate import setup_logging, get_logger\n\n\ndef get_yaml(filename):\n    with open(filename, \'r+\') as fd:\n        return yaml.safe_load(fd)\n\n\nschema = get_yaml(\'test.yaml\')\nsetup_logging(profiles=schema.get(\'profiles\'))\n\nlogger = get_logger(\'component\')\n\nlogger.debug(\'Loading resources for the component\')\nlogger.info(\'Initialize of the component\')\nlogger.warning(\'The component is not ready\')\n```\nThe console output is the same as above, but now we send logs to Loki as well.\n\n*Loki output:*\n\n![loki output](https://github.com/calcite/loggate/raw/master/img/loki1.png)\n\n\n\n# Description\n## Methods\n- `get_logger`\n  - `name` - Return logger for this name. Empty name returns root logger.\n  - `meta` - Metadata (dict), which are sent only by this logger.\n\n- `getLogger` - only alias for `get_logger`\n- `setup_logging` - init setup of application logging.\n  - `profiles` - Profiles (dict) of logging profiles. When we do not set this parameter, application use predefined profile with log `INFO` level (this level can be set by parameter `level`). \n  - `default_profile` - name of the default profile (default: `default`)\n  - `level` - This is special parameter for situation when  application use predefined profile (default `INFO`).  \n\n## Filters\n### Class `loggate.LowerLogLevelFilter`\nThis filters out all logs which are higher than `level`.\n- `level` - log level\n\n## Formatters\n### Class `loggate.LogColorFormatter`\nColorized formatter for stdout/stderr.\n- `fmt` - message format (default: `%(LEVEL_COLOR)s%(asctime)s\\t [%(levelname)s] %(name)s:%(COLOR_RESET)s %(message)s`)\n- `datefmt` - datetime format (default: `%Y-%m-%d %H:%M:%S`)\n- `style` - style of templating (default: `%`). \n- `validate` - validate the input format (default: True)\n- `INDENTATION_TRACEBACK` - default: `\\t\\t\\t`\n- `INDENTATION_METADATA` - default: `\\t\\t\\t\\t`\n- `COLOR_DEBUG`, ..., `COLOR_CRITICAL` - set color of this levels (e.g. `\\x1b[1;31m`, see [more colors](https://dev.to/ifenna__/adding-colors-to-bash-scripts-48g4)).\n- `COLOR_METADATA` - set color metadata\n- `COLOR_TRACEBACK` - set color of tracebacks\n- `COLOR_...` - set custom color\n\n### Class `loggate.loki.LokiLogFormatter`\nThis is special loki formatter, this converts log records to jsons.\n\n\n## Handlers\n### Class `loggate.loki.LokiHandler`\nThis handler send log records to Loki server. This is blocking implementation of handler.\nIt means, when we call log method (`debug`, ... `critical`) the message is sent in the same thread. We should use\nthis only for tiny scripts where other ways have a big overhead.\n- `level` - This handler sends only log records with log level equal or higher than this (default: all = `logging.NOTSET`).\n- `urls` - List of loki entrypoints.\n- `strategy` - Deploy strategy (default: `random`).\n  - `random` - At the beginning the handler choose random Loki server and others are fallbacks.\n  - `fallbacks` - The handler uses the first Loki server and others are fallbacks.\n  - `all` - The handler send the log record to all loki servers.\n- `auth` - The Loki authentication, the list with two items (`username`, `password`).\n- `timeout` - Timeout for one delivery try (default: 5s).\n- `ssl_verify` - Enable ssl verify (default: True).\n- `loki_tags` - the list of metadata keys, which are sent to Loki server as label (defailt: [`logger`, `level`]).\n- `meta` - Metadata (dict), which are sent only by this handler.  \n\n### Class `loggate.loki.LokiAsyncioHandler`\nThis is non-bloking extending of LokiHandler. We register an extra asyncio task for sending messages to the Loki server.\nParameters are the same as `loggate.loki.LokiHandler`. This handler uses `urllib.requests` module in default ([aiohttp](https://pypi.org/project/aiohttp/) as optional). \nUnfortunately `urllib.requests` module does not support asyncio, it means the sending itself is blocking.\nThe `loggate.loki.Loki AsyncioHandler` can use the optional dependency [aiohttp](https://pypi.org/project/aiohttp/) for non-bloking sending.\n\n### Class `loggate.loki.LokiThreadHandler`\nThis is non-bloking extending of LokiHandler. We register and start an extra thread for sending messages to the Loki server.\nParameters are the same as `loggate.loki.LokiHandler`.\n\n## Profiles\nThe structure of profiles (parameter `profiles` of `setup_logging`).\n\n```yaml\n<profile_name>:\n  \n  filters:\n    <filter_name>:\n      class: <filter_class>\n      <filter_attribute_name>: <filter_attribute_value>\n  \n  formatters:\n    <formatter_name>:\n      class: <formatter_class>\n      <formatter_attribute_name>: <formatter_attribute_value>\n  \n  handlers:\n    <handler_name>:\n      class: <handler_class>\n      <handler_attribute_name>: <handler_attribute_value>\n\n  loggers:\n    <logger_name>|root:   # definition of root logger\n      level: <log_level>\n      handlers: \n        - <name_of_handler>|<definition_of_handler>\n      disabled: True|False    # default: False\n      propagate: True|False   # default: True\n      meta: <logger_metadata>  \n```\n',
    'author': 'Martin Korbel',
    'author_email': 'mkorbel@alps.cz',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/calcite/loggate',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
