# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['jdfile', 'jdfile._config', 'jdfile.models', 'jdfile.utils']

package_data = \
{'': ['*']}

install_requires = \
['loguru>=0.6.0,<0.7.0',
 'nltk>=3.8.1,<4.0.0',
 'questionary>=1.10.0,<2.0.0',
 'rich>=13.3.1,<14.0.0',
 'shellingham>=1.4.0,<2.0.0',
 'tomli>=2.0.1,<3.0.0',
 'typer>=0.7.0,<0.8.0']

entry_points = \
{'console_scripts': ['jdfile = jdfile.cli:app']}

setup_kwargs = {
    'name': 'jdfile',
    'version': '1.1.0',
    'description': 'File Manager for the Johnny Decimal System',
    'long_description': '[![PyPI version](https://badge.fury.io/py/jdfile.svg)](https://badge.fury.io/py/jdfile) ![PyPI - Python Version](https://img.shields.io/pypi/pyversions/jdfile) [![Python Code Checker](https://github.com/natelandau/jdfile/actions/workflows/automated-tests.yml/badge.svg)](https://github.com/natelandau/jdfile/actions/workflows/automated-tests.yml) [![codecov](https://codecov.io/gh/natelandau/jdfile/branch/main/graph/badge.svg?token=Y11Z883PMI)](https://codecov.io/gh/natelandau/jdfile)\n\n# jdfile\n\n`jdfile` cleans and normalizes filenames. In addition, if you have directories which follow the [Johnny Decimal](https://johnnydecimal.com), jdfile can move your files into the appropriate directory.\n\n`jdfile` cleans filenames based on your preferences.\n\n-   Remove special characters\n-   Trim multiple separators (`word----word` becomes `word-word`)\n-   Normalize to `lower case`, `upper case`, `sentence case`, or `title case`\n-   Normalize all files to a common word separator (`_`, `-`, ` `)\n-   Enforce lowercase file extensions\n-   Remove common English stopwords\n-   Split `camelCase` words into separate words (`camel Case`)\n-   Parse the filename for a date in many different formats\n-   Remove or reformat the date and add it to the the beginning of the filename\n-   Avoid overwriting files by adding a unique integer when renaming/moving\n-   Clean entire directory trees\n-   Optionally, show previews of changes to be made before commiting\n-   Ignore files listed in a config file\n-   Specify casing for words which should never be changed (ie. `iMac` will never be re-cased)\n\n`jdfile` can organize your files into folders.\n\n-   Move files into directory trees following the [Johnny Decimal](https://johnnydecimal.com) system\n-   Parse files and folder names looking for matching terms\n-   Uses [nltk](https://www.nltk.org) to lookup synonyms to improve matching\n-   Add `.jdfile` files to directories containing a list of words that will match files\n\n### Why build this?\n\nIt\'s nearly impossible to file away documents with normalized names when everyone has a different convention for naming files. On any given day, tons of files are attached to emails or sent via Slack by people who have their won way of naming files. For example:\n\n-   `department 2023 financials and budget 08232002.xlsx`\n-   `some contract Jan7 reviewed NOT FINAL (NL comments) v13.docx`\n-   `John&Jane-meeting-notes.txt`\n-   `Project_mockups(WIP)___sep92022.pdf`\n-   `FIRSTNAMElastname Resume (#1) [companyname].PDF`\n-   `code_to_review.js`\n\nIf you are a person who archives documents there are a number of problems with these files.\n\n-   No self-evident way to organize them into folders\n-   No common patterns to search for\n-   Dates all over the place or nonexistent\n-   No consistent casing\n-   No consistent word separators\n-   Special characters within text\n-   I could go on and on...\n\nAdditionally, even if the filenames were normalized, filing documents manually is a pain.\n\n`jdfile` is created to solve for these problems by providing an easy CLI to normalize the filename and organize it into an appropriate directory on your computer.\n\n## Install\n\njdfile requires Python v3.10 or above\n\n```bash\npip install pip install obsidian-metadata\n```\n\n## Usage\n\nRun `jdfile --help` for usage\n\n### Configuration\n\nTo organize files into folders, a valid [toml](https://toml.io/en/) configuration file is required at `~/.jdfile/jdfile.toml`\n\n```toml\n# The name of the project is used as a command line option.\n# (e.g. --organize=project_name)\n[project_name]\n    # (Required) Path to the folder containing the Johnny Decimal project\n    path = "~/johnnydecimal"\n\n    # An optional date format. If specified, the date will be appended to the filename\n    # See https://docs.python.org/3/library/datetime.html#strftime-and-strptime-format-codes for details on how to specify a date.\n    date_format = "None"\n\n    # Ignores dotfiles (files that start with a period) when cleaning a directory.  true or false\n    ignore_dotfiles = true\n\n    # Files in this list will be skipped.\n    ignored_files = [\'file1.txt\', \'file2.txt\']\n\n    # Force the casing of certain words. Great for acronyms or proper nouns.\n    match_case = ["CEO", "CEOs", "iMac", "iPhone"]\n\n    # Overwrite existing files. true or false. If false, unique integers will be appended to the filename.\n    overwrite_existing = false\n\n    # Separator to use between words. Options: "ignore", "underscore", "space", "dash", "none"\n    separator = "ignore"\n\n    # Split CamelCase words into separate words. true or false\n    split_words = false\n\n    # Optional list of project specific stopwords to be stripped from filenames\n    stopwords = ["stopword1", "stopword2"]\n\n    # Strip stopwords from filenames. true or false\n    strip_stopwords = true\n\n    # Transform case of filenames.\n    # Options: "lower", "upper", "title", "CamelCase", "sentence", "ignore",\n    transform_case = "ignore"\n\n    # Use the nltk wordnet corpus to find synonyms for words in filenames. true or false\n    # Note, this will download a large corpus (~400mb) the first time it is run.\n    use_synonyms = false\n```\n\n### Example usage\n\n```bash\n# Normalize all files in a directory to lowercase, with underscore separators\n$ jdfile --case=lower --separator=underscore /path/to/directory\n\n# Clean all files in a directory and confirm all changes before committing them\n$ jdfile --clean /path/to/directory\n\n# Strip common English stopwords from all files in a directory\n$ jdfile --stopwords /path/to/directory\n\n# Transform a date and add it to the filename\n$ jdfile --date-format="%Y-%m-%d" ./somefile_march 3rd, 2022.txt\n\n# Print a tree representation of a Johnny Decimal project\n$ jdfile --project=[project_name] --tree\n\n# Use the settings of a project in the config file to clean filenames without\n# organizing them into folders\n$ jdfile --project=[project_name] --no-organize path/to/some_file.jpg\n\n# Organize files into a Johnny Decimal project with specified terms with title casing\n$ jdfile ---project=[project_name] --term=term1 --term=term2 path/to/some_file.jpg\n```\n\n### Tips\n\nAdding custom functions to your `.bashrc` or `.zshrc` can save time and ensure your filename preferences are always used.\n\n```bash\n# ~/.bashrc\nif command -v jdfile &>/dev/null; then\n\n    clean() {\n        # DESC:\t Clean filenames using the jdfile package\n        if [[ $1 == "--help" || $1 == "-h" ]]; then\n            jdfile --help\n        else\n            jdfile --sep=space --case=title --confirm "$@"\n        fi\n    }\n\n    wfile() {\n        # DESC:\t File work documents\n        if [[ $1 == "--help" || $1 == "-h" ]]; then\n            jdfile --help\n        else\n            jdfile --project=work "$@"\n        fi\n    }\nfi\n```\n\n## Caveats\n\n`jdfile` is built for my own personal use. YMMV depending on your system and requirements. I make no warranties for any data loss that may result from use. I strongly recommend running in `--dry-run` mode prior to updating files.\n\n## Contributing\n\n### Setup: Once per project\n\nThere are two ways to contribute to this project.\n\n#### 1. Local development\n\n1. Install Python 3.10 and [Poetry](https://python-poetry.org)\n2. Clone this repository. `git clone https://github.com/natelandau/jdfile.git`\n3. Install the Poetry environment with `poetry install`.\n4. Activate your Poetry environment with `poetry shell`.\n5. Install the pre-commit hooks with `pre-commit install --install-hooks`.\n\n#### 2. Containerized development\n\n1. Clone this repository. `git clone https://github.com/natelandau/jdfile.git`\n2. Open the repository in Visual Studio Code\n3. Start the [Dev Container](https://code.visualstudio.com/docs/remote/containers). Run <kbd>Ctrl/⌘</kbd> + <kbd>⇧</kbd> + <kbd>P</kbd> → _Remote-Containers: Reopen in Container_.\n4. Run `poetry env info -p` to find the PATH to the Python interpreter if needed by VSCode.\n\n### Developing\n\n-   This project follows the [Conventional Commits](https://www.conventionalcommits.org/) standard to automate [Semantic Versioning](https://semver.org/) and [Keep A Changelog](https://keepachangelog.com/) with [Commitizen](https://github.com/commitizen-tools/commitizen).\n    -   When you\'re ready to commit changes run `cz c`\n-   Run `poe` from within the development environment to print a list of [Poe the Poet](https://github.com/nat-n/poethepoet) tasks available to run on this project. Common commands:\n    -   `poe lint` runs all linters\n    -   `poe test` runs all tests with Pytest\n-   Run `poetry add {package}` from within the development environment to install a run time dependency and add it to `pyproject.toml` and `poetry.lock`.\n-   Run `poetry remove {package}` from within the development environment to uninstall a run time dependency and remove it from `pyproject.toml` and `poetry.lock`.\n-   Run `poetry update` from within the development environment to upgrade all dependencies to the latest versions allowed by `pyproject.toml`.\n',
    'author': 'Nate Landau',
    'author_email': 'github@natenate.org',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/natelandau/jdfile',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
