# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyknp_eventgraph']

package_data = \
{'': ['*']}

install_requires = \
['graphviz>=0.16,<0.17', 'pyknp>=0.4.6,<0.5.0']

entry_points = \
{'console_scripts': ['evg = pyknp_eventgraph.cli:evg',
                     'evgviz = pyknp_eventgraph.cli:evgviz']}

setup_kwargs = {
    'name': 'pyknp-eventgraph',
    'version': '6.1.4',
    'description': 'A development platform for high-level NLP applications in Japanese',
    'long_description': "# pyknp-eventgraph\n\n**EventGraph** is a development platform for high-level NLP applications in Japanese.\nThe core concept of EventGraph is event, a language information unit that is closely related to predicate-argument structure but more application-oriented.\nEvents are linked to each other based on their syntactic and semantic relations.\n\n## Requirements\n\n- Python 3.6 or later\n- pyknp\n- graphviz\n\n## Installation\n\nTo install pyknp-eventgraph, use `pip`.\n\n```\n$ pip install pyknp-eventgraph\n```\n\nor\n\n```\n$ python setup.py install\n```\n\n## Quick Tour\n\n### Step 1: Create an EventGraph\n\nAn EventGraph is built on language analysis given in a KNP format.\n\n```python\n# Add imports.\nfrom pyknp import KNP\nfrom pyknp_eventgraph import EventGraph\n\n# Parse a document.\ndocument = ['\xe5\xbd\xbc\xe5\xa5\xb3\xe3\x81\xaf\xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81\xe8\x8b\xb1\xe8\xaa\x9e\xe3\x81\x8c\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84\xe3\x81\xab\xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84\xe3\x80\x82', '\xe7\xa7\x81\xe3\x81\xaf\xe3\x81\x9d\xe3\x81\x86\xe7\xa2\xba\xe4\xbf\xa1\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\x9f\xe3\x80\x82']\nknp = KNP()\nanalysis = [knp.parse(sentence) for sentence in document]\n\n# Create an EventGraph.\nevg = EventGraph.build(analysis)\nprint(evg)  # <EventGraph, #sentences: 2, #events: 3, #relations: 1>\n```\n\n### Step 2: Extract Information\n\nUsers can obtain various information about language analysis via a simple interface.\n\n#### Step 2.1: Sentence\n\n```python\n# Extract sentences.\nsentences = evg.sentences\nprint(sentences)\n# [\n#   <Sentence, sid: 1, ssid: 0, surf: \xe5\xbd\xbc\xe5\xa5\xb3\xe3\x81\xaf\xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81\xe8\x8b\xb1\xe8\xaa\x9e\xe3\x81\x8c\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84\xe3\x81\xab\xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84\xe3\x80\x82>,\n#   <Sentence, sid: 2, ssid: 1, surf: \xe7\xa7\x81\xe3\x81\xaf\xe3\x81\x9d\xe3\x81\x86\xe7\xa2\xba\xe4\xbf\xa1\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\x9f\xe3\x80\x82>\n# ]\n\n# Convert a sentence into various forms.\nsentence = evg.sentences[0]\nprint(sentence.surf)   # \xe5\xbd\xbc\xe5\xa5\xb3\xe3\x81\xaf\xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81\xe8\x8b\xb1\xe8\xaa\x9e\xe3\x81\x8c\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84\xe3\x81\xab\xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84\xe3\x80\x82\nprint(sentence.mrphs)  # \xe5\xbd\xbc\xe5\xa5\xb3 \xe3\x81\xaf \xe6\xb5\xb7\xe5\xa4\x96 \xe5\x8b\xa4\xe5\x8b\x99 \xe3\x81\x8c \xe9\x95\xb7\xe3\x81\x84 \xe3\x81\xae\xe3\x81\xa7 \xe3\x80\x81 \xe8\x8b\xb1\xe8\xaa\x9e \xe3\x81\x8c \xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84 \xe3\x81\xab \xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84 \xe3\x80\x82\nprint(sentence.reps)   # \xe5\xbd\xbc\xe5\xa5\xb3/\xe3\x81\x8b\xe3\x81\xae\xe3\x81\x98\xe3\x82\x87 \xe3\x81\xaf/\xe3\x81\xaf \xe6\xb5\xb7\xe5\xa4\x96/\xe3\x81\x8b\xe3\x81\x84\xe3\x81\x8c\xe3\x81\x84 \xe5\x8b\xa4\xe5\x8b\x99/\xe3\x81\x8d\xe3\x82\x93\xe3\x82\x80 \xe3\x81\x8c/\xe3\x81\x8c \xe9\x95\xb7\xe3\x81\x84/\xe3\x81\xaa\xe3\x81\x8c\xe3\x81\x84 \xe3\x81\xae\xe3\x81\xa7/\xe3\x81\xae\xe3\x81\xa7 \xe3\x80\x81/\xe3\x80\x81 \xe8\x8b\xb1\xe8\xaa\x9e/\xe3\x81\x88\xe3\x81\x84\xe3\x81\x94 \xe3\x81\x8c/\xe3\x81\x8c \xe4\xb8\x8a\xe6\x89\x8b\xe3\x81\x84/\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84 \xe3\x81\xab/\xe3\x81\xab \xe9\x81\x95\xe3\x81\x84\xe7\x84\xa1\xe3\x81\x84/\xe3\x81\xa1\xe3\x81\x8c\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84 \xe3\x80\x82/\xe3\x80\x82\n```\n\n#### Step 2.2: Event\n\n```python\n# Extract events.\nevents = evg.events\nprint(events)\n# [\n#   <Event, evid: 0, surf: \xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81>,\n#   <Event, evid: 1, surf: \xe5\xbd\xbc\xe5\xa5\xb3\xe3\x81\xaf\xe8\x8b\xb1\xe8\xaa\x9e\xe3\x81\x8c\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84\xe3\x81\xab\xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84\xe3\x80\x82>,\n#   <Event, evid: 2, surf: \xe7\xa7\x81\xe3\x81\xaf\xe3\x81\x9d\xe3\x81\x86\xe7\xa2\xba\xe4\xbf\xa1\xe3\x81\x97\xe3\x81\xa6\xe3\x81\x84\xe3\x81\x9f\xe3\x80\x82>\n# ]\n\n# Convert an event into various forms.\nevent = evg.events[0]\nprint(event.surf)              # \xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81\nprint(event.mrphs)             # \xe6\xb5\xb7\xe5\xa4\x96 \xe5\x8b\xa4\xe5\x8b\x99 \xe3\x81\x8c \xe9\x95\xb7\xe3\x81\x84 \xe3\x81\xae\xe3\x81\xa7 \xe3\x80\x81\nprint(event.normalized_mrphs)  # \xe6\xb5\xb7\xe5\xa4\x96 \xe5\x8b\xa4\xe5\x8b\x99 \xe3\x81\x8c \xe9\x95\xb7\xe3\x81\x84\nprint(event.reps)              # \xe6\xb5\xb7\xe5\xa4\x96/\xe3\x81\x8b\xe3\x81\x84\xe3\x81\x8c\xe3\x81\x84 \xe5\x8b\xa4\xe5\x8b\x99/\xe3\x81\x8d\xe3\x82\x93\xe3\x82\x80 \xe3\x81\x8c/\xe3\x81\x8c \xe9\x95\xb7\xe3\x81\x84/\xe3\x81\xaa\xe3\x81\x8c\xe3\x81\x84 \xe3\x81\xae\xe3\x81\xa7/\xe3\x81\xae\xe3\x81\xa7 \xe3\x80\x81/\xe3\x80\x81\nprint(event.normalized_reps)   # \xe6\xb5\xb7\xe5\xa4\x96/\xe3\x81\x8b\xe3\x81\x84\xe3\x81\x8c\xe3\x81\x84 \xe5\x8b\xa4\xe5\x8b\x99/\xe3\x81\x8d\xe3\x82\x93\xe3\x82\x80 \xe3\x81\x8c/\xe3\x81\x8c \xe9\x95\xb7\xe3\x81\x84/\xe3\x81\xaa\xe3\x81\x8c\xe3\x81\x84\nprint(event.content_rep_list)  # ['\xe6\xb5\xb7\xe5\xa4\x96/\xe3\x81\x8b\xe3\x81\x84\xe3\x81\x8c\xe3\x81\x84', '\xe5\x8b\xa4\xe5\x8b\x99/\xe3\x81\x8d\xe3\x82\x93\xe3\x82\x80', '\xe9\x95\xb7\xe3\x81\x84/\xe3\x81\xaa\xe3\x81\x8c\xe3\x81\x84']\n\n# Extract an event's PAS.\npas = event.pas\nprint(pas)            # <PAS, predicate: \xe9\x95\xb7\xe3\x81\x84/\xe3\x81\xaa\xe3\x81\x8c\xe3\x81\x84, arguments: {\xe3\x82\xac: \xe5\x8b\xa4\xe5\x8b\x99/\xe3\x81\x8d\xe3\x82\x93\xe3\x82\x80}>\nprint(pas.predicate)  # <Predicate, type: \xe5\xbd\xa2, surf: \xe9\x95\xb7\xe3\x81\x84>\nprint(pas.arguments)  # defaultdict(<class 'list'>, {'\xe3\x82\xac': [<Argument, case: \xe3\x82\xac, surf: \xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c>]})\n\n# Extract an event's features.\nfeatures = event.features\nprint(features)  # <Features, modality: None, tense: \xe9\x9d\x9e\xe9\x81\x8e\xe5\x8e\xbb, negation: False, state: \xe7\x8a\xb6\xe6\x85\x8b\xe8\xbf\xb0\xe8\xaa\x9e, complement: False>\n```\n\n#### Step 2.3: Event-to-event Relation\n\n```python\n# Extract event-to-event relations.\nrelations = evg.relations\nprint(relations)  # [<Relation, label: \xe5\x8e\x9f\xe5\x9b\xa0\xe3\x83\xbb\xe7\x90\x86\xe7\x94\xb1, modifier_evid: 0, head_evid: 1>]\n\n# Take a closer look at an event-to-event relation\nrelation = relations[0]\nprint(relation.label)     # \xe5\x8e\x9f\xe5\x9b\xa0\xe3\x83\xbb\xe7\x90\x86\xe7\x94\xb1\nprint(relation.surf)      # \xe3\x81\xae\xe3\x81\xa7\nprint(relation.modifier)  # <Event, evid: 0, surf: \xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81>\nprint(relation.head)      # <Event, evid: 1, surf: \xe5\xbd\xbc\xe5\xa5\xb3\xe3\x81\xaf\xe8\x8b\xb1\xe8\xaa\x9e\xe3\x81\x8c\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84\xe3\x81\xab\xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84\xe3\x80\x82>\n```\n\n### Step 3: Seve/Load an EventGraph\n\nUsers can save and load an EventGraph by serializing it as a JSON object.\n\n```python\n# Save an EventGraph as a JSON file.\nevg.save('evg.json')\n\n# Load an EventGraph from a JSON file.\nwith open('evg.json') as f:\n    evg = EventGraph.load(f)\n```\n\n### Step 4: Visualize an EventGraph\n\nUsers can visualize an EventGraph using [graphviz](https://graphviz.org/).\n\n```python\nfrom pyknp_eventgraph import make_image\nmake_image(evg, 'evg.svg')  # Currently, only supports 'svg'.\n```\n\n## Advanced Usage\n\n### Merging modifiers\n\nBy merging a modifier event to the modifiee, users can construct a larger information unit.\n\n```python\nfrom pyknp import KNP\nfrom pyknp_eventgraph import EventGraph\n\ndocument = ['\xe3\x82\x82\xe3\x81\xa3\xe3\x81\xa8\xe3\x81\xa8\xe3\x82\x8d\xe3\x81\xbf\xe3\x81\x8c\xe6\x8c\x81\xe7\xb6\x9a\xe3\x81\x99\xe3\x82\x8b\xe4\xbd\x9c\xe3\x82\x8a\xe6\x96\xb9\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe6\xac\xb2\xe3\x81\x97\xe3\x81\x84\xe3\x80\x82']\nknp = KNP()\nanalysis = [knp.parse(sentence) for sentence in document]\n\nevg = EventGraph.build(analysis)\nprint(evg)  # <EventGraph, #sentences: 1, #events: 2, #relations: 1>\n\n# Investigate the relation.\nrelation = evg.relations[0]\nprint(relation)           # <Relation, label: \xe9\x80\xa3\xe4\xbd\x93\xe4\xbf\xae\xe9\xa3\xbe, modifier_evid: 0, head_evid: 1>\nprint(relation.modifier)  # <Event, evid: 0, surf: \xe3\x82\x82\xe3\x81\xa3\xe3\x81\xa8\xe3\x81\xa8\xe3\x82\x8d\xe3\x81\xbf\xe3\x81\x8c\xe6\x8c\x81\xe7\xb6\x9a\xe3\x81\x99\xe3\x82\x8b>\nprint(relation.head)      # <Event, evid: 1, surf: \xe4\xbd\x9c\xe3\x82\x8a\xe6\x96\xb9\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe6\xac\xb2\xe3\x81\x97\xe3\x81\x84\xe3\x80\x82>\n\n# To merge modifier events, enable `include_modifiers`.\nprint(relation.head.surf)                           # \xe4\xbd\x9c\xe3\x82\x8a\xe6\x96\xb9\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe6\xac\xb2\xe3\x81\x97\xe3\x81\x84\xe3\x80\x82\nprint(relation.head.surf_(include_modifiers=True))  # \xe3\x82\x82\xe3\x81\xa3\xe3\x81\xa8\xe3\x81\xa8\xe3\x82\x8d\xe3\x81\xbf\xe3\x81\x8c\xe6\x8c\x81\xe7\xb6\x9a\xe3\x81\x99\xe3\x82\x8b\xe4\xbd\x9c\xe3\x82\x8a\xe6\x96\xb9\xe3\x82\x92\xe3\x81\x97\xe3\x81\xa6\xe6\xac\xb2\xe3\x81\x97\xe3\x81\x84\xe3\x80\x82\n\n# Other formats also support `include_modifiers`.\nprint(relation.head.mrphs_(include_modifiers=True))  # \xe3\x82\x82\xe3\x81\xa3\xe3\x81\xa8 \xe3\x81\xa8\xe3\x82\x8d\xe3\x81\xbf \xe3\x81\x8c \xe6\x8c\x81\xe7\xb6\x9a \xe3\x81\x99\xe3\x82\x8b \xe4\xbd\x9c\xe3\x82\x8a \xe6\x96\xb9 \xe3\x82\x92 \xe3\x81\x97\xe3\x81\xa6 \xe6\xac\xb2\xe3\x81\x97\xe3\x81\x84 \xe3\x80\x82\nprint(relation.head.normalized_mrphs_(include_modifiers=True))  # \xe3\x82\x82\xe3\x81\xa3\xe3\x81\xa8 \xe3\x81\xa8\xe3\x82\x8d\xe3\x81\xbf \xe3\x81\x8c \xe6\x8c\x81\xe7\xb6\x9a \xe3\x81\x99\xe3\x82\x8b \xe4\xbd\x9c\xe3\x82\x8a \xe6\x96\xb9 \xe3\x82\x92 \xe3\x81\x97\xe3\x81\xa6 \xe6\xac\xb2\xe3\x81\x97\xe3\x81\x84\n```\n\n### Binary serialization\n\nWhen an EventGraph is serialized in a JSON format, it will lose some functionality, including access to KNP objects and modifier merging.\nTo keep full functionality, use Python's pickle utility for serialization.\n\n```python\n# Save an EventGraph using Python's pickle utility.\nevg.save('evg.pkl', binary=True)\n\n# Load an EventGraph using Python's pickle utility.\nwith open('evg.pkl', 'rb') as f:\n    evg_ = EventGraph.load(f, binary=True)\n```\n\n## CLI\n\n### EventGraph Construction\n\n```\n$ echo '\xe5\xbd\xbc\xe5\xa5\xb3\xe3\x81\xaf\xe6\xb5\xb7\xe5\xa4\x96\xe5\x8b\xa4\xe5\x8b\x99\xe3\x81\x8c\xe9\x95\xb7\xe3\x81\x84\xe3\x81\xae\xe3\x81\xa7\xe3\x80\x81\xe8\x8b\xb1\xe8\xaa\x9e\xe3\x81\x8c\xe3\x81\x86\xe3\x81\xbe\xe3\x81\x84\xe3\x81\xab\xe9\x81\x95\xe3\x81\x84\xe3\x81\xaa\xe3\x81\x84\xe3\x80\x82' | jumanpp | knp -tab | evg -o example-eventgraph.json\n```\n\n### EventGraph Visualization\n\n```\n$ evgviz example-eventgraph.json example-eventgraph.svg\n```\n\n## Documents\n\n[https://pyknp-eventgraph.readthedocs.io/en/latest/](https://pyknp-eventgraph.readthedocs.io/en/latest/)\n\n## Authors\n\n- Kurohashi-Kawahara Lab, Kyoto University.\n- contact@nlp.ist.i.kyoto-u.ac.jp\n",
    'author': 'Hirokazu Kiyomaru',
    'author_email': 'h.kiyomaru@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ku-nlp/pyknp-eventgraph',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
