"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new aws_servicecatalog_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.portfolioId, tagOptions);
    }
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    setLocalLaunchRoleName(product, launchRoleName, options = {}) {
        const launchRole = new iam.Role(this, `LaunchRole${launchRoleName}`, {
            roleName: launchRoleName,
            assumedBy: new iam.ServicePrincipal('servicecatalog.amazonaws.com'),
        });
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
        return launchRole;
    }
    setLocalLaunchRole(product, launchRole, options = {}) {
        validation_1.InputValidator.validateRoleNameSetForLocalLaunchRole(launchRole);
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
    }
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new aws_servicecatalog_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * A Service Catalog portfolio.
 */
class Portfolio extends PortfolioBase {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_alpha_PortfolioProps(props);
        this.validatePortfolioProps(props);
        this.portfolio = new aws_servicecatalog_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    /**
     * Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog-alpha.Portfolio", version: "2.20.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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