"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.EndpointType = exports.SecurityPolicy = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigatewayv2_1 = require("aws-cdk-lib/aws-apigatewayv2");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    /** Cipher suite TLS 1.2 */
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
/**
 * Endpoint type for a domain name.
 */
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
/**
 * Custom domain resource for the API
 */
class DomainName extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.domainNameConfigurations = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_DomainNameProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DomainName);
            }
            throw error;
        }
        if (props.domainName === '') {
            throw new Error('empty string for domainName not allowed');
        }
        // validation for ownership certificate
        if (props.ownershipCertificate && !props.mtls) {
            throw new Error('ownership certificate can only be used with mtls domains');
        }
        const mtlsConfig = this.configureMTLS(props.mtls);
        const domainNameProps = {
            domainName: props.domainName,
            domainNameConfigurations: aws_cdk_lib_1.Lazy.any({ produce: () => this.domainNameConfigurations }),
            mutualTlsAuthentication: mtlsConfig,
        };
        const resource = new aws_apigatewayv2_1.CfnDomainName(this, 'Resource', domainNameProps);
        this.name = resource.ref;
        this.regionalDomainName = aws_cdk_lib_1.Token.asString(resource.getAtt('RegionalDomainName'));
        this.regionalHostedZoneId = aws_cdk_lib_1.Token.asString(resource.getAtt('RegionalHostedZoneId'));
        if (props.certificate) {
            this.addEndpoint(props);
        }
    }
    /**
     * Import from attributes
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_DomainNameAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDomainNameAttributes);
            }
            throw error;
        }
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.regionalDomainName = attrs.regionalDomainName;
                this.regionalHostedZoneId = attrs.regionalHostedZoneId;
                this.name = attrs.name;
            }
        }
        return new Import(scope, id);
    }
    configureMTLS(mtlsConfig) {
        if (!mtlsConfig)
            return undefined;
        return {
            truststoreUri: mtlsConfig.bucket.s3UrlForObject(mtlsConfig.key),
            truststoreVersion: mtlsConfig.version,
        };
    }
    /**
     * Adds an endpoint to a domain name.
     * @param options domain name endpoint properties to be set
     */
    addEndpoint(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigatewayv2_alpha_EndpointOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEndpoint);
            }
            throw error;
        }
        const domainNameConfig = {
            certificateArn: options.certificate.certificateArn,
            certificateName: options.certificateName,
            endpointType: options.endpointType ? options.endpointType?.toString() : 'REGIONAL',
            ownershipVerificationCertificateArn: options.ownershipCertificate?.certificateArn,
            securityPolicy: options.securityPolicy?.toString(),
        };
        this.validateEndpointType(domainNameConfig.endpointType);
        this.domainNameConfigurations.push(domainNameConfig);
    }
    // validates that the new domain name configuration has a unique endpoint
    validateEndpointType(endpointType) {
        for (let config of this.domainNameConfigurations) {
            if (endpointType && endpointType == config.endpointType) {
                throw new Error(`an endpoint with type ${endpointType} already exists`);
            }
        }
    }
}
exports.DomainName = DomainName;
_a = JSII_RTTI_SYMBOL_1;
DomainName[_a] = { fqn: "@aws-cdk/aws-apigatewayv2-alpha.DomainName", version: "2.51.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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