"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   this, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.extraLambdaEnv = {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props?.vpc,
            vpcSubnets: this.props?.vpcSubnets,
            allowPublicSubnet: this.props?.allowPublicSubnet,
            securityGroups: this.props?.securityGroup ? [this.props.securityGroup] : undefined,
            layers: this.props?.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        if (this.props?.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props?.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild'),
                new lambda_1.LambdaRunner(this, 'Lambda'),
                new fargate_1.FargateRunner(this, 'Fargate'),
            ];
        }
        this.orchestrator = this.stateMachine(props);
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine(props) {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        let deleteRunnerFunction = this.deleteRunner();
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: false,
            }),
        });
        deleteRunnerTask.addRetry({
            errors: [
                'RunnerBusy',
            ],
            interval: cdk.Duration.minutes(1),
            backoffRate: 1,
            maxAttempts: 60,
        });
        const waitForIdleRunner = new aws_cdk_lib_1.aws_stepfunctions.Wait(this, 'Wait', {
            time: aws_cdk_lib_1.aws_stepfunctions.WaitTime.duration(props?.idleTimeout ?? cdk.Duration.minutes(10)),
        });
        const deleteIdleRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Idle Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: true,
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.and(...provider.labels.map(label => aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${label}`))), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .branch(waitForIdleRunner.next(deleteIdleRunnerTask))
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        let logOptions;
        if (this.props?.logOptions) {
            this.stateMachineLogGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                logGroupName: props?.logOptions?.logGroupName,
                retention: props?.logOptions?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            logOptions = {
                destination: this.stateMachineLogGroup,
                includeExecutionData: props?.logOptions?.includeExecutionData ?? true,
                level: props?.logOptions?.level ?? aws_cdk_lib_1.aws_stepfunctions.LogLevel.ALL,
            };
        }
        const stateMachine = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
            logs: logOptions,
        });
        for (const provider of this.providers) {
            provider.grantStateMachine(stateMachine);
        }
        return stateMachine;
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                STEP_FUNCTION_LOG_GROUP: this.stateMachineLogGroup?.logGroupName ?? '',
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        const providers = this.providers.map(provider => provider.status(statusFunction));
        // expose providers as stack metadata as it's too big for Lambda environment variables
        // specifically integration testing got an error because lambda update request was >5kb
        const stack = cdk.Stack.of(this);
        const f = statusFunction.node.defaultChild;
        f.addPropertyOverride('Environment.Variables.LOGICAL_ID', f.logicalId);
        f.addPropertyOverride('Environment.Variables.STACK_NAME', stack.stackName);
        f.addMetadata('providers', providers);
        statusFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource'],
            resources: [stack.stackId],
        }));
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${stack.region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.7.5" };
//# sourceMappingURL=data:application/json;base64,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