"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const utils_1 = require("../utils");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * GitHub Actions runner provider using Lambda to execute jobs.
 *
 * Creates a Docker-based function that gets executed for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class LambdaRunner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.labels = this.labelsFromProperties('lambda', props?.label, props?.labels);
        this.vpc = props?.vpc;
        this.securityGroups = props?.securityGroup ? [props.securityGroup] : props?.securityGroups;
        const imageBuilder = props?.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: LambdaRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = this.image = imageBuilder.bind();
        let architecture;
        if (image.os.is(common_1.Os.LINUX)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                architecture = aws_cdk_lib_1.aws_lambda.Architecture.X86_64;
            }
            if (image.architecture.is(common_1.Architecture.ARM64)) {
                architecture = aws_cdk_lib_1.aws_lambda.Architecture.ARM_64;
            }
        }
        if (!architecture) {
            throw new Error(`Unable to find support Lambda architecture for ${image.os.name}/${image.architecture.name}`);
        }
        // get image digest and make sure to get it every time the lambda function might be updated
        // pass all variables that may change and cause a function update
        // if we don't get the latest digest, the update may fail as a new image was already built outside the stack on a schedule
        // we automatically delete old images, so we must always get the latest digest
        const imageDigest = this.imageDigest(image, {
            version: 1,
            labels: this.labels,
            architecture: architecture.name,
            vpc: this.vpc?.vpcId,
            securityGroups: this.securityGroups?.map(sg => sg.securityGroupId),
            vpcSubnets: props?.subnetSelection?.subnets?.map(s => s.subnetId),
            timeout: props?.timeout?.toSeconds(),
            memorySize: props?.memorySize,
            ephemeralStorageSize: props?.ephemeralStorageSize?.toKibibytes(),
            logRetention: props?.logRetention?.toFixed(),
        });
        this.function = new aws_cdk_lib_1.aws_lambda.DockerImageFunction(this, 'Function', {
            description: `GitHub Actions runner for labels ${this.labels}`,
            // CDK requires "sha256:" literal prefix -- https://github.com/aws/aws-cdk/blob/ba91ca45ad759ab5db6da17a62333e2bc11e1075/packages/%40aws-cdk/aws-ecr/lib/repository.ts#L184
            code: aws_cdk_lib_1.aws_lambda.DockerImageCode.fromEcr(image.imageRepository, { tagOrDigest: `sha256:${imageDigest}` }),
            architecture,
            vpc: this.vpc,
            securityGroups: this.securityGroups,
            vpcSubnets: props?.subnetSelection,
            timeout: props?.timeout || cdk.Duration.minutes(15),
            memorySize: props?.memorySize || 2048,
            ephemeralStorageSize: props?.ephemeralStorageSize || cdk.Size.gibibytes(10),
            logRetention: props?.logRetention || aws_logs_1.RetentionDays.ONE_MONTH,
        });
        this.grantPrincipal = this.function.grantPrincipal;
        this.addImageUpdater(image);
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.function.connections;
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        const invoke = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, this.labels.join(', '), {
            lambdaFunction: this.function,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                token: parameters.runnerTokenPath,
                runnerName: parameters.runnerNamePath,
                label: this.labels.join(','),
                githubDomain: parameters.githubDomainPath,
                owner: parameters.ownerPath,
                repo: parameters.repoPath,
            }),
        });
        this.addRetry(invoke, ['Lambda.LambdaException', 'Lambda.Ec2ThrottledException', 'Lambda.Ec2UnexpectedException', 'Lambda.EniLimitReachedException', 'Lambda.TooManyRequestsException']);
        return invoke;
    }
    addImageUpdater(image) {
        // Lambda needs to be pointing to a specific image digest and not just a tag.
        // Whenever we update the tag to a new digest, we need to update the lambda.
        const updater = utils_1.BundledNodejsFunction.singleton(this, 'update-lambda', {
            description: 'Function that updates a GitHub Actions runner function with the latest image digest after the image has been rebuilt',
            timeout: cdk.Duration.minutes(15),
        });
        updater.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['lambda:UpdateFunctionCode'],
            resources: [this.function.functionArn],
        }));
        let lambdaTarget = new aws_cdk_lib_1.aws_events_targets.LambdaFunction(updater, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                lambdaName: this.function.functionName,
                repositoryUri: image.imageRepository.repositoryUri,
                repositoryTag: image.imageTag,
            }),
        });
        const rule = image.imageRepository.onEvent('Push rule', {
            description: 'Update GitHub Actions runner Lambda on ECR image push',
            eventPattern: {
                detailType: ['ECR Image Action'],
                detail: {
                    'action-type': ['PUSH'],
                    'repository-name': [image.imageRepository.repositoryName],
                    'image-tag': [image.imageTag],
                    'result': ['SUCCESS'],
                },
            },
            target: lambdaTarget,
        });
        // the event never triggers without this - not sure why
        rule.node.defaultChild.addDeletionOverride('Properties.EventPattern.resources');
    }
    grantStateMachine(_) {
    }
    status(statusFunctionRole) {
        this.image.imageRepository.grant(statusFunctionRole, 'ecr:DescribeImages');
        return {
            type: this.constructor.name,
            labels: this.labels,
            vpcArn: this.vpc?.vpcArn,
            securityGroups: this.securityGroups?.map(sg => sg.securityGroupId),
            roleArn: this.function.role?.roleArn,
            image: {
                imageRepository: this.image.imageRepository.repositoryUri,
                imageTag: this.image.imageTag,
                imageBuilderLogGroup: this.image.logGroup?.logGroupName,
            },
        };
    }
    imageDigest(image, variableSettings) {
        // describe ECR image to get its digest
        // the physical id is random so the resource always runs and always gets the latest digest, even if a scheduled build replaced the stack image
        const reader = new aws_cdk_lib_1.custom_resources.AwsCustomResource(this, 'Image Digest Reader', {
            onCreate: {
                service: 'ECR',
                action: 'describeImages',
                parameters: {
                    repositoryName: image.imageRepository.repositoryName,
                    imageIds: [
                        {
                            imageTag: image.imageTag,
                        },
                    ],
                },
                physicalResourceId: aws_cdk_lib_1.custom_resources.PhysicalResourceId.of('ImageDigest'),
            },
            onUpdate: {
                service: 'ECR',
                action: 'describeImages',
                parameters: {
                    repositoryName: image.imageRepository.repositoryName,
                    imageIds: [
                        {
                            imageTag: image.imageTag,
                        },
                    ],
                },
                physicalResourceId: aws_cdk_lib_1.custom_resources.PhysicalResourceId.of('ImageDigest'),
            },
            onDelete: {
                // this will NOT be called thanks to RemovalPolicy.RETAIN below
                // we only use this to force the custom resource to be called again and get a new digest
                service: 'fake',
                action: 'fake',
                parameters: variableSettings,
            },
            policy: aws_cdk_lib_1.custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [image.imageRepository.repositoryArn],
            }),
            resourceType: 'Custom::EcrImageDigest',
            installLatestAwsSdk: false,
            logRetention: aws_logs_1.RetentionDays.ONE_MONTH,
        });
        const res = reader.node.tryFindChild('Resource');
        if (res) {
            // don't actually call the fake onDelete above
            res.applyRemovalPolicy(cdk.RemovalPolicy.RETAIN);
        }
        else {
            throw new Error('Resource not found in AwsCustomResource. Report this bug at https://github.com/CloudSnorkel/cdk-github-runners/issues.');
        }
        // return only the digest because CDK expects 'sha256:' literal above
        return cdk.Fn.split(':', reader.getResponseField('imageDetails.0.imageDigest'), 2)[1];
    }
}
exports.LambdaRunner = LambdaRunner;
_a = JSII_RTTI_SYMBOL_1;
LambdaRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.LambdaRunner", version: "0.7.5" };
/**
 * Path to Dockerfile for Linux x64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
LambdaRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'lambda', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
LambdaRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'lambda', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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