"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImageBuilderBase = exports.ImageBuilderComponent = exports.ImageBuilderObjectBase = exports.uniqueImageBuilderName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
/**
 * @internal
 */
function uniqueImageBuilderName(scope) {
    return cdk.Names.uniqueResourceName(scope, {
        maxLength: 126,
        separator: '-',
        allowedSpecialCharacters: '_-',
    });
}
exports.uniqueImageBuilderName = uniqueImageBuilderName;
/**
 * @internal
 */
class ImageBuilderObjectBase extends cdk.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    version(type, name, data) {
        return new aws_cdk_lib_1.CustomResource(this, 'Version', {
            serviceToken: this.versionFunction().functionArn,
            resourceType: `Custom::ImageBuilder-${type}-Version`,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            properties: {
                ObjectType: type,
                ObjectName: name,
                VersionedData: data,
            },
        }).ref;
    }
    versionFunction() {
        return utils_1.BundledNodejsFunction.singleton(this, 'aws-image-builder-versioner', {
            description: 'Custom resource handler that bumps up Image Builder versions',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'imagebuilder:ListComponents',
                        'imagebuilder:ListContainerRecipes',
                        'imagebuilder:ListImageRecipes',
                    ],
                    resources: ['*'],
                }),
            ],
        });
    }
}
exports.ImageBuilderObjectBase = ImageBuilderObjectBase;
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
class ImageBuilderComponent extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push('$ErrorActionPreference = \'Stop\'');
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: extractCommands,
                    },
                });
            }
        }
        steps.push({
            name: 'Run',
            action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
            inputs: {
                commands: props.commands,
            },
        });
        const data = {
            name: props.displayName,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = uniqueImageBuilderName(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            description: props.description,
            platform: props.platform,
            version: this.version('Component', name, {
                platform: props.platform,
                data,
                description: props.description,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
}
exports.ImageBuilderComponent = ImageBuilderComponent;
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.7.5" };
/**
 * @internal
 */
class ImageBuilderBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        // arch
        this.architecture = props?.architecture ?? common_1.Architecture.X86_64;
        if (!this.architecture.isIn(props.supportedArchitectures)) {
            throw new Error(`Unsupported architecture: ${this.architecture.name}. Consider CodeBuild for faster image builds.`);
        }
        // os
        this.os = props?.os ?? common_1.Os.LINUX;
        if (!this.os.isIn(props.supportedOs)) {
            throw new Error(`Unsupported OS: ${this.os.name}.`);
        }
        // platform
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.platform = 'Windows';
        }
        else if (this.os.is(common_1.Os.LINUX)) {
            this.platform = 'Linux';
        }
        else {
            throw new Error(`Unsupported OS: ${this.os.name}.`);
        }
        // builder options
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        // vpc settings
        if (props?.vpc) {
            this.vpc = props.vpc;
            this.subnetId = props.vpc.selectSubnets(props.subnetSelection).subnetIds[0];
        }
        else {
            this.vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'Default VPC', { isDefault: true });
        }
        if (props?.securityGroups) {
            this.securityGroups = props.securityGroups;
        }
        else {
            this.securityGroups = [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })];
        }
        // instance type
        this.instanceType = props?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        if (!this.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`Builder architecture (${this.architecture.name}) doesn't match selected instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        // log settings
        this.logRetention = props?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        // runner version
        this.runnerVersion = props?.runnerVersion ?? common_1.RunnerVersion.latest();
        // description
        this.description = `Build ${props.imageTypeName} for GitHub Actions runner ${this.node.path} (${this.os.name}/${this.architecture.name})`;
    }
    createLog(recipeName) {
        return new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Log', {
            logGroupName: `/aws/imagebuilder/${recipeName}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
    }
    createInfrastructure(managedPolicies) {
        let role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: managedPolicies,
        });
        for (const component of this.components) {
            component.grantAssetsRead(role);
        }
        return new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: uniqueImageBuilderName(this),
            description: this.description,
            subnetId: this.subnetId,
            securityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            instanceTypes: [this.instanceType.toString()],
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    role.roleName,
                ],
            }).ref,
        });
    }
    createImage(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, 'Image', {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        image.node.addDependency(infra);
        image.node.addDependency(log);
        return image;
    }
    createPipeline(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'Pipeline', {
            name: uniqueImageBuilderName(this),
            description: this.description,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            schedule: scheduleOptions,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        pipeline.node.addDependency(infra);
        pipeline.node.addDependency(log);
        return pipeline;
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
}
exports.ImageBuilderBase = ImageBuilderBase;
//# sourceMappingURL=data:application/json;base64,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