"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const https = require("https");
const URL = require("url");
/**
* Custom CloudFormation resource helper
*/
class CustomResource {
    constructor(event, context, callback, logger) {
        /**
        * Stores functions executed when resource creation is requested
        */
        this.createFunctions = [];
        /**
        * Stores functions executed when resource update is requested
        */
        this.updateFunctions = [];
        /**
        * Stores functions executed when resource deletion is requested
        */
        this.deleteFunctions = [];
        if (typeof event.ResponseURL === 'undefined') {
            throw new Error('ResponseURL missing');
        }
        this.event = event;
        this.context = context;
        this.callback = callback;
        this.logger = logger || new StandardLogger();
        this.logger.debug(`REQUEST RECEIVED:\n${JSON.stringify(event)}`);
        this.timeout();
    }
    /**
    * Adds a function to the CREATE queue
    */
    onCreate(func) {
        this.createFunctions.push(func);
        return this;
    }
    /**
    * Adds a function to the UPDATE queue
    */
    onUpdate(func) {
        this.updateFunctions.push(func);
        return this;
    }
    /**
    * Adds a function to the DELETE queue
    */
    onDelete(func) {
        this.deleteFunctions.push(func);
        return this;
    }
    /**
    * Handles the Lambda event
    */
    handle(input) {
        const construct = this;
        try {
            let queue;
            if (this.event.RequestType == 'Create')
                queue = this.createFunctions;
            else if (this.event.RequestType == 'Update')
                queue = this.updateFunctions;
            else if (this.event.RequestType == 'Delete')
                queue = this.deleteFunctions;
            else {
                this.sendResponse('FAILED', `Unexpected request type: ${this.event.RequestType}`);
                return this;
            }
            let result = queue.reduce((current, next) => {
                return current.then((value) => {
                    return next(value);
                });
            }, Promise.resolve(input));
            result.then(function (response) {
                construct.logger.debug(response);
                construct.sendResponse('SUCCESS', `${construct.event.RequestType} completed successfully`);
            }).catch(function (err) {
                construct.logger.error(err, err.stack);
                construct.sendResponse('FAILED', err.message || err.code);
            });
        }
        catch (err) {
            construct.sendResponse('FAILED', err.message || err.code);
        }
        return this;
    }
    /**
    * Sends CloudFormation response just before the Lambda times out
    */
    timeout() {
        const construct = this;
        const handler = () => {
            construct.logger.error('Timeout FAILURE!');
            new Promise(() => construct.sendResponse('FAILED', 'Function timed out'))
                .then(() => construct.callback(new Error('Function timed out')));
        };
        setTimeout(handler, this.context.getRemainingTimeInMillis() - 1000);
    }
    /**
    * Sends CloudFormation response
    */
    sendResponse(responseStatus, responseData) {
        const construct = this;
        construct.logger.debug(`Sending response ${responseStatus}:\n${JSON.stringify(responseData)}`);
        var body = JSON.stringify({
            Status: responseStatus,
            Reason: `${responseData} | Full error in CloudWatch ${construct.context.logStreamName}`,
            PhysicalResourceId: construct.event.ResourceProperties.Name,
            StackId: construct.event.StackId,
            RequestId: construct.event.RequestId,
            LogicalResourceId: construct.event.LogicalResourceId,
            Data: {
                Message: responseData,
                Name: construct.event.ResourceProperties.Name,
            },
        });
        construct.logger.debug('RESPONSE BODY:\n', body);
        var url = URL.parse(construct.event.ResponseURL);
        var options = {
            hostname: url.hostname,
            port: 443,
            path: url.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': body.length,
            }
        };
        construct.logger.info('SENDING RESPONSE...\n');
        var request = https.request(options, function (response) {
            construct.logger.debug(`STATUS: ${response.statusCode}`);
            construct.logger.debug(`HEADERS: ${JSON.stringify(response.headers)}`);
            construct.context.done();
        });
        request.on('error', function (error) {
            construct.logger.error(`sendResponse Error: ${error}`);
            construct.context.done();
        });
        request.write(body);
        request.end();
    }
}
exports.CustomResource = CustomResource;
/**
* Standard logger class
*/
class StandardLogger {
    constructor(level) {
        this.level = level || 1 /* WARN */;
    }
    /**
    * Logs message with level ERROR
    */
    error(message, ...optionalParams) {
        if (this.level < 0 /* ERROR */)
            return;
        console.error(message, ...optionalParams);
    }
    /**
    * Logs message with level WARN
    */
    warn(message, ...optionalParams) {
        if (this.level < 1 /* WARN */)
            return;
        console.warn(message, ...optionalParams);
    }
    /**
    * Logs message with level INFO
    */
    info(message, ...optionalParams) {
        if (this.level < 2 /* INFO */)
            return;
        console.info(message, ...optionalParams);
    }
    /**
    * Logs message with level DEBUG
    */
    debug(message, ...optionalParams) {
        if (this.level < 3 /* DEBUG */)
            return;
        console.debug(message, ...optionalParams);
    }
    /**
    * Alias for info
    */
    log(message, ...optionalParams) {
        this.info(message, ...optionalParams);
    }
}
exports.StandardLogger = StandardLogger;
//# sourceMappingURL=data:application/json;base64,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