# -*- coding: utf-8 -*-
"""
-------------------------------------------------
   File Name：     joblib
   Description :
   Author :        Asdil
   date：          2020/3/3
-------------------------------------------------
   Change Activity:
                   2020/3/3:
-------------------------------------------------
"""
__author__ = 'Asdil'
import joblib
from aftool import tool
from aftool import fo
import tempfile


@tool.runtime
def dump(py_object, save_path, compress=3):
    """dump方法用于持久化python数据

    Parameters
    ----------
    py_object : object
        python 变量
    save_path : str
        存储地址
    compress : int
        压缩率默认3, 数值越高压缩比越大

    Returns
    ----------
    """
    joblib.dump(py_object, save_path, compress=compress)
    print(f'{save_path} 存储完成!')


@tool.runtime
def load(file_path):
    """load方法用于读取持久化文件

    Parameters
    ----------
    file_path : str
        文件路径

    Returns
    ----------
    读取的二进制文件
    """
    assert fo.is_exist(file_path), f'{file_path} 文件不存在!'
    obj = joblib.load(file_path)
    print(f'{file_path} 读取成功!')
    return obj


@tool.runtime
def parallel(args, func, njobs, backend=0, verbose=0):
    """parallel方法用于并行计算

    Parameters
    ----------
    args : list
        参数列表
    func : funcion
        对象函数
    njobs : int
        并行数
    backend : int
        并行任务种类
        0: loky 默认
        1: multiprocessing 多进程，鲁棒性不如loky
        2: threading 多线程, 当释放GIL效率高
    verbose : int
        日志输出级别, 最好在0-10 默认0
    Returns
    ----------
    """
    # 并行种类
    backend_dict = {0: 'loky', 1: 'multiprocessing', 2: 'threading'}
    backend = backend_dict[backend]

    ret = joblib.Parallel(
        n_jobs=njobs, backend=backend, verbose=verbose)(
        joblib.delayed(func)(arg) for arg in args)
    return ret


def memmap(data):
    """memmap方法用于参数共享内存的变量

    Parameters
    ----------
    data : object
        python变量
    Returns
    ----------
    内存共享的变量
    """
    tmp_folder = tempfile.mkdtemp()
    tmp_path = tool.path_join(tmp_folder, 'joblib.mmap')
    if fo.is_exist(tmp_path):  # 若存在则删除
        fo.del_file(tmp_path)
    _ = joblib.dump(data, tmp_path)
    memmap_data = joblib.load(tmp_path, mmap_mode='r+')
    return memmap_data


def example_ray():
    """ray方法用于ray是例子

    Parameters
    ----------
    param : str

    Returns
    ----------
    """
    print('import ray')
    print('ray.init(num_cpus=8)')
    print('args = []')
    print('')
    print('@ray.remote')
    print('def func(args):')
    print('    return 0')
    print('')
    print('df_id = ray.put(df) # 共享内存')
    print('ret = [f.remote() for i in args]')


def example_parallel():
    """example_parallel方法用于jobs.parallel例子

    Parameters
    ----------

    Returns
    ----------
    """
    print('from aftool import jobs')
    print('')
    print('def func():')
    print('    return 0')
    print('')
    print('args = []')
    print('mmp_df = jobs.memmap(mmp_df)')
    print('ret = jobs.parallel(args, func, njobs=8, backend=1)')
