# -*- coding: utf-8 -*-

from ldap3 import Server, Connection, ObjectDef, Reader, AttrDef, core
from datetime import datetime as dt
import re
import json
import argparse
import sys
import os
import logging

__author__ = "Robert Wikman <rbw@vault13.org>"
__version__ = "0.1.3"


class ParseError(Exception):
    pass


class Authz(object):
    """Creates a new Authz object for interfacing with the SVN access

    :param file: svn authz file
    :param config: authzync configuration dict
    :param local_db_file: Local user authz config
    """
    def __init__(self, file, config, local_db_file):
        self.logger = logging.getLogger(__name__)
        self.file = file
        self.config = config
        self.local_db_file = local_db_file
        self.repositories = {}

        self.user_count = 0

    def add(self, name, path, access, members):
        """Builds repository configuration dict, later used to generate the authz configuration file

        :param name: repository name
        :param path: repository path (i.e. /branches/test
        :param access: permissions (RO/RW)
        :param members: users allowed to access the repo
        """
        if name not in self.repositories:
            self.repositories.update({name: {}})

        if path not in self.repositories[name]:
            self.repositories[name].update(
                {
                    path: {
                        'RO': [],
                        'RW': []
                    }
                }
            )

        if members:
            self.repositories[name][path][access].extend(members)
            self.user_count += len(members)

    def write(self):
        """ Writes header and contents from `repository`

        """
        fh = open(self.file, 'w')
        fh.write("\n## AUTOMATICALLY GENERATED FILE - DO NOT EDIT ##")
        fh.write("\n##")
        fh.write("\n## Generated by: %s" % sys.argv[0])
        fh.write("\n## Generated at: %s" % dt.now())
        fh.write("\n## Sources used:")
        fh.write("\n##   - Local: %s" % self.local_db_file)
        fh.write("\n##   - LDAP: %s (%s)" % (self.config['ldap']['host'], self.config['ldap']['base_dn']))
        fh.write("\n##")
        fh.write("\n## Visit https://github.com/rbw0/authzync for more info\n")

        for repo_name in self.repositories:
            sections = self.repositories[repo_name]
            for path in sections:
                fh.write("\n[%s:%s]\n" % (repo_name, path))

                if sections[path]['RO']:
                    fh.write(''.join('%s = ro\n' % t for t in sections[path]['RO']))

                if sections[path]['RW']:
                    fh.write(''.join('%s = rw\n' % t for t in sections[path]['RW']))

        fh.write("\n")
        fh.close()

        return os.stat(self.file).st_size


class Repository(object):
    """Creates a repository object from an LDAP group

    :param group: LDAP group object
    :param config: authzync config dictionary
    """
    def __init__(self, group, config):
        self.logger = logging.getLogger(__name__)
        self.config = config
        self.group = group
        self.group_raw = str(group.name)
        self.section_raw = str(group.section)

        self.m_access = re.match(self.config['patterns']['access_pattern'], self.group_raw, re.IGNORECASE)
        self.m_section = re.match(self.config['patterns']['section_pattern'], self.section_raw)

    @property
    def name(self):
        """ `name` setter

        :raises: Raises ParseError if unable to parse repo name
        """
        try:
            return self.m_section.group('repo_name')
        except (IndexError, AttributeError):
            raise ParseError("Couldn't parse repository name in attribute '%s' (%s) for group '%s'. Pattern: %s" %
                             (self.config['mappings']['section_name'],
                              self.section_raw, self.group_raw,
                              self.config['patterns']['section_pattern']))

    @property
    def access(self):
        """ `access` setter

        :raises: ParseError if unable to parse repo access level
        """
        try:
            return self.m_access.group('repo_access').upper()
        except (IndexError, AttributeError):
            raise ParseError("Couldn't get repository permission from group '%s', using nocase pattern: %s" %
                             (self.group_raw, self.config['patterns']['access_pattern']))

    @property
    def path(self):
        """ `path` setter

        :raises: ParseError if unable to parse repo path
        """
        try:
            return self.m_section.group('repo_path')
        except (IndexError, AttributeError):
            raise ParseError("Couldn't parse repository path in attribute '%s' (%s) for group '%s'. Pattern: %s" %
                             (self.config['mappings']['section_name'],
                              self.section_raw, self.group_raw,
                              self.config['patterns']['section_pattern']))

    @property
    def members(self):
        """ `members` setter
        Sets members property to a `list` of users
        Silently catches and skips repo groups without members
        """
        try:
            return [str(user.name) for user in self.group.members]
        except core.exceptions.LDAPKeyError:
            self.logger.warning("No members (LDAP) found for repository '%s'" % self.name)
            pass


def create_logger(config):
    """Creates a logger used throughout the application

    :return: logger instance
    """
    logger = logging.getLogger(__name__)
    logger.setLevel(logging.DEBUG)

    h = logging.FileHandler(config['logging']['file'])
    # h = logging.StreamHandler()
    # h.setLevel(logging.DEBUG)

    formatter = logging.Formatter(config['logging']['format'])
    h.setFormatter(formatter)
    logger.addHandler(h)

    return logger


def main():
    # Argument parser
    parser = argparse.ArgumentParser(description='SVN AuthZ-LDAP sync tool')
    parser.add_argument('--config', dest="config_file", required=True, help='configuration file')
    parser.add_argument('--authz', dest="authz_file", required=True, help='SVN authz file')
    parser.add_argument('--local_db', dest="local_db_file", required=False, help='local users / groups')
    args = parser.parse_args()

    # Read config
    with open(args.config_file) as config_file:
        config = json.load(config_file)

    # Create logger
    logger = create_logger(config)
    logger.info('### SVN Authz-LDAP sync starting ###')

    # Create authz instance
    authz = Authz(file=args.authz_file, config=config, local_db_file=args.local_db_file)

    # Check if local_db_file was passed as an argument, and if so, `add` its config to `authz`
    if args.local_db_file:
        with open(args.local_db_file) as local_db_file:
            local_db = json.load(local_db_file)
            logger.info("Getting local repository data")
            for repo_name in local_db:
                repo = local_db[repo_name]
                for path in repo:
                    authz.add(repo_name, path, 'RO', repo[path]['RO'])
                    authz.add(repo_name, path, 'RW', repo[path]['RW'])

    server = Server(config['ldap']['host'], port=config['ldap']['port'], use_ssl=config['ldap']['use_ssl'])

    # User object definition
    repo_user = ObjectDef('user')
    repo_user += AttrDef(config['mappings']['user_name'], key='name')

    # Group object definition
    repo_group = ObjectDef('group')
    repo_group += AttrDef(config['mappings']['group_name'], key='name')
    repo_group += AttrDef(config['mappings']['group_members'], key='members', dereference_dn=repo_user)
    repo_group += AttrDef(config['mappings']['section_name'], key='section')

    try:
        # Connect to server
        conn = Connection(server, raise_exceptions=True,
                          user=config['ldap']['bind_user'], password=config['ldap']['bind_password'])

        # Create reader instance
        r = Reader(conn, repo_group, base=config['ldap']['base_dn'], query=config['ldap']['group_filter'])

        logger.info("Getting LDAP repository data")

        # Create a `Repository` instance for each group (repo) returned from the LDAP directory
        repositories = [Repository(group=group, config=config) for group in r.search()]
    except (core.exceptions.LDAPSocketOpenError,
            core.exceptions.LDAPInvalidCredentialsResult,
            core.exceptions.LDAPNoSuchObjectResult) as e:
        logger.critical(e)
        sys.exit(1)
    except KeyError:
        logger.critical("Error decoding response from server. Made sure base DN(%s) is correct." % config['ldap']['base_dn'])
        sys.exit(1)

    # Iterate over `Repository` instances and add them to Authz
    for r in repositories:
        try:
            authz.add(r.name, r.path, r.access, r.members)
        except ParseError as e:
            logger.error(e)

    logger.info("Loaded %d user entries in %d repositories" % (authz.user_count, len(authz.repositories)))

    if not authz.repositories:
        logger.critical("No valid authz rules found. Bailing out...")
        sys.exit(1)

    size = authz.write()
    logger.info("Wrote %d bytes to authz file: %s" % (size, args.authz_file))

if __name__ == "__main__":
    main()
