"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StacIngestor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_python_alpha_1 = require("@aws-cdk/aws-lambda-python-alpha");
const constructs_1 = require("constructs");
class StacIngestor extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.table = this.buildTable();
        const env = {
            DYNAMODB_TABLE: this.table.tableName,
            ROOT_PATH: `/${props.stage}`,
            NO_PYDANTIC_SSM_SETTINGS: "1",
            STAC_URL: props.stacUrl,
            DATA_ACCESS_ROLE: props.dataAccessRole.roleArn,
            ...props.apiEnv,
        };
        const handler = this.buildApiLambda({
            table: this.table,
            env,
            dataAccessRole: props.dataAccessRole,
            stage: props.stage,
            dbSecret: props.stacDbSecret,
            dbVpc: props.vpc,
            dbSecurityGroup: props.stacDbSecurityGroup,
            subnetSelection: props.subnetSelection,
        });
        this.buildApiEndpoint({
            handler,
            stage: props.stage,
            endpointConfiguration: props.apiEndpointConfiguration,
            policy: props.apiPolicy,
        });
        this.buildIngestor({
            table: this.table,
            env: env,
            dbSecret: props.stacDbSecret,
            dbVpc: props.vpc,
            dbSecurityGroup: props.stacDbSecurityGroup,
            subnetSelection: props.subnetSelection,
        });
        this.registerSsmParameter({
            name: "dynamodb_table",
            value: this.table.tableName,
            description: "Name of table used to store ingestions",
        });
    }
    buildTable() {
        const table = new aws_cdk_lib_1.aws_dynamodb.Table(this, "ingestions-table", {
            partitionKey: { name: "created_by", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            sortKey: { name: "id", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            billingMode: aws_cdk_lib_1.aws_dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            stream: aws_cdk_lib_1.aws_dynamodb.StreamViewType.NEW_IMAGE,
        });
        table.addGlobalSecondaryIndex({
            indexName: "status",
            partitionKey: { name: "status", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
            sortKey: { name: "created_at", type: aws_cdk_lib_1.aws_dynamodb.AttributeType.STRING },
        });
        return table;
    }
    buildApiLambda(props) {
        const handler_role = new aws_cdk_lib_1.aws_iam.Role(this, "execution-role", {
            description: "Role used by STAC Ingestor. Manually defined so that we can choose a name that is supported by the data access roles trust policy",
            roleName: `stac-ingestion-api-${props.stage}`,
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("lambda.amazonaws.com"),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaBasicExecutionRole"),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AWSLambdaVPCAccessExecutionRole"),
            ],
        });
        const handler = new aws_lambda_python_alpha_1.PythonFunction(this, "api-handler", {
            entry: `${__dirname}/runtime`,
            index: "src/handler.py",
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: { DB_SECRET_ARN: props.dbSecret.secretArn, ...props.env },
            vpc: props.dbVpc,
            vpcSubnets: props.subnetSelection,
            allowPublicSubnet: true,
            role: handler_role,
            memorySize: 2048,
        });
        // Allow handler to read DB secret
        props.dbSecret.grantRead(handler);
        // Allow handler to connect to DB
        props.dbSecurityGroup.addIngressRule(handler.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connections from STAC Ingestor");
        props.table.grantReadWriteData(handler);
        props.dataAccessRole.grantAssumeRole(handler_role);
        return handler;
    }
    buildIngestor(props) {
        const handler = new aws_lambda_python_alpha_1.PythonFunction(this, "stac-ingestor", {
            entry: `${__dirname}/runtime`,
            index: "src/ingestor.py",
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            timeout: aws_cdk_lib_1.Duration.seconds(180),
            environment: { DB_SECRET_ARN: props.dbSecret.secretArn, ...props.env },
            vpc: props.dbVpc,
            vpcSubnets: props.subnetSelection,
            allowPublicSubnet: true,
            memorySize: 2048,
        });
        // Allow handler to read DB secret
        props.dbSecret.grantRead(handler);
        // Allow handler to connect to DB
        props.dbSecurityGroup.addIngressRule(handler.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connections from STAC Ingestor");
        // Allow handler to write results back to DBƒ
        props.table.grantWriteData(handler);
        // Trigger handler from writes to DynamoDB table
        handler.addEventSource(new aws_cdk_lib_1.aws_lambda_event_sources.DynamoEventSource(props.table, {
            // Read when batches reach size...
            batchSize: 1000,
            // ... or when window is reached.
            maxBatchingWindow: aws_cdk_lib_1.Duration.seconds(10),
            // Read oldest data first.
            startingPosition: aws_cdk_lib_1.aws_lambda.StartingPosition.TRIM_HORIZON,
            retryAttempts: 1,
        }));
        return handler;
    }
    buildApiEndpoint(props) {
        return new aws_cdk_lib_1.aws_apigateway.LambdaRestApi(this, `${aws_cdk_lib_1.Stack.of(this).stackName}-ingestor-api`, {
            handler: props.handler,
            proxy: true,
            cloudWatchRole: true,
            deployOptions: { stageName: props.stage },
            endpointExportName: `ingestor-api-${props.stage}`,
            endpointConfiguration: props.endpointConfiguration,
            policy: props.policy,
        });
    }
    registerSsmParameter(props) {
        const parameterNamespace = aws_cdk_lib_1.Stack.of(this).stackName;
        return new aws_cdk_lib_1.aws_ssm.StringParameter(this, `${props.name.replace("_", "-")}-parameter`, {
            description: props.description,
            parameterName: `/${parameterNamespace}/${props.name}`,
            stringValue: props.value,
        });
    }
}
exports.StacIngestor = StacIngestor;
_a = JSII_RTTI_SYMBOL_1;
StacIngestor[_a] = { fqn: "cdk-pgstac.StacIngestor", version: "2.6.1" };
//# sourceMappingURL=data:application/json;base64,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