"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const mediastore = require("@aws-cdk/aws-mediastore");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Test the default deployment pattern variables
// --------------------------------------------------------------
test('Test the default deployment pattern variables', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(cloudFrontToMediaStore.cloudFrontWebDistribution).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.mediaStoreContainer).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontLoggingBucket).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginRequestPolicy).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontFunction).not.toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment without HTTP security headers
// --------------------------------------------------------------
test('Test the deployment without HTTP security headers', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        insertHttpSecurityHeaders: false
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontFunction).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with existing MediaStore container
// --------------------------------------------------------------
test('Test the deployment with existing MediaStore container', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MyMediaStoreContainer', {
        containerName: 'MyMediaStoreContainer'
    });
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        existingMediaStoreContainerObj: mediaStoreContainer
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MyMediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer'
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided MediaStore properties
// --------------------------------------------------------------
test('Test the deployment with the user provided MediaStore properties', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        mediaStoreContainerProps: {
            containerName: 'MyMediaStoreContainer',
            policy: '{}',
            lifecyclePolicy: '{}',
            corsPolicy: [],
            metricPolicy: {
                containerLevelMetrics: 'DISABLED'
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer',
        Policy: '{}',
        LifecyclePolicy: '{}',
        CorsPolicy: [],
        MetricPolicy: {
            ContainerLevelMetrics: 'DISABLED'
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided CloudFront properties
// --------------------------------------------------------------
test('Test the deployment with the user provided CloudFront properties', () => {
    // Initial setup
    const stack = new core_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        cloudFrontDistributionProps: {
            defaultBehavior: {
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.HTTPS_ONLY,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'https-only'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        AccessLoggingEnabled: true,
        ContainerName: {
            Ref: 'AWS::StackName'
        },
        CorsPolicy: [
            {
                AllowedHeaders: ['*'],
                AllowedMethods: ['GET'],
                AllowedOrigins: ['*'],
                ExposeHeaders: ["*"],
                MaxAgeSeconds: 3000
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
        CloudFrontOriginAccessIdentityConfig: {
            Comment: {
                'Fn::Join': [
                    '',
                    [
                        'access-identity-',
                        {
                            Ref: 'AWS::Region',
                        },
                        '-',
                        {
                            Ref: 'AWS::StackName',
                        }
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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