"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const sns = require("../lib");
const lib_1 = require("../lib");
module.exports = {
    'topic tests': {
        'all defaults'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic');
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic"
                    }
                }
            });
            test.done();
        },
        'specify topicName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "TopicName": "topicName"
                        }
                    }
                }
            });
            test.done();
        },
        'specify displayName'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                displayName: 'displayName'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName"
                        }
                    }
                }
            });
            test.done();
        },
        'specify both'(test) {
            const stack = new cdk.Stack();
            new sns.Topic(stack, 'MyTopic', {
                topicName: 'topicName',
                displayName: 'displayName'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyTopic86869434": {
                        "Type": "AWS::SNS::Topic",
                        "Properties": {
                            "DisplayName": "displayName",
                            "TopicName": "topicName"
                        }
                    }
                }
            });
            test.done();
        },
    },
    'can add a policy to the topic'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        topic.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sns:*'],
            principals: [new iam.ArnPrincipal('arn')]
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        "Sid": "0",
                        "Action": "sns:*",
                        "Effect": "Allow",
                        "Principal": { "AWS": "arn" },
                        "Resource": "*"
                    }]
            }
        }));
        test.done();
    },
    'give publishing permissions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const user = new iam.User(stack, 'User');
        // WHEN
        topic.grantPublish(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                Version: '2012-10-17',
                "Statement": [
                    {
                        "Action": "sns:Publish",
                        "Effect": "Allow",
                        "Resource": stack.resolve(topic.topicArn)
                    }
                ],
            }
        }));
        test.done();
    },
    'topic resource policy includes unique SIDs'(test) {
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'MyTopic');
        topic.addToResourcePolicy(new iam.PolicyStatement({ actions: ['statement0'] }));
        topic.addToResourcePolicy(new iam.PolicyStatement({ actions: ['statement1'] }));
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyTopic86869434": {
                    "Type": "AWS::SNS::Topic"
                },
                "MyTopicPolicy12A5EC17": {
                    "Type": "AWS::SNS::TopicPolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "statement0",
                                    "Effect": "Allow",
                                    "Sid": "0"
                                },
                                {
                                    "Action": "statement1",
                                    "Effect": "Allow",
                                    "Sid": "1"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Topics": [
                            {
                                "Ref": "MyTopic86869434"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'fromTopicArn'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const imported = sns.Topic.fromTopicArn(stack2, 'Imported', 'arn:aws:sns:*:123456789012:my_corporate_topic');
        // THEN
        test.deepEqual(imported.topicName, 'my_corporate_topic');
        test.deepEqual(imported.topicArn, 'arn:aws:sns:*:123456789012:my_corporate_topic');
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.deepEqual(stack.resolve(topic.metricNumberOfMessagesPublished()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'NumberOfMessagesPublished',
            period: cdk.Duration.minutes(5),
            statistic: 'Sum'
        });
        test.deepEqual(stack.resolve(topic.metricPublishSize()), {
            dimensions: { TopicName: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } },
            namespace: 'AWS/SNS',
            metricName: 'PublishSize',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    },
    'subscription is created under the topic scope by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const topic = new lib_1.Topic(stack, 'Topic');
        // WHEN
        topic.addSubscription({
            bind: () => ({
                protocol: lib_1.SubscriptionProtocol.HTTP,
                endpoint: 'http://foo/bar',
                subscriberId: 'my-subscription'
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription'));
        test.done();
    },
    'if "scope" is defined, subscription will be created under that scope'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'A');
        const stack2 = new core_1.Stack(app, 'B');
        const topic = new lib_1.Topic(stack, 'Topic');
        // WHEN
        topic.addSubscription({
            bind: () => ({
                protocol: lib_1.SubscriptionProtocol.HTTP,
                endpoint: 'http://foo/bar',
                subscriberScope: stack2,
                subscriberId: 'subscriberId'
            })
        });
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::SNS::Subscription'));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::SNS::Subscription'));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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