/*
 * Decompiled with CFR 0.152.
 */
package org.xerial.snappy;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Arrays;
import org.xerial.snappy.Snappy;
import org.xerial.snappy.SnappyFramed;

public final class SnappyFramedInputStream
extends InputStream
implements ReadableByteChannel {
    private final ReadableByteChannel rbc;
    private final ByteBuffer frameHeader;
    private final boolean verifyChecksums;
    private ByteBuffer input;
    private ByteBuffer uncompressedDirect;
    private boolean closed;
    private boolean eof;
    private int valid;
    private int position;
    private byte[] buffer;

    public SnappyFramedInputStream(InputStream inputStream) throws IOException {
        this(inputStream, true);
    }

    public SnappyFramedInputStream(InputStream inputStream, boolean bl) throws IOException {
        this(Channels.newChannel(inputStream), bl);
    }

    public SnappyFramedInputStream(ReadableByteChannel readableByteChannel) throws IOException {
        this(readableByteChannel, true);
    }

    public SnappyFramedInputStream(ReadableByteChannel readableByteChannel, boolean bl) throws IOException {
        if (readableByteChannel == null) {
            throw new NullPointerException("in is null");
        }
        this.rbc = readableByteChannel;
        this.verifyChecksums = bl;
        this.allocateBuffersBasedOnSize(65541);
        this.frameHeader = ByteBuffer.allocate(4);
        byte[] byArray = SnappyFramed.HEADER_BYTES;
        byte[] byArray2 = new byte[byArray.length];
        ByteBuffer byteBuffer = ByteBuffer.wrap(byArray2);
        int n2 = SnappyFramed.readBytes(readableByteChannel, byteBuffer);
        if (n2 < byArray.length) {
            throw new EOFException("encountered EOF while reading stream header");
        }
        if (!Arrays.equals(byArray, byArray2)) {
            throw new IOException("invalid stream header");
        }
    }

    private void allocateBuffersBasedOnSize(int n2) {
        if (this.input != null) {
            SnappyFramed.releaseDirectByteBuffer(this.input);
        }
        if (this.uncompressedDirect != null) {
            SnappyFramed.releaseDirectByteBuffer(this.uncompressedDirect);
        }
        this.input = ByteBuffer.allocateDirect(n2);
        int n3 = Snappy.maxCompressedLength(n2);
        this.uncompressedDirect = ByteBuffer.allocateDirect(n3);
        this.buffer = new byte[n3];
    }

    @Override
    public int read() throws IOException {
        if (this.closed) {
            return -1;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        return this.buffer[this.position++] & 0xFF;
    }

    @Override
    public int read(byte[] byArray, int n2, int n3) throws IOException {
        if (byArray == null) {
            throw new IllegalArgumentException("output is null");
        }
        if (n2 < 0 || n3 < 0 || n2 + n3 > byArray.length) {
            throw new IllegalArgumentException("invalid offset [" + n2 + "] and length [" + n3 + ']');
        }
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (n3 == 0) {
            return 0;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        int n4 = Math.min(n3, this.available());
        System.arraycopy(this.buffer, this.position, byArray, n2, n4);
        this.position += n4;
        return n4;
    }

    @Override
    public int available() throws IOException {
        if (this.closed) {
            return 0;
        }
        return this.valid - this.position;
    }

    @Override
    public boolean isOpen() {
        return !this.closed;
    }

    @Override
    public int read(ByteBuffer byteBuffer) throws IOException {
        if (byteBuffer == null) {
            throw new IllegalArgumentException("dst is null");
        }
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (byteBuffer.remaining() == 0) {
            return 0;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        int n2 = Math.min(byteBuffer.remaining(), this.available());
        byteBuffer.put(this.buffer, this.position, n2);
        this.position += n2;
        return n2;
    }

    @Override
    public long transferTo(OutputStream outputStream) throws IOException {
        if (outputStream == null) {
            throw new IllegalArgumentException("os is null");
        }
        if (this.closed) {
            throw new ClosedChannelException();
        }
        long l2 = 0L;
        while (this.ensureBuffer()) {
            int n2 = this.available();
            outputStream.write(this.buffer, this.position, n2);
            this.position += n2;
            l2 += (long)n2;
        }
        return l2;
    }

    public long transferTo(WritableByteChannel writableByteChannel) throws IOException {
        if (writableByteChannel == null) {
            throw new IllegalArgumentException("wbc is null");
        }
        if (this.closed) {
            throw new ClosedChannelException();
        }
        ByteBuffer byteBuffer = ByteBuffer.wrap(this.buffer);
        long l2 = 0L;
        while (this.ensureBuffer()) {
            byteBuffer.clear();
            byteBuffer.position(this.position);
            byteBuffer.limit(this.position + this.available());
            writableByteChannel.write(byteBuffer);
            int n2 = byteBuffer.position() - this.position;
            this.position += n2;
            l2 += (long)n2;
        }
        return l2;
    }

    @Override
    public void close() throws IOException {
        try {
            this.rbc.close();
        }
        finally {
            if (!this.closed) {
                this.closed = true;
            }
            if (this.input != null) {
                SnappyFramed.releaseDirectByteBuffer(this.input);
            }
            if (this.uncompressedDirect != null) {
                SnappyFramed.releaseDirectByteBuffer(this.uncompressedDirect);
            }
        }
    }

    private boolean ensureBuffer() throws IOException {
        int n2;
        if (this.available() > 0) {
            return true;
        }
        if (this.eof) {
            return false;
        }
        if (!this.readBlockHeader()) {
            this.eof = true;
            return false;
        }
        FrameMetaData frameMetaData = this.getFrameMetaData(this.frameHeader);
        if (FrameAction.SKIP == frameMetaData.frameAction) {
            SnappyFramed.skip(this.rbc, frameMetaData.length, ByteBuffer.wrap(this.buffer));
            return this.ensureBuffer();
        }
        if (frameMetaData.length > this.input.capacity()) {
            this.allocateBuffersBasedOnSize(frameMetaData.length);
        }
        this.input.clear();
        this.input.limit(frameMetaData.length);
        int n3 = SnappyFramed.readBytes(this.rbc, this.input);
        if (n3 != frameMetaData.length) {
            throw new EOFException("unexpectd EOF when reading frame");
        }
        this.input.flip();
        FrameData frameData = this.getFrameData(this.input);
        if (FrameAction.UNCOMPRESS == frameMetaData.frameAction) {
            this.input.position(frameData.offset);
            n2 = Snappy.uncompressedLength(this.input);
            if (n2 > this.uncompressedDirect.capacity()) {
                this.uncompressedDirect = ByteBuffer.allocateDirect(n2);
                this.buffer = new byte[Math.max(this.input.capacity(), n2)];
            }
            this.uncompressedDirect.clear();
            this.valid = Snappy.uncompress(this.input, this.uncompressedDirect);
            this.uncompressedDirect.get(this.buffer, 0, this.valid);
            this.position = 0;
        } else {
            this.input.position(frameData.offset);
            this.position = 0;
            this.valid = this.input.remaining();
            this.input.get(this.buffer, 0, this.input.remaining());
        }
        if (this.verifyChecksums && frameData.checkSum != (n2 = SnappyFramed.maskedCrc32c(this.buffer, this.position, this.valid - this.position))) {
            throw new IOException("Corrupt input: invalid checksum");
        }
        return true;
    }

    private boolean readBlockHeader() throws IOException {
        this.frameHeader.clear();
        int n2 = SnappyFramed.readBytes(this.rbc, this.frameHeader);
        if (n2 == -1) {
            return false;
        }
        if (n2 < this.frameHeader.capacity()) {
            throw new EOFException("encountered EOF while reading block header");
        }
        this.frameHeader.flip();
        return true;
    }

    private FrameMetaData getFrameMetaData(ByteBuffer byteBuffer) throws IOException {
        FrameAction frameAction;
        assert (byteBuffer.hasArray());
        byte[] byArray = byteBuffer.array();
        int n2 = byArray[1] & 0xFF;
        n2 |= (byArray[2] & 0xFF) << 8;
        n2 |= (byArray[3] & 0xFF) << 16;
        int n3 = 0;
        int n4 = byArray[0] & 0xFF;
        switch (n4) {
            case 0: {
                frameAction = FrameAction.UNCOMPRESS;
                n3 = 5;
                break;
            }
            case 1: {
                frameAction = FrameAction.RAW;
                n3 = 5;
                break;
            }
            case 255: {
                if (n2 != 6) {
                    throw new IOException("stream identifier chunk with invalid length: " + n2);
                }
                frameAction = FrameAction.SKIP;
                n3 = 6;
                break;
            }
            default: {
                if (n4 <= 127) {
                    throw new IOException("unsupported unskippable chunk: " + Integer.toHexString(n4));
                }
                frameAction = FrameAction.SKIP;
                n3 = 0;
            }
        }
        if (n2 < n3) {
            throw new IOException("invalid length: " + n2 + " for chunk flag: " + Integer.toHexString(n4));
        }
        return new FrameMetaData(frameAction, n2);
    }

    private FrameData getFrameData(ByteBuffer byteBuffer) throws IOException {
        return new FrameData(this.getCrc32c(byteBuffer), 4);
    }

    private int getCrc32c(ByteBuffer byteBuffer) {
        int n2 = byteBuffer.position();
        return (byteBuffer.get(n2 + 3) & 0xFF) << 24 | (byteBuffer.get(n2 + 2) & 0xFF) << 16 | (byteBuffer.get(n2 + 1) & 0xFF) << 8 | byteBuffer.get(n2) & 0xFF;
    }

    public static final class FrameData {
        final int checkSum;
        final int offset;

        public FrameData(int n2, int n3) {
            this.checkSum = n2;
            this.offset = n3;
        }
    }

    public static final class FrameMetaData {
        final int length;
        final FrameAction frameAction;

        public FrameMetaData(FrameAction frameAction, int n2) {
            this.frameAction = frameAction;
            this.length = n2;
        }
    }

    static enum FrameAction {
        RAW,
        SKIP,
        UNCOMPRESS;

    }
}

