from __future__ import annotations

import typing

import matplotlib.axis
import numpy as np
import pandas as pd

from conflowgen.domain_models.data_types.mode_of_transport import ModeOfTransport
from conflowgen.analyses.inbound_and_outbound_vehicle_capacity_analysis import \
    InboundAndOutboundVehicleCapacityAnalysis
from conflowgen.reporting import AbstractReportWithMatplotlib


class InboundAndOutboundVehicleCapacityAnalysisReport(AbstractReportWithMatplotlib):
    """
    This analysis report takes the data structure as generated by :class:`.InboundAndOutboundVehicleCapacityAnalysis`
    and creates a comprehensible representation for the user, either as text or as a graph.
    """

    report_description = """
    Analyze the container volumes transported by vehicle type for the inbound and outbound journeys and check for the
    maximum capacity of each vehicle type on its outbound journey.
    If, e.g., for the vehicle type 'feeder' the maximum capacity is reached, most likely there are more vehicles that
    deliver containers destined for feeders but could not get a free spot.
    You might consider adding more vehicles of that type or adjusting the modal split.
    """

    def __init__(self):
        super().__init__()
        self._df = None
        self.analysis = InboundAndOutboundVehicleCapacityAnalysis(
            transportation_buffer=self.transportation_buffer
        )

    def get_report_as_text(self, **kwargs) -> str:
        """
        Keyword Args:
            start_date (datetime.datetime):
                Only include containers that arrive after the given start time.
            end_date (datetime.datetime):
                Only include containers that depart before the given end time.
            use_cache:
                Use internally cached values. Please set this to false if data are altered between analysis runs.

        Returns:
             The report in text format spanning over several lines.
        """
        inbound_capacities, outbound_actual_capacities, outbound_maximum_capacities = \
            self._get_container_volumes_in_teu(kwargs)

        assert len(kwargs) == 0, f"Keyword(s) {list(kwargs.keys())} have not been processed."

        # create string representation
        report = "\n(all numbers are reported in TEU)\n"
        report += "vehicle type      "
        report += "inbound volume      "
        report += "outbound volume     "
        report += "outbound max capacity"
        report += "\n"
        for vehicle_type in self.order_of_vehicle_types_in_report:
            vehicle_type_as_text = str(vehicle_type).replace("_", " ")
            max_capacities_repr = -1 if np.isnan(outbound_maximum_capacities[vehicle_type]) \
                else outbound_maximum_capacities[vehicle_type]
            report += f"{vehicle_type_as_text:<15} "
            report += f"{inbound_capacities[vehicle_type]:>16.1f} "
            report += f"{outbound_actual_capacities[vehicle_type]:>20.1f} "
            report += f"{max_capacities_repr:>25.1f}"
            report += "\n"
        report += "(rounding errors might exist)\n"
        return report

    def get_report_as_graph(self, **kwargs) -> matplotlib.axis.Axis:
        """
        The report as a graph is represented as a bar chart using pandas.

        Keyword Args:
            start_date (datetime.datetime):
                Only include containers that arrive after the given start time.
            end_date (datetime.datetime):
                Only include containers that depart before the given end time.
            use_cache:
                Use internally cached values. Please set this to false if data are altered between analysis runs.

        Returns:
             The matplotlib axis of the bar chart.
        """
        inbound_capacities, outbound_actual_capacities, outbound_maximum_capacities = \
            self._get_container_volumes_in_teu(kwargs)

        assert len(kwargs) == 0, f"Keyword(s) {list(kwargs.keys())} have not been processed."

        self._df = pd.DataFrame({
            "inbound volume (in TEU)": inbound_capacities,
            "outbound volume (in TEU)": outbound_actual_capacities,
            "outbound maximum capacity": outbound_maximum_capacities
        })
        self._df.index = [str(i).replace("_", " ") for i in self._df.index]
        ax = self._df.plot.barh()
        ax.set_xlabel("Capacity (in TEU)")
        ax.set_title("Inbound and outbound vehicle capacity analysis")
        return ax

    def _get_container_volumes_in_teu(
            self,
            kwargs
    ) -> typing.Tuple[typing.Dict[ModeOfTransport, float], typing.Dict[
         ModeOfTransport, float], typing.Dict[ModeOfTransport, float]]:

        assert self.transportation_buffer is not None
        self.analysis.update(
            transportation_buffer=self.transportation_buffer
        )
        start_date = kwargs.pop("start_date", None)
        end_date = kwargs.pop("end_date", None)
        use_cache = kwargs.pop("use_cache", True)

        # gather data
        inbound_container_volume = self.analysis.get_inbound_container_volumes_by_vehicle_type(
            start_date=start_date,
            end_date=end_date,
            use_cache=use_cache
        )
        outbound_container_volume, outbound_maximum_container_volume = \
            self.analysis.get_outbound_container_volume_by_vehicle_type(
                start_date=start_date,
                end_date=end_date,
                use_cache=use_cache
            )

        return inbound_container_volume.teu, outbound_container_volume.teu, outbound_maximum_container_volume.teu
