"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) &&
            !core_1.Token.isUnresolved((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) || ((_d = props.customDomain) === null || _d === void 0 ? void 0 : _d.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        const sdkCall = {
            service: 'CognitoIdentityServiceProvider',
            action: 'describeUserPoolDomain',
            parameters: {
                Domain: this.domainName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
        };
        const customResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
            resourceType: 'Custom::UserPoolCloudFrontDomainName',
            onCreate: sdkCall,
            onUpdate: sdkCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                // DescribeUserPoolDomain only supports access level '*'
                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                resources: ['*'],
            }),
        });
        return customResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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