"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const fields_1 = require("../fields");
const types_1 = require("../types");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Base class for all other state classes.
 *
 * @stability stable
 */
class State extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
        this.resultSelector = props.resultSelector;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree.
     *
     * @stability stable
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...constructs_1.Node.of(el).children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     *
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     *
     * @stability stable
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state.
     *
     * @stability stable
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states.
     *
     * @stability stable
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    /**
     * Descriptive identifier for this chainable.
     *
     * @stability stable
     */
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID.
     *
     * @stability stable
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state.
     *
     * @stability stable
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph.
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     *
     * @stability stable
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        var _b;
        validateErrors(props.errors);
        this.retries.push({
            ...props,
            errors: (_b = props.errors) !== null && _b !== void 0 ? _b : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        var _b;
        validateErrors(props.errors);
        this.catches.push({
            next: handler,
            props: {
                errors: (_b = props.errors) !== null && _b !== void 0 ? _b : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state.
     *
     * @stability stable
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state.
     *
     * @stability stable
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state.
     *
     * @stability stable
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state.
     *
     * @stability stable
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state.
     *
     * @stability stable
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format.
     *
     * @stability stable
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format.
     *
     * @stability stable
     */
    renderChoices() {
        var _b;
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: (_b = this.defaultChoice) === null || _b === void 0 ? void 0 : _b.stateId,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format.
     *
     * @stability stable
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format.
     *
     * @stability stable
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format.
     *
     * @stability stable
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format.
     *
     * @stability stable
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry, (a, b) => compareErrors(a.errors, b.errors)),
            Catch: renderList(this.catches, renderCatch, (a, b) => compareErrors(a.props.errors, b.props.errors)),
        };
    }
    /**
     * Render ResultSelector in ASL JSON format.
     *
     * @stability stable
     */
    renderResultSelector() {
        return fields_1.FieldUtils.renderObject({
            ResultSelector: this.resultSelector,
        });
    }
    /**
     * Called whenever this state is bound to a graph.
     *
     * Can be overridden by subclasses.
     *
     * @stability stable
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
_a = JSII_RTTI_SYMBOL_1;
State[_a] = { fqn: "@aws-cdk/aws-stepfunctions.State", version: "1.110.0" };
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Compares a list of Errors to move Errors.ALL last in a sort function
 */
function compareErrors(a, b) {
    if (a === null || a === void 0 ? void 0 : a.includes(types_1.Errors.ALL)) {
        return 1;
    }
    if (b === null || b === void 0 ? void 0 : b.includes(types_1.Errors.ALL)) {
        return -1;
    }
    return 0;
}
/**
 * Validates an errors list
 */
function validateErrors(errors) {
    if ((errors === null || errors === void 0 ? void 0 : errors.includes(types_1.Errors.ALL)) && errors.length > 1) {
        throw new Error(`${types_1.Errors.ALL} must appear alone in an error list`);
    }
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, mapFn, sortFn) {
    if (xs.length === 0) {
        return undefined;
    }
    let list = xs;
    if (sortFn) {
        list = xs.sort(sortFn);
    }
    return list.map(mapFn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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