# Copyright UL Research Institutes
# SPDX-License-Identifier: Apache-2.0

# mypy: disable-error-code="import-untyped"
import ast
from typing import Iterable, Optional

import pyarrow
import pyarrow.dataset
import pydantic

from dyff.audit.scoring.base import Rubric
from dyff.schema.base import int32
from dyff.schema.dataset import ReplicatedItem
from dyff.schema.dataset.arrow import arrow_schema, schema_function


def is_valid_python_syntax(code):
    try:
        assert code is not None
        tree = ast.parse(code)
        assert tree is not None
    except:
        return 1
    return 0


class GetValidSyntaxScoreItem(ReplicatedItem):
    language: str = pydantic.Field(description="The target programming language")
    valid: Optional[int32(ge=0, le=1)] = pydantic.Field(  # type: ignore
        default=None,
        description="Valid syntax score for a code generated by the LLM",
    )


class GetValidSyntaxScore(Rubric):
    """
    Get the syntax validation score for code generated by an LLM.
    Return 0 for invalid syntax and 1 for valid syntax.
    """

    @property
    def name(self) -> str:
        return "evaluator_code_rubric.GetValidSyntaxScore"

    @property
    @schema_function(arrow_schema(GetValidSyntaxScoreItem))
    def schema(self) -> pyarrow.Schema:
        pass

    def apply(
        self, input_data: pyarrow.dataset.Dataset, output_data: pyarrow.dataset.Dataset
    ) -> Iterable[pyarrow.RecordBatch]:
        input_items = {}
        for input_batch in input_data.to_batches(
            columns=["_index_", "language", "text"]
        ):
            for item in input_batch.to_pylist():
                input_items[item["_index_"]] = item

        for output_batch in output_data.to_batches(
            columns=["_index_", "_replication_", "responses"]
        ):
            batch = []
            for item in output_batch.to_pylist():
                index = item["_index_"]
                replication = item["_replication_"]
                language = input_items[index]["language"]
                input_text = input_items[index]["text"]
                output_text = item["responses"][0]["text"]
                output_text = output_text.replace(input_text, "")
                try:
                    output_code = str(output_text).split("</code>")[0]
                    is_valid = None
                    if language == "Python":
                        is_valid = is_valid_python_syntax(output_code)
                    output = GetValidSyntaxScoreItem(
                        _index_=index,
                        _replication_=replication,
                        language=language,
                        valid=is_valid,
                    ).dict()
                except:
                    output = GetValidSyntaxScoreItem(
                        _index_=index,
                        _replication_=replication,
                        language=language,
                        valid=None,
                    ).dict()
                batch.append(output)

            yield pyarrow.RecordBatch.from_pylist(batch, schema=self.schema)
