"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
/**
* An EC2 Key Pair
*/
class KeyPair extends cdk.Construct {
    /**
    * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
    */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
        * ARN of the private key in AWS Secrets Manager
        */
        this.arn = '';
        /**
        * Name of the Key Pair
        */
        this.name = '';
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const tags = props.tags || {};
        tags.CreatedBy = ID;
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                name: props.name,
                description: props.description || '',
                keyLength: props.keyLength || KeyLength.L2048,
                kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                StackName: stack,
                tags: tags,
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(fn.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(fn.role);
        }
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new iam.PolicyStatement({
                    actions: [
                        'ec2:CreateKeyPair',
                        'ec2:DeleteKeyPair',
                        'ec2:DescribeKeyPairs',
                        'secretsmanager:ListSecrets',
                    ],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:TagResource',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:RequestTag/CreatedBy': ID,
                        }
                    },
                }),
                new iam.PolicyStatement({
                    actions: [
                        'secretsmanager:DeleteResourcePolicy',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:DescribeSecret',
                        'secretsmanager:GetResourcePolicy',
                        'secretsmanager:ListSecretVersionIds',
                        'secretsmanager:PutResourcePolicy',
                        'secretsmanager:PutSecretValue',
                        'secretsmanager:RestoreSecret',
                        'secretsmanager:UntagResource',
                        'secretsmanager:UpdateSecret',
                        'secretsmanager:UpdateSecretVersionStage',
                    ],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/CreatedBy': ID,
                        }
                    },
                }),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ]
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout)
        });
        return fn;
    }
    /**
    * Grants read access to the private key in the AWS Secrets Manager
    */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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