"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const aws_auth_1 = require("../lib/aws-auth");
const util_1 = require("./util");
/* eslint-disable max-len */
const CLUSTER_VERSION = lib_1.KubernetesVersion.V1_16;
module.exports = {
    'throws when adding a role from a different stack'(test) {
        const app = new cdk.App();
        const clusterStack = new cdk.Stack(app, 'ClusterStack');
        const roleStack = new cdk.Stack(app, 'RoleStack');
        const awsAuth = new aws_auth_1.AwsAuth(clusterStack, 'Auth', {
            cluster: new lib_1.Cluster(clusterStack, 'Cluster', { version: lib_1.KubernetesVersion.V1_17 }),
        });
        const role = new iam.Role(roleStack, 'Role', { assumedBy: new iam.AnyPrincipal() });
        try {
            awsAuth.addRoleMapping(role, { groups: ['group'] });
            test.ok(false, 'expected error');
        }
        catch (err) {
            test.equal(err.message, 'RoleStackRole6729D0A7 should be defined in the scope of the ClusterStack stack to prevent circular dependencies');
        }
        test.done();
    },
    'throws when adding a user from a different stack'(test) {
        const app = new cdk.App();
        const clusterStack = new cdk.Stack(app, 'ClusterStack');
        const userStack = new cdk.Stack(app, 'UserStack');
        const awsAuth = new aws_auth_1.AwsAuth(clusterStack, 'Auth', {
            cluster: new lib_1.Cluster(clusterStack, 'Cluster', { version: lib_1.KubernetesVersion.V1_17 }),
        });
        const user = new iam.User(userStack, 'User');
        try {
            awsAuth.addUserMapping(user, { groups: ['group'] });
            test.ok(false, 'expected error');
        }
        catch (err) {
            test.equal(err.message, 'UserStackUser0406F94E should be defined in the scope of the ClusterStack stack to prevent circular dependencies');
        }
        test.done();
    },
    'empty aws-auth'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
        // WHEN
        new aws_auth_1.AwsAuth(stack, 'AwsAuth', { cluster });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: JSON.stringify([{
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: { name: 'aws-auth', namespace: 'kube-system' },
                    data: { mapRoles: '[]', mapUsers: '[]', mapAccounts: '[]' },
                }]),
        }));
        test.done();
    },
    'addRoleMapping and addUserMapping can be used to define the aws-auth ConfigMap'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        const user = new iam.User(stack, 'user');
        // WHEN
        cluster.awsAuth.addRoleMapping(role, { groups: ['role-group1'], username: 'roleuser' });
        cluster.awsAuth.addRoleMapping(role, { groups: ['role-group2', 'role-group3'] });
        cluster.awsAuth.addUserMapping(user, { groups: ['user-group1', 'user-group2'] });
        cluster.awsAuth.addUserMapping(user, { groups: ['user-group1', 'user-group2'], username: 'foo' });
        cluster.awsAuth.addAccount('112233');
        cluster.awsAuth.addAccount('5566776655');
        // THEN
        assert_1.expect(stack).to(assert_1.countResources(lib_1.KubernetesManifest.RESOURCE_TYPE, 1));
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterNodegroupDefaultCapacityNodeGroupRole55953B04',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"roleuser\\",\\"groups\\":[\\"role-group1\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"role-group2\\",\\"role-group3\\"]}]","mapUsers":"[{\\"userarn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'user2C2B57AE',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'user2C2B57AE',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"user-group1\\",\\"user-group2\\"]},{\\"userarn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'user2C2B57AE',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"foo\\",\\"groups\\":[\\"user-group1\\",\\"user-group2\\"]}]","mapAccounts":"[\\"112233\\",\\"5566776655\\"]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'imported users and roles can be also be used'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
        const role = iam.Role.fromRoleArn(stack, 'imported-role', 'arn:aws:iam::123456789012:role/S3Access');
        const user = iam.User.fromUserName(stack, 'import-user', 'MyUserName');
        // WHEN
        cluster.awsAuth.addRoleMapping(role, { groups: ['group1'] });
        cluster.awsAuth.addUserMapping(user, { groups: ['group2'] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterNodegroupDefaultCapacityNodeGroupRole55953B04',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]},{\\"rolearn\\":\\"arn:aws:iam::123456789012:role/S3Access\\",\\"username\\":\\"arn:aws:iam::123456789012:role/S3Access\\",\\"groups\\":[\\"group1\\"]}]","mapUsers":"[{\\"userarn\\":\\"arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::',
                        {
                            Ref: 'AWS::AccountId',
                        },
                        ':user/MyUserName\\",\\"username\\":\\"arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::',
                        {
                            Ref: 'AWS::AccountId',
                        },
                        ':user/MyUserName\\",\\"groups\\":[\\"group2\\"]}]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addMastersRole after addNodegroup correctly'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        const cluster = new lib_1.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
        cluster.addNodegroup('NG');
        const role = iam.Role.fromRoleArn(stack, 'imported-role', 'arn:aws:iam::123456789012:role/S3Access');
        // WHEN
        cluster.awsAuth.addMastersRole(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(lib_1.KubernetesManifest.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterMastersRole9AA35625',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterNodegroupDefaultCapacityNodeGroupRole55953B04',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]},{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterNodegroupNGNodeGroupRole7C078920',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]},{\\"rolearn\\":\\"arn:aws:iam::123456789012:role/S3Access\\",\\"username\\":\\"arn:aws:iam::123456789012:role/S3Access\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
            ClusterName: {
                Ref: 'Cluster9EE0221C',
            },
            RoleArn: {
                'Fn::GetAtt': [
                    'ClusterCreationRole360249B6',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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