"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ pragma:ignore-assets
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const pinger_1 = require("./pinger/pinger");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        const secretsEncryptionKey = new kms.Key(this, 'SecretsKey');
        // just need one nat gateway to simplify the test
        this.vpc = new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // create the cluster with a default nodegroup capacity
        this.cluster = new eks.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            mastersRole,
            defaultCapacity: 2,
            version: eks.KubernetesVersion.V1_16,
            secretsEncryptionKey,
        });
        this.assertFargateProfile();
        this.assertCapacity();
        this.assertBottlerocket();
        this.assertSpotCapacity();
        this.assertInferenceInstances();
        this.assertNodeGroup();
        this.assertSimpleManifest();
        this.assertSimpleHelmChart();
        this.assertCreateNamespace();
        this.assertServiceAccount();
        this.assertServiceLoadBalancerAddress();
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: this.cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: this.cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: this.cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterSecurityGroupId', { value: this.cluster.clusterSecurityGroupId });
        new core_1.CfnOutput(this, 'ClusterEncryptionConfigKeyArn', { value: this.cluster.clusterEncryptionConfigKeyArn });
        new core_1.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
    }
    assertServiceAccount() {
        // add a service account connected to a IAM role
        this.cluster.addServiceAccount('MyServiceAccount');
    }
    assertCreateNamespace() {
        // deploy an nginx ingress in a namespace
        const nginxNamespace = this.cluster.addManifest('nginx-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'nginx',
            },
        });
        const nginxIngress = this.cluster.addChart('nginx-ingress', {
            chart: 'nginx-ingress',
            repository: 'https://helm.nginx.com/stable',
            namespace: 'nginx',
            wait: true,
            createNamespace: false,
            timeout: core_1.Duration.minutes(15),
        });
        // make sure namespace is deployed before the chart
        nginxIngress.node.addDependency(nginxNamespace);
    }
    assertSimpleHelmChart() {
        // deploy the Kubernetes dashboard through a helm chart
        this.cluster.addChart('dashboard', {
            chart: 'kubernetes-dashboard',
            repository: 'https://kubernetes.github.io/dashboard/',
        });
    }
    assertSimpleManifest() {
        // apply a kubernetes manifest
        this.cluster.addManifest('HelloApp', ...hello.resources);
    }
    assertNodeGroup() {
        // add a extra nodegroup
        this.cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: this.cluster.defaultCapacity ? this.cluster.defaultCapacity.role : undefined,
        });
    }
    assertInferenceInstances() {
        // inference instances
        this.cluster.addCapacity('InferenceInstances', {
            instanceType: new ec2.InstanceType('inf1.2xlarge'),
            minCapacity: 1,
        });
    }
    assertSpotCapacity() {
        // spot instances (up to 10)
        this.cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
    }
    assertBottlerocket() {
        // add bottlerocket nodes
        this.cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
    }
    assertCapacity() {
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        this.cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
    }
    assertFargateProfile() {
        // fargate profile for resources in the "default" namespace
        this.cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
    }
    assertServiceLoadBalancerAddress() {
        const serviceName = 'webservice';
        const labels = { app: 'simple-web' };
        const containerPort = 80;
        const servicePort = 9000;
        const pingerSecurityGroup = new ec2.SecurityGroup(this, 'WebServiceSecurityGroup', {
            vpc: this.vpc,
        });
        pingerSecurityGroup.addIngressRule(pingerSecurityGroup, ec2.Port.tcp(servicePort), `allow http ${servicePort} access from myself`);
        this.cluster.addManifest('simple-web-pod', {
            kind: 'Pod',
            apiVersion: 'v1',
            metadata: { name: 'webpod', labels: labels },
            spec: {
                containers: [{
                        name: 'simplewebcontainer',
                        image: 'nginx',
                        ports: [{ containerPort: containerPort }],
                    }],
            },
        });
        this.cluster.addManifest('simple-web-service', {
            kind: 'Service',
            apiVersion: 'v1',
            metadata: {
                name: serviceName,
                annotations: {
                    // this is furtile soil for cdk8s-plus! :)
                    'service.beta.kubernetes.io/aws-load-balancer-internal': 'true',
                    'service.beta.kubernetes.io/aws-load-balancer-extra-security-groups': pingerSecurityGroup.securityGroupId,
                },
            },
            spec: {
                type: 'LoadBalancer',
                ports: [{ port: servicePort, targetPort: containerPort }],
                selector: labels,
            },
        });
        const loadBalancerAddress = this.cluster.getServiceLoadBalancerAddress(serviceName);
        // create a resource that hits the load balancer to make sure
        // everything is wired properly.
        const pinger = new pinger_1.Pinger(this, 'ServicePinger', {
            url: `http://${loadBalancerAddress}:${servicePort}`,
            securityGroup: pingerSecurityGroup,
            vpc: this.vpc,
        });
        // this should display a proper nginx response
        // <title>Welcome to nginx!</title>...
        new core_1.CfnOutput(this, 'Response', {
            value: pinger.response,
        });
    }
}
// this test uses both the bottlerocket image and the inf1 instance, which are only supported in these
// regions. see https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-eks#bottlerocket
// and https://aws.amazon.com/about-aws/whats-new/2019/12/introducing-amazon-ec2-inf1-instances-high-performance-and-the-lowest-cost-machine-learning-inference-in-the-cloud/
const supportedRegions = [
    'us-east-1',
    'us-west-2',
];
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
const stack = new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
if (process.env.CDK_INTEG_ACCOUNT !== '12345678') {
    // only validate if we are about to actually deploy.
    // TODO: better way to determine this, right now the 'CDK_INTEG_ACCOUNT' seems like the only way.
    if (core_1.Token.isUnresolved(stack.region)) {
        throw new Error(`region (${stack.region}) cannot be a token and must be configured to one of: ${supportedRegions}`);
    }
    if (!supportedRegions.includes(stack.region)) {
        throw new Error(`region (${stack.region}) must be configured to one of: ${supportedRegions}`);
    }
}
app.synth();
//# sourceMappingURL=data:application/json;base64,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