import * as autoscaling from '@aws-cdk/aws-autoscaling';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { Construct, Resource } from '@aws-cdk/core';
import { ICluster, ClusterAttributes, DefaultCapacityType, CapacityOptions, AutoScalingGroupOptions, CommonClusterOptions } from './cluster';
import { HelmChartOptions, HelmChart } from './helm-chart';
import { KubernetesManifest } from './k8s-manifest';
import { Nodegroup, NodegroupOptions } from './managed-nodegroup';
/**
 * Common configuration props for EKS clusters.
 */
export interface LegacyClusterProps extends CommonClusterOptions {
    /**
     * Number of instances to allocate as an initial capacity for this cluster.
     * Instance type can be configured through `defaultCapacityInstanceType`,
     * which defaults to `m5.large`.
     *
     * Use `cluster.addCapacity` to add additional customized capacity. Set this
     * to `0` is you wish to avoid the initial capacity allocation.
     *
     * @default 2
     */
    readonly defaultCapacity?: number;
    /**
     * The instance type to use for the default capacity. This will only be taken
     * into account if `defaultCapacity` is > 0.
     *
     * @default m5.large
     */
    readonly defaultCapacityInstance?: ec2.InstanceType;
    /**
     * The default capacity type for the cluster.
     *
     * @default NODEGROUP
     */
    readonly defaultCapacityType?: DefaultCapacityType;
    /**
     * KMS secret for envelope encryption for Kubernetes secrets.
     *
     * @default - By default, Kubernetes stores all secret object data within etcd and
     *            all etcd volumes used by Amazon EKS are encrypted at the disk-level
     *            using AWS-Managed encryption keys.
     */
    readonly secretsEncryptionKey?: kms.IKey;
}
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 *
 * @resource AWS::EKS::Cluster
 */
export declare class LegacyCluster extends Resource implements ICluster {
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope: Construct, id: string, attrs: ClusterAttributes): ICluster;
    /**
     * The VPC in which this Cluster was created
     */
    readonly vpc: ec2.IVpc;
    /**
     * The Name of the created EKS Cluster
     */
    readonly clusterName: string;
    /**
     * The AWS generated ARN for the Cluster resource
     *
     * @example arn:aws:eks:us-west-2:666666666666:cluster/prod
     */
    readonly clusterArn: string;
    /**
     * The endpoint URL for the Cluster
     *
     * This is the URL inside the kubeconfig file to use with kubectl
     *
     * @example https://5E1D0CEXAMPLEA591B746AFC5AB30262.yl4.us-west-2.eks.amazonaws.com
     */
    readonly clusterEndpoint: string;
    /**
     * The certificate-authority-data for your cluster.
     */
    readonly clusterCertificateAuthorityData: string;
    /**
     * The cluster security group that was created by Amazon EKS for the cluster.
     */
    readonly clusterSecurityGroupId: string;
    /**
     * Amazon Resource Name (ARN) or alias of the customer master key (CMK).
     */
    readonly clusterEncryptionConfigKeyArn: string;
    /**
     * Manages connection rules (Security Group Rules) for the cluster
     *
     * @type {ec2.Connections}
     * @memberof Cluster
     */
    readonly connections: ec2.Connections;
    /**
     * IAM role assumed by the EKS Control Plane
     */
    readonly role: iam.IRole;
    /**
     * The auto scaling group that hosts the default capacity for this cluster.
     * This will be `undefined` if the `defaultCapacityType` is not `EC2` or
     * `defaultCapacityType` is `EC2` but default capacity is set to 0.
     */
    readonly defaultCapacity?: autoscaling.AutoScalingGroup;
    /**
     * The node group that hosts the default capacity for this cluster.
     * This will be `undefined` if the `defaultCapacityType` is `EC2` or
     * `defaultCapacityType` is `NODEGROUP` but default capacity is set to 0.
     */
    readonly defaultNodegroup?: Nodegroup;
    private readonly version;
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope: Construct, id: string, props: LegacyClusterProps);
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     */
    addCapacity(id: string, options: CapacityOptions): autoscaling.AutoScalingGroup;
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroup(id: string, options?: NodegroupOptions): Nodegroup;
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup: autoscaling.AutoScalingGroup, options: AutoScalingGroupOptions): void;
    addManifest(_id: string, ..._manifest: any[]): KubernetesManifest;
    addChart(_id: string, _options: HelmChartOptions): HelmChart;
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    private tagSubnets;
}
