"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: props.cluster.adminRole,
        });
        this.podExecutionRole = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster.adminRole);
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster.adminRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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