import typing

from .client import Client, Response
from .helpers.mail import Mail


class BaseInterface:
    def __init__(self, auth: str, host: str) -> None:
        """
        Construct the Smtp.com v4 API object.

        :param auth: the authorization header
        :type auth: string
        :param host: base URL for API calls
        :type host: string
        """
        from . import __version__

        self.auth = auth
        self.host = host
        self.version = __version__
        self.useragent = "smtp.com/{};python".format(self.version)

        self.client = Client(
            host=self.host, request_headers=self._default_headers, version=4
        )

    @property
    def _default_headers(self) -> typing.Dict:
        headers = {
            "Authorization": self.auth,
            "User-Agent": self.useragent,
            "Accept": "application/json",
        }

        return headers

    def reset_request_headers(self) -> None:
        self.client.request_headers = self._default_headers

    def send(
        self,
        message: typing.Union[Mail, dict],
        request_headers: typing.Optional[dict] = None,
    ) -> Response:
        """Make a request with the request body generated by
           the Mail object

        :param message: request body generated by the Mail object or dict
        :type message: Mail
        :param request_headers: optional headers for request
        :type request_headers: dict
        """
        if not isinstance(message, dict):
            message = message.get()
        if "mime" in message:  # type: ignore
            return self.client.messages.mime.post(
                request_body=message, request_headers=request_headers
            )
        return self.client.messages.post(
            request_body=message, request_headers=request_headers
        )
