"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomCloud9Ssm = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloud9 = require("aws-cdk-lib/aws-cloud9");
const ssm = require("aws-cdk-lib/aws-ssm");
const iam = require("aws-cdk-lib/aws-iam");
const yaml = require('yaml');
const fs = require('fs');
/**
 * @experimental
 */
class CustomCloud9Ssm extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        let cloud9Env;
        let ssmAssociation;
        let customResource;
        // Create the Cloud9 environment using the default configuration
        if (!props.cloud9Ec2Props) {
            cloud9Env = new cloud9.CfnEnvironmentEC2(this, 'Cloud9Ec2Environment', { instanceType: "t3.large" });
        }
        // Create the Cloud9 environment using the received props
        else {
            cloud9Env = new cloud9.CfnEnvironmentEC2(this, 'Cloud9Ec2Environment', props.cloud9Ec2Props);
        }
        // Add a unique tag to the environment to use it as a target for the SSM Association
        aws_cdk_lib_1.Tags.of(cloud9Env).add('stack-id', aws_cdk_lib_1.Stack.of(this).stackId);
        // Create a Role for the EC2 instance and an instance profile with it
        this.ec2Role = new iam.Role(this, 'Ec2Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            roleName: id + '-CustomCloud9SsmEc2Role',
            managedPolicies: [
                iam.ManagedPolicy.fromManagedPolicyArn(this, 'SsmManagedPolicy', 'arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore')
            ]
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'Ec2InstanceProfile', {
            roles: [this.ec2Role.roleName]
        });
        // Create the SSM Document using the default configuration
        if (!props.ssmDocumentProps) {
            let content = fs.readFileSync(CustomCloud9Ssm.DEFAULT_DOCUMENT_FILE_NAME, 'utf8');
            const ssmDocumentProps = {
                documentType: 'Command',
                content: yaml.parse(content),
                name: id + '-' + CustomCloud9Ssm.DEFAULT_DOCUMENT_NAME
            };
            this.document = new ssm.CfnDocument(this, 'SsmDocument', ssmDocumentProps);
            this.resizeEBSTo(CustomCloud9Ssm.DEFAULT_EBS_SIZE);
        }
        // Create the SSM Document using the received props
        else {
            if (!props.ssmDocumentProps.name) {
                throw new Error("The document name must be specified.");
            }
            this.document = new ssm.CfnDocument(this, 'SsmDocument', props.ssmDocumentProps);
        }
        // Create an SSM Association to apply the document configuration
        ssmAssociation = new ssm.CfnAssociation(this, 'SsmAssociation', {
            name: this.document.name,
            targets: [
                {
                    key: 'tag:stack-id',
                    values: [aws_cdk_lib_1.Stack.of(this).stackId]
                }
            ]
        });
        // Create the Lambda function that attaches the instance profile to the EC2 instance
        let code = fs.readFileSync(CustomCloud9Ssm.ATTACH_PROFILE_FILE_NAME, 'utf8');
        const lambdaFunction = new lambda.Function(this, 'ProfileAttachLambdaFunction', {
            runtime: lambda.Runtime.PYTHON_3_6,
            code: lambda.Code.fromInline(code),
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(800)
        });
        // Give permissions to the function to execute some APIs
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeInstances',
                'ec2:AssociateIamInstanceProfile',
                'ec2:ReplaceIamInstanceProfileAssociation',
                'ec2:RebootInstances',
                'iam:ListInstanceProfiles',
                'iam:PassRole',
                'ssm:DescribeAssociationExecutions',
                'ssm:DescribeAssociationExecutionTargets'
            ],
            resources: ['*']
        }));
        // Create the Custom Resource that invokes the Lambda function
        customResource = new aws_cdk_lib_1.CustomResource(this, 'CustomResource', {
            serviceToken: lambdaFunction.functionArn,
            properties: {
                stack_id: aws_cdk_lib_1.Stack.of(this).stackId,
                profile_arn: instanceProfile.attrArn,
                association_id: ssmAssociation.attrAssociationId
            }
        });
        // Add resource dependencies
        instanceProfile.node.addDependency(this.ec2Role);
        ssmAssociation.node.addDependency(cloud9Env);
        ssmAssociation.node.addDependency(this.document);
        customResource.node.addDependency(instanceProfile);
        customResource.node.addDependency(ssmAssociation);
    }
    /**
     * (experimental) Adds one or more steps to the content of the SSM Document.
     *
     * @param steps : YAML formatted string containing one or more steps to be added to the mainSteps section of the SSM Document.
     * @experimental
     */
    addDocumentSteps(steps) {
        // Add the mainSteps section if it doesn't exist
        if (!('mainSteps' in this.document.content)) {
            this.document.content['mainSteps'] = [];
        }
        // Add the new step
        this.document.content['mainSteps'].push(...yaml.parse(steps));
    }
    /**
     * (experimental) Adds one or more parameters to the content of the SSM Document.
     *
     * @param parameters : YAML formatted string containing one or more parameters to be added to the parameters section of the SSM Document.
     * @experimental
     */
    addDocumentParameters(parameters) {
        // Add the parameters section if it doesn't exist
        if (!('parameters' in this.document.content)) {
            this.document.content['parameters'] = {};
        }
        // Add the new parameter
        this.document.content['parameters'] = Object.assign({}, this.document.content['parameters'], yaml.parse(parameters));
    }
    /**
     * (experimental) Adds a step to the SSM Document content that resizes the EBS volume of the EC2 instance.
     *
     * Attaches the required policies to ec2Role.
     *
     * @param size : size in GiB to resize the EBS volume to.
     * @experimental
     */
    resizeEBSTo(size) {
        let steps = fs.readFileSync(CustomCloud9Ssm.RESIZE_STEP_FILE_NAME, 'utf8');
        steps = steps.replace('{{ size }}', String(size));
        // Add the resizing step
        this.addDocumentSteps(steps);
        // Grant permission to the EC2 instance to execute the statements included in the SSM Document
        this.ec2Role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeInstances',
                'ec2:ModifyVolume',
                'ec2:DescribeVolumesModifications'
            ],
            resources: ['*']
        }));
    }
}
exports.CustomCloud9Ssm = CustomCloud9Ssm;
_a = JSII_RTTI_SYMBOL_1;
CustomCloud9Ssm[_a] = { fqn: "@cdk-use-cases/custom-cloud9-ssm.CustomCloud9Ssm", version: "1.2.0" };
CustomCloud9Ssm.DEFAULT_EBS_SIZE = 100;
CustomCloud9Ssm.DEFAULT_DOCUMENT_FILE_NAME = `${__dirname}/assets/default_document.yml`;
CustomCloud9Ssm.RESIZE_STEP_FILE_NAME = `${__dirname}/assets/resize_ebs_step.yml`;
CustomCloud9Ssm.ATTACH_PROFILE_FILE_NAME = `${__dirname}/assets/profile_attach.py`;
CustomCloud9Ssm.DEFAULT_DOCUMENT_NAME = 'CustomCloudSsm-SsmDocument';
//# sourceMappingURL=data:application/json;base64,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