"""
Tests for structured data models in CAT Cafe SDK.

These tests verify the data model functionality, serialization,
and object creation without requiring server infrastructure.
"""

from cat_cafe.sdk import DatasetImport, DatasetExample, Dataset, Example, Experiment, ExperimentResult, DatasetConfig


class TestDatasetExample:
    """Test DatasetExample data model"""

    def test_create_minimal_dataset_example(self):
        """Test creating a minimal DatasetExample"""
        example = DatasetExample(
            input={"messages": [{"role": "user", "content": "Hello"}]},
            output={"messages": [{"role": "assistant", "content": "Hi there!"}]},
        )

        assert example.input == {"messages": [{"role": "user", "content": "Hello"}]}
        assert example.output == {"messages": [{"role": "assistant", "content": "Hi there!"}]}
        assert example.tags == []  # Default empty list
        assert example.metadata == {}  # Default empty dict
        assert example.source_trace_id is None
        assert example.source_node_id is None

    def test_create_full_dataset_example(self):
        """Test creating a DatasetExample with all fields"""
        example = DatasetExample(
            input={"messages": [{"role": "user", "content": "Test input"}]},
            output={"messages": [{"role": "assistant", "content": "Test output"}]},
            metadata={"category": "test", "difficulty": "easy"},
        )
        example.tags = ["test", "example"]
        example.source_trace_id = "trace-123"
        example.source_node_id = "node-456"

        assert example.tags == ["test", "example"]
        assert example.metadata["category"] == "test"
        assert example.metadata["difficulty"] == "easy"
        assert example.metadata["tags"] == ["test", "example"]
        assert example.metadata["source_trace_id"] == "trace-123"
        assert example.metadata["source_node_id"] == "node-456"
        assert example.source_trace_id == "trace-123"
        assert example.source_node_id == "node-456"


class TestDatasetImport:
    """Test DatasetImport data model"""

    def test_create_dataset_import_with_examples(self):
        """Test creating DatasetImport with examples"""
        examples = [
            DatasetExample(
                input={"messages": [{"role": "user", "content": "Hello"}]},
                output={"messages": [{"role": "assistant", "content": "Hi!"}]},
                metadata={"tags": ["greeting"]},
            ),
            DatasetExample(
                input={"messages": [{"role": "user", "content": "Goodbye"}]},
                output={"messages": [{"role": "assistant", "content": "Bye!"}]},
                metadata={"tags": ["farewell"]},
            ),
        ]

        dataset_import = DatasetImport(
            name="Test Dataset",
            description="A test dataset",
            tags=["test", "import"],
            metadata={"version": "1.0"},
            examples=examples,
        )

        assert dataset_import.name == "Test Dataset"
        assert dataset_import.description == "A test dataset"
        assert dataset_import.tags == ["test", "import"]
        assert dataset_import.metadata == {"version": "1.0"}
        assert len(dataset_import.examples) == 2

        # Verify examples are preserved correctly
        assert all(isinstance(ex, DatasetExample) for ex in dataset_import.examples)
        assert dataset_import.examples[0].tags == ["greeting"]
        assert dataset_import.examples[1].tags == ["farewell"]


class TestExample:
    """Test Example data model (API response object)"""

    def test_create_example_from_api(self):
        """Test creating Example object as if from API response"""
        example = Example(
            id="example-123",
            input={"messages": [{"role": "user", "content": "Test question"}]},
            output={"messages": [{"role": "assistant", "content": "Test answer"}]},
            metadata={"api_version": "1.0"},
            created_at="2024-01-01T00:00:00Z",
            updated_at="2024-01-01T01:00:00Z",
        )
        example.source_trace_id = "trace-abc"
        example.source_node_id = "node-def"
        example.tags = ["api", "test"]

        assert example.id == "example-123"
        assert example.input == {"messages": [{"role": "user", "content": "Test question"}]}
        assert example.output == {"messages": [{"role": "assistant", "content": "Test answer"}]}
        assert example.source_trace_id == "trace-abc"
        assert example.source_node_id == "node-def"
        assert example.metadata == {
            "api_version": "1.0",
            "tags": ["api", "test"],
            "source_trace_id": "trace-abc",
            "source_node_id": "node-def",
        }
        assert example.tags == ["api", "test"]
        assert example.created_at == "2024-01-01T00:00:00Z"
        assert example.updated_at == "2024-01-01T01:00:00Z"


class TestDataset:
    """Test Dataset data model"""

    def test_create_dataset_with_examples(self):
        """Test creating Dataset with Example objects"""
        examples = [
            Example(
                id="ex-1",
                input={"messages": [{"role": "user", "content": "Q1"}]},
                output={"messages": [{"role": "assistant", "content": "A1"}]},
                metadata={"tags": ["q1"]},
            ),
            Example(
                id="ex-2",
                input={"messages": [{"role": "user", "content": "Q2"}]},
                output={"messages": [{"role": "assistant", "content": "A2"}]},
                metadata={"tags": ["q2"]},
            ),
        ]

        dataset = Dataset(
            id="dataset-456",
            name="Test Dataset",
            description="Dataset with examples",
            tags=["test"],
            metadata={"type": "test_dataset"},
            example_count=2,
            version=1,
            created_at="2024-01-01T00:00:00Z",
            updated_at="2024-01-01T00:00:00Z",
            examples=examples,
        )

        assert dataset.id == "dataset-456"
        assert dataset.name == "Test Dataset"
        assert dataset.example_count == 2
        assert dataset.version == 1
        assert len(dataset.examples) == 2
        assert all(isinstance(ex, Example) for ex in dataset.examples)
        assert dataset.examples[0].id == "ex-1"
        assert dataset.examples[1].id == "ex-2"


class TestExperiment:
    """Test Experiment data model"""

    def test_create_minimal_experiment(self):
        """Test creating minimal Experiment configuration"""
        experiment = Experiment(
            name="Test Experiment", description="Testing experiment creation", dataset_id="dataset-123"
        )

        assert experiment.name == "Test Experiment"
        assert experiment.description == "Testing experiment creation"
        assert experiment.dataset_id == "dataset-123"
        assert experiment.dataset_version is None
        assert experiment.tags == []
        assert experiment.metadata == {}

    def test_create_full_experiment(self):
        """Test creating Experiment with all fields"""
        experiment = Experiment(
            name="Full Test Experiment",
            description="Complete experiment configuration",
            dataset_id="dataset-456",
            dataset_version="v2",
            tags=["full", "test", "experiment"],
            metadata={"priority": "high", "timeout": 3600},
        )

        assert experiment.dataset_version == "v2"
        assert experiment.tags == ["full", "test", "experiment"]
        assert experiment.metadata == {"priority": "high", "timeout": 3600}


class TestExperimentResult:
    """Test ExperimentResult data model"""

    def test_create_successful_result(self):
        """Test creating successful ExperimentResult"""
        result = ExperimentResult(
            example_id="example-789",
            input_data={"messages": [{"role": "user", "content": "Test input"}]},
            output={"messages": [{"role": "assistant", "content": "Expected response"}]},
            actual_output={"messages": [{"role": "assistant", "content": "Actual response"}]},
            evaluation_scores={"accuracy": 0.95, "relevance": 0.88},
            metadata={"processing_time": 1.23, "model": "test-model"},
            error=None,
        )

        assert result.example_id == "example-789"
        assert result.output == {"messages": [{"role": "assistant", "content": "Expected response"}]}
        assert result.actual_output == {"messages": [{"role": "assistant", "content": "Actual response"}]}
        assert result.evaluation_scores == {"accuracy": 0.95, "relevance": 0.88}
        assert result.metadata == {"processing_time": 1.23, "model": "test-model"}
        assert result.error is None

    def test_create_error_result(self):
        """Test creating ExperimentResult with error"""
        result = ExperimentResult(
            example_id="example-error",
            input_data={"messages": [{"role": "user", "content": "Error test"}]},
            output={"messages": [{"role": "assistant", "content": "Should have worked"}]},
            actual_output="",
            error="Connection timeout",
        )

        assert result.example_id == "example-error"
        assert result.actual_output == ""
        assert result.error == "Connection timeout"
        assert result.evaluation_scores == {}  # Default empty dict


class TestDatasetConfig:
    """Test DatasetConfig data model"""

    def test_create_dataset_config(self):
        """Test creating DatasetConfig for dataset creation"""
        config = DatasetConfig(
            name="Config Test Dataset",
            description="Dataset created from config",
            tags=["config", "test"],
            metadata={"created_by": "test_suite"},
        )

        assert config.name == "Config Test Dataset"
        assert config.description == "Dataset created from config"
        assert config.tags == ["config", "test"]
        assert config.metadata == {"created_by": "test_suite"}


class TestDataModelCompatibility:
    """Test compatibility and interoperability between data models"""

    def test_dataset_example_to_example_conversion(self):
        """Test that DatasetExample can be conceptually converted to Example"""
        # This would happen when importing examples and then fetching them back
        dataset_example = DatasetExample(
            input={"messages": [{"role": "user", "content": "Convert me"}]},
            output={"messages": [{"role": "assistant", "content": "Converted!"}]},
            tags=["conversion"],
            metadata={"type": "conversion_test"},
            source_trace_id="trace-convert",
            source_node_id="node-convert",
        )

        # Simulate what happens when this becomes an Example after API processing
        api_example = Example(
            id="generated-id-123",  # API would generate this
            input=dataset_example.input,
            output=dataset_example.output,
            source_trace_id=dataset_example.source_trace_id,
            source_node_id=dataset_example.source_node_id,
            metadata=dataset_example.metadata,
            tags=dataset_example.tags,
            created_at="2024-01-01T00:00:00Z",  # API would add these
            updated_at="2024-01-01T00:00:00Z",
        )

        # Verify the conversion preserved all data
        assert api_example.input == dataset_example.input
        assert api_example.output == dataset_example.output
        assert api_example.source_trace_id == dataset_example.source_trace_id
        assert api_example.source_node_id == dataset_example.source_node_id
        assert api_example.metadata == dataset_example.metadata
        assert api_example.tags == dataset_example.tags
