"""
Tests for the CAT Testing SDK dataset import functionality.

These tests verify the import_dataset method in the CATCafeClient.
"""

import pytest

from cat_cafe.sdk import CATCafeClient, DatasetImport, DatasetExample


@pytest.fixture
def client(memory_client):
    """Test client fixture using memory storage."""
    return memory_client


class TestCATCafeClientImport:
    """Test CAT Testing SDK import functionality"""

    def test_sdk_import_dataset_basic(self, client):
        """Test SDK import_dataset method with basic dataset"""

        # Arrange
        cat_client = CATCafeClient(base_url="http://test", session=client)

        dataset_import = DatasetImport(
            name="SDK Test Dataset",
            description="Dataset imported via SDK",
            tags=["sdk", "test"],
            metadata={"source": "sdk_test"},
            examples=[
                DatasetExample(
                    input={"messages": [{"role": "user", "content": "Hello SDK"}]},
                    output={"messages": [{"role": "assistant", "content": "Hi from SDK!"}]},
                    tags=["greeting"],
                    metadata={"test": "sdk"},
                ),
                DatasetExample(
                    input={"messages": [{"role": "user", "content": "SDK calculation: 5+3?"}]},
                    output={"messages": [{"role": "assistant", "content": "8"}]},
                    tags=["math"],
                    metadata={"calculation": True},
                ),
            ],
        )

        # Act
        result = cat_client.import_dataset(dataset_import)

        # Assert
        assert "dataset" in result
        assert "examples_added" in result
        assert "examples_requested" in result
        assert "message" in result

        dataset = result["dataset"]
        assert dataset["name"] == "SDK Test Dataset"
        assert dataset["description"] == "Dataset imported via SDK"
        assert dataset["tags"] == ["sdk", "test"]
        assert dataset["metadata"] == {"source": "sdk_test"}
        assert dataset["example_count"] == 2

        assert result["examples_added"] == 2
        assert result["examples_requested"] == 2

        # Verify we can retrieve the examples
        dataset_id = dataset["id"]
        examples = cat_client.get_dataset_examples(dataset_id)
        assert len(examples) == 2

        # Check example content
        example_contents = [ex["input"][0]["content"] for ex in examples]
        assert "Hello SDK" in example_contents
        assert "SDK calculation: 5+3?" in example_contents

    def test_sdk_import_dataset_empty(self, client):
        """Test SDK import_dataset method with empty dataset"""

        # Arrange
        cat_client = CATCafeClient(base_url="http://test", session=client)

        dataset_import = DatasetImport(
            name="Empty SDK Dataset",
            description="Empty dataset via SDK",
            tags=["empty", "sdk"],
            metadata={"type": "empty"},
            examples=[],
        )

        # Act
        result = cat_client.import_dataset(dataset_import)

        # Assert
        dataset = result["dataset"]
        assert dataset["name"] == "Empty SDK Dataset"
        assert dataset["example_count"] == 0
        assert result["examples_added"] == 0
        assert result["examples_requested"] == 0

    def test_sdk_import_dataset_integration_with_weather_example(self, client):
        """Test SDK import_dataset method matches weather assistant example usage"""

        # Arrange - using the exact structure from weather assistant example
        cat_client = CATCafeClient(base_url="http://test", session=client)

        dataset_import = DatasetImport(
            name="Weather Assistant Test Dataset",
            description="Dataset for evaluating weather assistant responses across different query types",
            tags=["weather", "assistant", "tool-calls", "openai"],
            metadata={
                "domain": "weather_assistance",
                "version": "v1.0",
                "created_by": "weather_assistant_test",
                "app": "weather_assistant_cli",
            },
            examples=[
                DatasetExample(
                    input={"messages": [{"role": "user", "content": "What's the weather like in San Francisco?"}]},
                    output={
                        "messages": [
                            {
                                "role": "assistant",
                                "content": "Current weather information for San Francisco including temperature, conditions, humidity and wind",
                            }
                        ]
                    },
                    tags=["current-weather", "san-francisco"],
                    metadata={"query_type": "current_weather", "city": "san_francisco"},
                ),
                DatasetExample(
                    input={"messages": [{"role": "user", "content": "Will it rain in London tomorrow?"}]},
                    output={
                        "messages": [
                            {
                                "role": "assistant",
                                "content": "Weather forecast for London showing tomorrow's conditions and rain probability",
                            }
                        ]
                    },
                    tags=["forecast", "rain", "london"],
                    metadata={"query_type": "forecast", "city": "london", "time_frame": "tomorrow"},
                ),
                DatasetExample(
                    input={"messages": [{"role": "user", "content": "What's the 3-day forecast for Tokyo?"}]},
                    output={
                        "messages": [
                            {
                                "role": "assistant",
                                "content": "3-day weather forecast for Tokyo with daily highs, lows, and conditions",
                            }
                        ]
                    },
                    tags=["forecast", "multi-day", "tokyo"],
                    metadata={"query_type": "forecast", "city": "tokyo", "time_frame": "3_days"},
                ),
            ],
        )

        # Act
        result = cat_client.import_dataset(dataset_import)

        # Assert
        dataset = result["dataset"]
        assert dataset["name"] == "Weather Assistant Test Dataset"
        assert dataset["example_count"] == 3
        assert result["examples_added"] == 3
        assert result["examples_requested"] == 3

        # Verify metadata structure matches weather assistant requirements
        assert dataset["metadata"]["domain"] == "weather_assistance"
        assert dataset["metadata"]["version"] == "v1.0"
        assert dataset["metadata"]["created_by"] == "weather_assistant_test"

        # Verify examples can be retrieved and used for testing
        dataset_id = dataset["id"]
        examples = cat_client.get_dataset_examples(dataset_id)
        assert len(examples) == 3

        # Check that all weather-related queries are present
        user_queries = [ex["input"][0]["content"] for ex in examples]
        assert "What's the weather like in San Francisco?" in user_queries
        assert "Will it rain in London tomorrow?" in user_queries
        assert "What's the 3-day forecast for Tokyo?" in user_queries

        # Verify metadata is preserved for each example
        for example in examples:
            assert "query_type" in example["metadata"]
            assert "city" in example["metadata"]
            if "forecast" in example["tags"]:
                assert example["metadata"]["query_type"] == "forecast"
            elif "current-weather" in example["tags"]:
                assert example["metadata"]["query_type"] == "current_weather"


if __name__ == "__main__":
    pytest.main([__file__])
