"""Tests for experiment result management methods."""

from unittest.mock import Mock, patch
import pytest

from cat_cafe.sdk import CATCafeClient


@pytest.fixture
def client():
    """Create a client instance for testing."""
    return CATCafeClient(base_url="http://test-server")


@pytest.fixture
def mock_response():
    """Create a mock response object."""
    mock = Mock()
    mock.raise_for_status = Mock()
    return mock


class TestListExperiments:
    """Test list_experiments method."""

    def test_list_experiments_success(self, client, mock_response):
        """Test successful listing of all experiments."""
        expected_experiments = [
            {"experiment_id": "exp-123", "name": "Test Experiment 1", "dataset_id": "dataset-1", "status": "completed"},
            {"experiment_id": "exp-456", "name": "Test Experiment 2", "dataset_id": "dataset-2", "status": "running"},
        ]
        mock_response.json.return_value = {"experiments": expected_experiments}

        with patch.object(client, "_make_request", return_value=mock_response) as mock_request:
            result = client.list_experiments()

            assert result == expected_experiments
            mock_request.assert_called_once_with("GET", "/api/experiments")
            mock_response.raise_for_status.assert_called_once()

    def test_list_experiments_empty(self, client, mock_response):
        """Test listing experiments when none exist."""
        mock_response.json.return_value = {"experiments": []}

        with patch.object(client, "_make_request", return_value=mock_response):
            result = client.list_experiments()
            assert result == []


class TestListExperimentsByDataset:
    """Test list_experiments_by_dataset method."""

    def test_list_experiments_by_dataset_success(self, client, mock_response):
        """Test successful listing of experiments for a dataset."""
        dataset_id = "dataset-123"
        expected_experiments = [
            {"experiment_id": "exp-1", "name": "Dataset Test 1", "dataset_id": dataset_id, "status": "completed"},
            {"experiment_id": "exp-2", "name": "Dataset Test 2", "dataset_id": dataset_id, "status": "completed"},
        ]
        mock_response.json.return_value = {"experiments": expected_experiments}

        with patch.object(client, "_make_request", return_value=mock_response) as mock_request:
            result = client.list_experiments_by_dataset(dataset_id)

            assert result == expected_experiments
            mock_request.assert_called_once_with("GET", f"/api/datasets/{dataset_id}/experiments")
            mock_response.raise_for_status.assert_called_once()


class TestGetExperimentDetail:
    """Test get_experiment_detail method."""

    def test_get_experiment_detail_success(self, client, mock_response):
        """Test successful retrieval of experiment detail."""
        experiment_id = "exp-123"
        expected_detail = {
            "experiment": {
                "experiment_id": experiment_id,
                "name": "Detailed Test",
                "dataset_id": "dataset-1",
                "status": "completed",
                "created_at": "2024-01-01T00:00:00Z",
            },
            "results": [
                {
                    "id": "result-1",
                    "example_id": "ex1",
                    "actual_output": "Test output 1",
                    "evaluation_scores": {"accuracy": 0.9},
                },
                {
                    "id": "result-2",
                    "example_id": "ex2",
                    "actual_output": "Test output 2",
                    "evaluation_scores": {"accuracy": 0.85},
                },
            ],
        }
        mock_response.json.return_value = expected_detail

        with patch.object(client, "_make_request", return_value=mock_response) as mock_request:
            result = client.get_experiment_detail(experiment_id)

            assert result == expected_detail
            assert "experiment" in result
            assert "results" in result
            assert len(result["results"]) == 2
            mock_request.assert_called_once_with("GET", f"/api/experiments/{experiment_id}/detail")


class TestCompareExperiments:
    """Test compare_experiments method."""

    def test_compare_experiments_success(self, client, mock_response):
        """Test successful comparison of two experiments."""
        exp_a = "exp-123"
        exp_b = "exp-456"

        expected_comparison = {
            "experiment_a": {"experiment_id": exp_a, "name": "Experiment A", "dataset_id": "dataset-1"},
            "experiment_b": {"experiment_id": exp_b, "name": "Experiment B", "dataset_id": "dataset-1"},
            "comparison_results": [
                {
                    "example_id": "ex1",
                    "result_a": {"evaluation_scores": {"accuracy": 0.8}},
                    "result_b": {"evaluation_scores": {"accuracy": 0.9}},
                    "metric_deltas": {"accuracy": 0.1},
                }
            ],
            "summary": {
                "total_examples": 1,
                "a_better": 0,
                "b_better": 1,
                "equal": 0,
                "metric_improvements": {"accuracy": 0.1},
            },
        }
        mock_response.json.return_value = expected_comparison

        with patch.object(client, "_make_request", return_value=mock_response) as mock_request:
            result = client.compare_experiments(exp_a, exp_b)

            assert result == expected_comparison
            assert result["summary"]["b_better"] == 1
            mock_request.assert_called_once_with(
                "GET", "/api/experiments/compare", params={"experiment_a": exp_a, "experiment_b": exp_b}
            )


class TestGetExperimentTimeline:
    """Test get_experiment_timeline method."""

    def test_get_experiment_timeline_success(self, client, mock_response):
        """Test successful retrieval of experiment timeline."""
        dataset_id = "dataset-123"

        expected_timeline = {
            "dataset_id": dataset_id,
            "experiments": [
                {
                    "id": "exp-1",
                    "name": "Early Experiment",
                    "created_at": "2024-01-01T00:00:00Z",
                    "metrics": {"accuracy": 0.7},
                    "example_count": 100,
                    "status": "completed",
                },
                {
                    "id": "exp-2",
                    "name": "Later Experiment",
                    "created_at": "2024-01-02T00:00:00Z",
                    "metrics": {"accuracy": 0.85},
                    "example_count": 100,
                    "status": "completed",
                },
            ],
            "metric_types": ["accuracy"],
        }
        mock_response.json.return_value = expected_timeline

        with patch.object(client, "_make_request", return_value=mock_response) as mock_request:
            result = client.get_experiment_timeline(dataset_id)

            assert result == expected_timeline
            assert len(result["experiments"]) == 2
            assert result["metric_types"] == ["accuracy"]
            mock_request.assert_called_once_with("GET", f"/api/datasets/{dataset_id}/experiments/timeline")

    def test_get_experiment_timeline_empty(self, client, mock_response):
        """Test timeline for dataset with no experiments."""
        dataset_id = "dataset-empty"

        expected_timeline = {"dataset_id": dataset_id, "experiments": [], "metric_types": []}
        mock_response.json.return_value = expected_timeline

        with patch.object(client, "_make_request", return_value=mock_response):
            result = client.get_experiment_timeline(dataset_id)

            assert result["experiments"] == []
            assert result["metric_types"] == []


class TestErrorHandling:
    """Test error handling in experiment management methods."""

    def test_http_error_propagation(self, client, mock_response):
        """Test that HTTP errors are properly propagated."""
        mock_response.raise_for_status.side_effect = Exception("404 Not Found")

        with patch.object(client, "_make_request", return_value=mock_response):
            with pytest.raises(Exception, match="404 Not Found"):
                client.list_experiments()

    def test_compare_experiments_invalid_ids(self, client, mock_response):
        """Test comparison with invalid experiment IDs."""
        mock_response.raise_for_status.side_effect = Exception("400 Bad Request: Experiments must be on same dataset")

        with patch.object(client, "_make_request", return_value=mock_response):
            with pytest.raises(Exception, match="400 Bad Request"):
                client.compare_experiments("invalid-1", "invalid-2")
