"""Tests for the cat-cafe-sdk core functionality."""

import pytest
from unittest.mock import Mock

from cat_cafe.sdk import CATCafeClient, Experiment, ExperimentResult


class TestClientInitialization:
    """Test SDK client initialization"""

    def test_client_creation_with_defaults(self):
        """Test creating client with default settings"""
        client = CATCafeClient()
        assert client.base_url == "http://localhost:8000"
        assert client._session is None

    def test_client_creation_with_custom_url(self):
        """Test creating client with custom base URL"""
        client = CATCafeClient(base_url="http://custom:9000")
        assert client.base_url == "http://custom:9000"

    def test_client_creation_with_session(self):
        """Test creating client with custom session"""
        mock_session = Mock()
        client = CATCafeClient(session=mock_session)
        assert client._session == mock_session


class TestDataModelCreation:
    """Test creating data model objects"""

    def test_experiment_creation(self):
        """Test creating Experiment objects"""
        experiment = Experiment(
            name="Test Experiment",
            description="A test experiment",
            dataset_id="dataset-123",
            tags=["test"],
            metadata={"key": "value"},
        )

        assert experiment.name == "Test Experiment"
        assert experiment.description == "A test experiment"
        assert experiment.dataset_id == "dataset-123"
        assert experiment.tags == ["test"]
        assert experiment.metadata == {"key": "value"}
        assert experiment.dataset_version is None

    def test_experiment_result_creation(self):
        """Test creating ExperimentResult objects"""
        result = ExperimentResult(
            example_id="example-456",
            input_data={"messages": [{"role": "user", "content": "test"}]},
            output={"messages": [{"role": "assistant", "content": "expected"}]},
            actual_output={"messages": [{"role": "assistant", "content": "actual"}]},
            evaluation_scores={"score": 0.8},
            metadata={"time": 1.5},
        )

        assert result.example_id == "example-456"
        assert result.input_data == {"messages": [{"role": "user", "content": "test"}]}
        assert result.output == {"messages": [{"role": "assistant", "content": "expected"}]}
        assert result.actual_output == {"messages": [{"role": "assistant", "content": "actual"}]}
        assert result.evaluation_scores == {"score": 0.8}
        assert result.metadata == {"time": 1.5}
        assert result.error is None


class TestClientMethodSignatures:
    """Test that client methods have correct signatures"""

    def test_client_has_required_methods(self):
        """Test that client has all expected methods"""
        client = CATCafeClient()

        # Core experiment methods
        assert hasattr(client, "start_experiment")
        assert hasattr(client, "get_experiment")
        assert hasattr(client, "submit_results")
        assert hasattr(client, "complete_experiment")
        assert hasattr(client, "get_experiment_results")

        # Dataset methods
        assert hasattr(client, "create_dataset")
        assert hasattr(client, "get_dataset")
        assert hasattr(client, "list_datasets")
        assert hasattr(client, "import_dataset")
        assert hasattr(client, "fetch_dataset")
        assert hasattr(client, "fetch_dataset_by_name")
        assert hasattr(client, "add_example_from_trace")


class TestRequestMethodValidation:
    """Test internal request method validation"""

    def test_make_request_method_validation(self):
        """Test that _make_request validates HTTP methods"""
        client = CATCafeClient()

        # Should raise error for unsupported HTTP method
        with pytest.raises(ValueError, match="Unsupported HTTP method"):
            client._make_request("PATCH", "/test/url")


class TestSDKPackageStructure:
    """Test SDK package structure and imports"""

    def test_main_imports_available(self):
        """Test that all main classes can be imported from root package"""
        from cat_cafe.sdk import (
            CATCafeClient,
            Experiment,
            ExperimentResult,
            DatasetConfig,
            DatasetExample,
            DatasetImport,
            Example,
            Dataset,
        )

        # All imports should succeed without error
        assert CATCafeClient is not None
        assert Experiment is not None
        assert ExperimentResult is not None
        assert DatasetConfig is not None
        assert DatasetExample is not None
        assert DatasetImport is not None
        assert Example is not None
        assert Dataset is not None

    def test_package_version_available(self):
        """Test that package version is available"""
        import cat_cafe.sdk

        assert hasattr(cat_cafe.sdk, "__version__")
        assert cat_cafe.sdk.__version__ == "0.1.2"
