"""
Tests for async functionality in CAT Cafe SDK.

These tests verify that the SDK can handle both sync and async test functions
without requiring server infrastructure.
"""

import pytest
import asyncio
import inspect

from cat_cafe.sdk import Example


class TestAsyncFunctionDetection:
    """Test async function detection logic"""

    def test_sync_function_detection(self):
        """Test that sync functions are correctly identified"""

        def sync_function(example):
            return "sync result"

        assert not inspect.iscoroutinefunction(sync_function)

    def test_async_function_detection(self):
        """Test that async functions are correctly identified"""

        async def async_function(example):
            return "async result"

        assert inspect.iscoroutinefunction(async_function)


class TestAsyncTestFunctionExecution:
    """Test execution of async test functions"""

    @pytest.fixture
    def sample_example(self):
        """Sample Example for testing"""
        example = Example(
            id="test-example",
            input={"messages": [{"role": "user", "content": "Test input"}]},
            output={"messages": [{"role": "assistant", "content": "Test output"}]},
            metadata={},
        )
        example.tags = ["test"]
        return example

    def test_sync_function_execution(self, sample_example):
        """Test that sync functions execute correctly"""

        def sync_test_function(example):
            messages = example.input.get("messages", [])
            user_input = messages[-1]["content"] if messages else ""
            return f"Processed: {user_input}"

        result = sync_test_function(sample_example)
        assert result == "Processed: Test input"

    @pytest.mark.asyncio
    async def test_async_function_execution(self, sample_example):
        """Test that async functions execute correctly"""

        async def async_test_function(example):
            await asyncio.sleep(0.001)  # Simulate async work
            messages = example.input.get("messages", [])
            user_input = messages[-1]["content"] if messages else ""
            return f"Async processed: {user_input}"

        result = await async_test_function(sample_example)
        assert result == "Async processed: Test input"

    def test_async_function_with_asyncio_run(self, sample_example):
        """Test that async functions can be run with asyncio.run()"""

        async def async_test_function(example):
            await asyncio.sleep(0.001)
            messages = example.input.get("messages", [])
            user_input = messages[-1]["content"] if messages else ""
            return f"Run result: {user_input}"

        # This simulates how the SDK would run async functions
        result = asyncio.run(async_test_function(sample_example))
        assert result == "Run result: Test input"


class TestAsyncEvaluatorHandling:
    """Test that evaluators work with both sync and async test functions"""

    def test_evaluators_with_sync_function_output(self):
        """Test that evaluators work with sync function output"""

        def accuracy_evaluator(actual_output, output):
            messages = []
            if isinstance(output, dict):
                messages = output.get("messages", [])
            elif isinstance(output, list):
                messages = output
            expected_content = messages[0]["content"] if messages else ""
            score = 1.0 if expected_content in actual_output else 0.5
            reason = "Contains expected" if score == 1.0 else "Partial match"
            return score, reason

        def length_evaluator(actual_output, output):
            score = min(len(actual_output) / 20.0, 1.0)
            reason = f"Length: {len(actual_output)}"
            return score, reason

        # Test with sync function output
        sync_output = "This is a sync response with expected content"
        expected = {"messages": [{"role": "assistant", "content": "expected content"}]}

        acc_score, acc_reason = accuracy_evaluator(sync_output, expected)
        assert acc_score == 1.0
        assert acc_reason == "Contains expected"

        len_score, len_reason = length_evaluator(sync_output, expected)
        assert len_score > 0.0
        assert "Length:" in len_reason

    def test_evaluators_with_async_function_output(self):
        """Test that evaluators work with async function output"""

        def evaluator(actual_output, output):
            score = 0.8 if "async" in actual_output.lower() else 0.3
            reason = "Contains async marker" if score == 0.8 else "No async marker"
            return score, reason

        # Test with async function output
        async_output = "This is an async response"
        expected = {"messages": [{"role": "assistant", "content": "response"}]}

        score, reason = evaluator(async_output, expected)
        assert score == 0.8
        assert reason == "Contains async marker"


class TestAsyncErrorHandling:
    """Test error handling with async functions"""

    def test_async_function_exception_handling(self, sample_example):
        """Test that exceptions in async functions can be handled"""

        async def failing_async_function(example):
            await asyncio.sleep(0.001)
            raise ValueError("Async function error")

        # Test that we can catch the exception
        with pytest.raises(ValueError, match="Async function error"):
            asyncio.run(failing_async_function(sample_example))

    def test_async_timeout_simulation(self, sample_example):
        """Test simulating timeout behavior with async functions"""

        async def slow_async_function(example):
            await asyncio.sleep(0.1)  # Simulate slow operation
            return "Eventually completed"

        # Test that the function eventually completes
        result = asyncio.run(slow_async_function(sample_example))
        assert result == "Eventually completed"


class TestEventLoopHandling:
    """Test event loop considerations"""

    def test_no_running_loop_detection(self):
        """Test detecting when no event loop is running"""
        try:
            asyncio.get_running_loop()
            running_loop = True
        except RuntimeError:
            running_loop = False

        # In regular test execution, this should be False
        # (unless using pytest-asyncio mode=auto)
        # The important thing is that we can detect the state
        assert isinstance(running_loop, bool)

    def test_nested_event_loop_prevention(self):
        """Test that we can detect nested event loop situations"""

        async def outer_async():
            # This would fail if we tried to use asyncio.run() here
            try:
                asyncio.get_running_loop()
                return "Already in loop"
            except RuntimeError:
                return "No loop running"

        # Run the test
        result = asyncio.run(outer_async())
        assert result == "Already in loop"


class TestMixedSyncAsyncUsage:
    """Test scenarios with both sync and async functions"""

    @pytest.fixture
    def mixed_test_functions(self):
        """Collection of both sync and async test functions"""

        def sync_function(example):
            messages = example.input.get("messages", [])
            content = messages[-1]["content"] if messages else ""
            return f"Sync: {content}"

        async def async_function(example):
            await asyncio.sleep(0.001)
            messages = example.input.get("messages", [])
            content = messages[-1]["content"] if messages else ""
            return f"Async: {content}"

        return {"sync": sync_function, "async": async_function}

    @pytest.fixture
    def evaluators(self):
        """Collection of evaluator functions"""

        def type_evaluator(actual_output, output):
            if "Sync:" in actual_output:
                return 0.7, "Sync function used"
            elif "Async:" in actual_output:
                return 0.9, "Async function used"
            else:
                return 0.1, "Unknown function type"

        def length_evaluator(actual_output, output):
            score = min(len(actual_output) / 25.0, 1.0)
            return score, f"Length score: {score:.2f}"

        return [type_evaluator, length_evaluator]

    def test_sync_function_with_evaluators(self, sample_example, mixed_test_functions, evaluators):
        """Test sync function with multiple evaluators"""
        sync_func = mixed_test_functions["sync"]
        result = sync_func(sample_example)

        assert result.startswith("Sync:")

        # Test evaluators work
        for evaluator in evaluators:
            score, reason = evaluator(result, sample_example.output)
            assert isinstance(score, (int, float))
            assert isinstance(reason, str)
            assert 0.0 <= score <= 1.0

    @pytest.mark.asyncio
    async def test_async_function_with_evaluators(self, sample_example, mixed_test_functions, evaluators):
        """Test async function with multiple evaluators"""
        async_func = mixed_test_functions["async"]
        result = await async_func(sample_example)

        assert result.startswith("Async:")

        # Test evaluators work with async result
        for evaluator in evaluators:
            score, reason = evaluator(result, sample_example.output)
            assert isinstance(score, (int, float))
            assert isinstance(reason, str)
            assert 0.0 <= score <= 1.0

    def test_function_type_detection_and_execution(self, sample_example, mixed_test_functions):
        """Test detecting function type and executing appropriately"""
        sync_func = mixed_test_functions["sync"]
        async_func = mixed_test_functions["async"]

        # Test sync detection and execution
        assert not inspect.iscoroutinefunction(sync_func)
        sync_result = sync_func(sample_example)
        assert isinstance(sync_result, str)

        # Test async detection and execution
        assert inspect.iscoroutinefunction(async_func)
        async_result = asyncio.run(async_func(sample_example))
        assert isinstance(async_result, str)

        # Both should process the same input differently
        assert sync_result != async_result
        assert "Test input" in sync_result
        assert "Test input" in async_result


@pytest.fixture
def sample_example():
    """Sample Example for testing"""
    return Example(
        id="test-example",
        input={"messages": [{"role": "user", "content": "Test input"}]},
        output={"messages": [{"role": "assistant", "content": "Test output"}]},
        tags=["test"],
        metadata={},
    )
