"""Ensure vendored API models stay in sync with backend sources when developing in the monorepo."""

from __future__ import annotations

import hashlib
from pathlib import Path

import pytest

FILES_TO_COMPARE = {
    "api_models/__init__.py": "api_models/__init__.py",
    "api_models/annotations.py": "api_models/annotations.py",
    "api_models/datasets.py": "api_models/datasets.py",
    "api_models/experiments.py": "api_models/experiments.py",
    "api_models/traces.py": "api_models/traces.py",
    "traces/api_models.py": "traces/api_models.py",
}


def _sha256(path: Path) -> str:
    return hashlib.sha256(path.read_bytes()).hexdigest()


@pytest.mark.skipif(
    not (Path(__file__).resolve().parents[3] / "src" / "cat_cafe").exists(),
    reason="Backend source tree not available for drift check",
)
def test_vendored_api_models_match_backend_sources():
    repo_root = Path(__file__).resolve().parents[3]
    backend_root = repo_root / "src" / "cat_cafe"
    sdk_root = repo_root / "packages" / "cat-cafe-sdk" / "src" / "cat_cafe"

    mismatches = []
    missing = []

    for source_rel, vendored_rel in FILES_TO_COMPARE.items():
        source_path = backend_root / source_rel
        vendored_path = sdk_root / vendored_rel

        if not source_path.exists() or not vendored_path.exists():
            missing.append((source_path, vendored_path))
            continue

        if _sha256(source_path) != _sha256(vendored_path):
            mismatches.append(vendored_path)

    if missing:
        missing_desc = "; ".join(f"{src} -> {vend}" for src, vend in missing)
        pytest.skip(f"API model drift check skipped due to missing files: {missing_desc}")

    assert not mismatches, f"Vendored API models drifted: {', '.join(str(path) for path in mismatches)}"
