"""
Test configuration and fixtures for cat-cafe-sdk.

This module provides test fixtures for testing the SDK independently,
including mock HTTP responses and test data.
"""

import pytest
from unittest.mock import Mock


@pytest.fixture
def mock_session():
    """Mock session that simulates API responses for testing."""
    session = Mock()

    # Default responses for common API calls
    def mock_response(status_code=200, json_data=None):
        response = Mock()
        response.status_code = status_code
        response.json.return_value = json_data or {}
        response.raise_for_status = Mock()
        return response

    # Configure mock responses for common endpoints
    session.get.return_value = mock_response(200, {"examples": []})
    session.post.return_value = mock_response(200, {"experiment_id": "test-123"})

    return session


@pytest.fixture
def sample_dataset_import():
    """Sample DatasetImport for testing."""
    from cat_cafe.sdk import DatasetImport, DatasetExample

    return DatasetImport(
        name="Test Dataset",
        description="A test dataset for SDK testing",
        tags=["test", "sdk"],
        metadata={"test": True},
        examples=[
            DatasetExample(
                input={"messages": [{"role": "user", "content": "Hello"}]},
                output={"messages": [{"role": "assistant", "content": "Hi there!"}]},
                tags=["greeting"],
                metadata={"example_type": "greeting"},
            ),
            DatasetExample(
                input={"messages": [{"role": "user", "content": "What's 2+2?"}]},
                output={"messages": [{"role": "assistant", "content": "4"}]},
                tags=["math"],
                metadata={"example_type": "calculation"},
            ),
        ],
    )


@pytest.fixture
def sample_experiment_config():
    """Sample Experiment configuration for testing."""
    from cat_cafe.sdk import Experiment

    return Experiment(
        name="Test Experiment",
        description="A test experiment for SDK testing",
        dataset_id="test-dataset-123",
        tags=["test", "automated"],
        metadata={"experiment_type": "sdk_test"},
    )


@pytest.fixture
def sample_examples():
    """Sample Example objects for testing."""
    from cat_cafe.sdk import Example

    def make_example(
        *,
        id: str,
        input: dict,
        output: dict,
        metadata: dict | None = None,
        created_at: str | None = None,
        updated_at: str | None = None,
        tags: list[str] | None = None,
    ) -> Example:
        example = Example(
            id=id,
            input=input,
            output=output,
            metadata=dict(metadata or {}),
            created_at=created_at,
            updated_at=updated_at,
        )
        if tags:
            example.tags = list(tags)
        return example

    return [
        make_example(
            id="example-1",
            input={"messages": [{"role": "user", "content": "Hello"}]},
            output={"messages": [{"role": "assistant", "content": "Hi there!"}]},
            metadata={"type": "greeting"},
            created_at="2024-01-01T00:00:00Z",
            updated_at="2024-01-01T00:00:00Z",
            tags=["greeting"],
        ),
        make_example(
            id="example-2",
            input={"messages": [{"role": "user", "content": "What's 2+2?"}]},
            output={"messages": [{"role": "assistant", "content": "4"}]},
            metadata={"type": "calculation"},
            created_at="2024-01-01T00:00:00Z",
            updated_at="2024-01-01T00:00:00Z",
            tags=["math"],
        ),
    ]


@pytest.fixture
def sample_dataset():
    """Sample Dataset object for testing."""
    from cat_cafe.sdk import Dataset

    return Dataset(
        id="dataset-123",
        name="Test Dataset",
        description="A test dataset",
        tags=["test"],
        metadata={"test": True},
        example_count=2,
        version=1,
        created_at="2024-01-01T00:00:00Z",
        updated_at="2024-01-01T00:00:00Z",
        examples=[],  # Will be populated by other fixtures if needed
    )


def test_function_sync(example):
    """Sample synchronous test function for testing."""
    messages = example.input.get("messages", [])
    user_input = messages[-1]["content"] if messages else ""
    return f"Response to: {user_input}"


async def test_function_async(example):
    """Sample asynchronous test function for testing."""
    import asyncio

    await asyncio.sleep(0.001)  # Simulate async work
    messages = example.input.get("messages", [])
    user_input = messages[-1]["content"] if messages else ""
    return f"Async response to: {user_input}"


def evaluator_accuracy(actual_output, output):
    """Sample evaluator function for testing."""
    # Simple evaluation: check if response contains expected content
    messages = output.get("messages", []) if isinstance(output, dict) else []
    expected_content = messages[0]["content"] if messages else ""
    score = 1.0 if expected_content.lower() in actual_output.lower() else 0.5
    reason = "Contains expected content" if score == 1.0 else "Partial match"
    return score, reason


def evaluator_length(actual_output, output):
    """Sample length-based evaluator for testing."""
    score = min(len(actual_output) / 20.0, 1.0)  # Score based on length
    reason = f"Length: {len(actual_output)} chars"
    return score, reason


# Export test functions and evaluators for use in tests
@pytest.fixture
def sync_test_function():
    """Sync test function fixture."""
    return test_function_sync


@pytest.fixture
def async_test_function():
    """Async test function fixture."""
    return test_function_async


@pytest.fixture
def accuracy_evaluator():
    """Accuracy evaluator fixture."""
    return evaluator_accuracy


@pytest.fixture
def length_evaluator():
    """Length evaluator fixture."""
    return evaluator_length


@pytest.fixture
def memory_client():
    """Mock memory client that simulates API responses for SDK testing."""
    session = Mock()

    # Mock dataset storage for testing
    datasets = {}
    examples = {}
    experiments = {}
    next_id = 1

    def generate_id():
        nonlocal next_id
        id_val = f"test-{next_id:03d}"
        next_id += 1
        return id_val

    def mock_post(url, **kwargs):
        response = Mock()
        response.status_code = 200
        response.raise_for_status = Mock()

        if "datasets/import" in url:
            # Handle dataset import
            data = kwargs.get("json", {})
            dataset_id = generate_id()

            dataset = {
                "id": dataset_id,
                "name": data.get("name", "Test Dataset"),
                "description": data.get("description", ""),
                "tags": data.get("tags", []),
                "metadata": data.get("metadata", {}),
                "example_count": len(data.get("examples", [])),
                "version": 1,
                "created_at": "2024-01-01T00:00:00Z",
                "updated_at": "2024-01-01T00:00:00Z",
            }

            datasets[dataset_id] = dataset

            # Store examples
            dataset_examples = []
            for example_data in data.get("examples", []):
                example_id = generate_id()
                example = {
                    "id": example_id,
                    "input": example_data.get("input", []),
                    "output": example_data.get("output", []),
                    "tags": example_data.get("tags", []),
                    "metadata": example_data.get("metadata", {}),
                    "created_at": "2024-01-01T00:00:00Z",
                    "updated_at": "2024-01-01T00:00:00Z",
                }
                dataset_examples.append(example)

            examples[dataset_id] = dataset_examples

            response.json.return_value = {
                "dataset": dataset,
                "examples_added": len(dataset_examples),
                "examples_requested": len(data.get("examples", [])),
                "message": "Dataset imported successfully",
            }

        elif "/datasets/" in url and "/examples/from-span" in url:
            # Handle creating example from trace span
            data = kwargs.get("json", {})
            dataset_id = url.split("/datasets/")[1].split("/examples")[0]
            dataset = datasets.get(dataset_id)

            if not dataset:
                response.status_code = 404
                response.json.return_value = {"detail": f"Dataset {dataset_id} not found"}
                return response

            example_id = generate_id()
            dataset_examples = examples.setdefault(dataset_id, [])

            example = {
                "id": example_id,
                # Provide simple placeholder payloads; the real service extracts these from traces
                "input": {"messages": [{"role": "user", "content": f"Trace {data.get('trace_id', '')} input"}]},
                "output": {"messages": [{"role": "assistant", "content": "Generated from trace"}]},
                "tags": data.get("tags", []),
                "metadata": {
                    **(data.get("metadata") or {}),
                    "trace_id": data.get("trace_id"),
                    "node_id": data.get("node_id"),
                },
                "created_at": "2024-01-01T00:00:00Z",
                "updated_at": "2024-01-01T00:00:00Z",
            }

            dataset_examples.append(example)
            dataset["example_count"] = len(dataset_examples)
            dataset["updated_at"] = "2024-01-01T00:00:00Z"

            response.json.return_value = {
                "message": "Example created from span successfully",
                "example": example,
                "dataset": dataset,
                "change_summary": f"Added example from trace {data.get('trace_id')}",
            }

        elif "experiments" in url and url.endswith("experiments"):
            # Handle experiment creation
            data = kwargs.get("json", {})
            experiment_id = generate_id()

            experiment = {
                "id": experiment_id,
                "experiment_id": experiment_id,  # Also include for compatibility
                "name": data.get("name", "Test Experiment"),
                "description": data.get("description", ""),
                "dataset_id": data.get("dataset_id"),
                "dataset_version": data.get("dataset_version"),
                "tags": data.get("tags", []),
                "metadata": data.get("metadata", {}),
                "status": "created",
                "created_at": "2024-01-01T00:00:00Z",
                "updated_at": "2024-01-01T00:00:00Z",
            }

            experiments[experiment_id] = experiment
            response.json.return_value = experiment

        else:
            response.json.return_value = {"message": "Success"}

        return response

    def mock_get(url, **kwargs):
        response = Mock()
        response.status_code = 200
        response.raise_for_status = Mock()

        if "datasets/" in url and "/examples" in url:
            # Handle get dataset examples
            dataset_id = url.split("/datasets/")[1].split("/examples")[0]
            dataset_examples = examples.get(dataset_id, [])
            response.json.return_value = dataset_examples

        elif "datasets/" in url:
            # Handle get dataset
            dataset_id = url.split("/datasets/")[1]
            dataset = datasets.get(dataset_id)
            if dataset:
                response.json.return_value = dataset
            else:
                response.status_code = 404

        elif "experiments/" in url:
            # Handle get experiment
            experiment_id = url.split("/experiments/")[1]
            experiment = experiments.get(experiment_id)
            if experiment:
                response.json.return_value = experiment
            else:
                response.status_code = 404

        else:
            response.json.return_value = {}

        return response

    session.post = mock_post
    session.get = mock_get
    session.put = mock_post  # Reuse post logic for puts

    return session
