#!/usr/bin/env python3
"""
Test script to verify the CAT Cafe SDK works independently.

This script tests the SDK without requiring the full CAT Cafe server to be running.
It verifies that the SDK can be imported and used correctly.

Run with:
    cd packages/cat-cafe-sdk
    python test_sdk.py
"""

from cat_cafe.sdk import CATCafeClient, DatasetImport, DatasetExample, Experiment, ExperimentResult


def test_imports():
    """Test that all SDK classes can be imported successfully."""
    print("✅ All SDK imports successful")


def test_data_models():
    """Test that data models can be created correctly."""

    # Test DatasetExample
    example = DatasetExample(
        input={"messages": [{"role": "user", "content": "Hello"}]},
        output={"messages": [{"role": "assistant", "content": "Hi there!"}]},
        tags=["greeting"],
        metadata={"test": True},
    )
    assert example.input["messages"][0]["content"] == "Hello"
    assert example.tags == ["greeting"]
    print("✅ DatasetExample creation works")

    # Test DatasetImport
    dataset_import = DatasetImport(name="Test Dataset", description="A test dataset", examples=[example], tags=["test"])
    assert dataset_import.name == "Test Dataset"
    assert len(dataset_import.examples) == 1
    print("✅ DatasetImport creation works")

    # Test Experiment
    experiment = Experiment(
        name="Test Experiment", description="A test experiment", dataset_id="test-123", tags=["automated"]
    )
    assert experiment.name == "Test Experiment"
    assert experiment.dataset_id == "test-123"
    print("✅ Experiment creation works")

    # Test ExperimentResult
    result = ExperimentResult(
        example_id="example-123",
        input_data={"messages": [{"role": "user", "content": "test"}]},
        output={"messages": [{"role": "assistant", "content": "Expected response"}]},
        actual_output={"messages": [{"role": "assistant", "content": "Actual response"}]},
        evaluation_scores={"accuracy": 0.8},
    )
    assert result.example_id == "example-123"
    assert result.evaluation_scores["accuracy"] == 0.8
    print("✅ ExperimentResult creation works")


def test_client_creation():
    """Test that clients can be created without errors."""

    # Test CATCafeClient (alias)
    client1 = CATCafeClient(base_url="http://localhost:8000")
    assert client1.base_url == "http://localhost:8000"
    print("✅ CATCafeClient creation works")

    # Test CATCafeClient
    client2 = CATCafeClient(base_url="http://localhost:8000")
    assert client2.base_url == "http://localhost:8000"
    print("✅ CATCafeClient creation works")

    # Verify they're the same class (alias)
    assert isinstance(client1, type(client2))
    print("✅ CATCafeClient is correctly aliased to CATCafeClient")


def test_backwards_compatibility():
    """Test that the SDK maintains backwards compatibility."""

    # Test that all expected classes exist
    required_classes = [
        "CATCafeClient",
        "CATCafeClient",
        "Experiment",
        "ExperimentResult",
        "DatasetConfig",
        "DatasetExample",
        "DatasetImport",
        "Example",
        "Dataset",
    ]

    import cat_cafe.sdk

    for class_name in required_classes:
        assert hasattr(cat_cafe.sdk, class_name), f"Missing class: {class_name}"

    print("✅ All required classes are exported")


def test_dataclass_features():
    """Test that dataclasses work correctly with default values."""

    # Test with minimal data
    minimal_example = DatasetExample(
        input={"messages": [{"role": "user", "content": "test"}]},
        output={"messages": [{"role": "assistant", "content": "response"}]},
    )
    assert minimal_example.tags == []  # Default empty list
    assert minimal_example.metadata == {}  # Default empty dict
    assert minimal_example.source_trace_id is None  # Default None
    print("✅ Dataclass defaults work correctly")

    # Test with full data
    full_example = DatasetExample(
        input={"messages": [{"role": "user", "content": "test"}]},
        output={"messages": [{"role": "assistant", "content": "response"}]},
        tags=["tag1", "tag2"],
        metadata={"key": "value"},
        source_trace_id="trace-123",
        source_node_id="node-456",
    )
    assert full_example.source_trace_id == "trace-123"
    assert full_example.source_node_id == "node-456"
    print("✅ Dataclass full initialization works")


def main():
    """Run all tests."""
    print("🧪 Testing CAT Cafe SDK...")
    print("=" * 40)

    try:
        test_imports()
        test_data_models()
        test_client_creation()
        test_backwards_compatibility()
        test_dataclass_features()

        print("=" * 40)
        print("🎉 All SDK tests passed!")
        print("\nThe CAT Cafe SDK is working correctly and ready to use.")
        print("\nNext steps:")
        print("1. Start the CAT Cafe server: uv run python main.py")
        print("2. Run examples: python examples/simple_weather_test.py")
        print("3. Build your own AI testing workflows!")

    except Exception as e:
        print(f"❌ Test failed: {e}")
        raise


if __name__ == "__main__":
    main()
