"""API request/response models for trace endpoints (re-exported for SDK sharing)."""

import json
from typing import Any, Dict, Optional

from cat_cafe.traces.api_models import (
    Span,
    SpanEvent,
    SpanEventAttribute,
    SpanStatus,
    Trace,
    TraceResponse,
)
from cat_cafe.traces.tree_models import ToolCall as _TreeToolCall


class ToolCall(_TreeToolCall):
    """Tool call model with compatibility helpers for SDK evaluation."""

    @property
    def args(self) -> Optional[Dict[str, Any]]:
        """Backwards-compatible accessor for function arguments."""
        value = getattr(self, "arguments", None)
        if value is None:
            return None
        if isinstance(value, str):
            try:
                return json.loads(value)
            except Exception:
                return None
        return value  # type: ignore[return-value]

    @args.setter
    def args(self, value: Optional[Dict[str, Any]]) -> None:
        """Allow setting args while keeping underlying arguments field."""
        self.arguments = value

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> "ToolCall":
        """Create from dict supporting both args and arguments keys."""
        arguments = data.get("arguments", data.get("args"))
        payload = {**data, "arguments": arguments}
        return cls(**payload)


__all__ = [
    "Span",
    "SpanEvent",
    "SpanEventAttribute",
    "SpanStatus",
    "Trace",
    "TraceResponse",
    "ToolCall",
]
