# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['flake8_error_link']
install_requires = \
['flake8>=6,<7']

entry_points = \
{'flake8.extension': ['ELI = flake8_error_link:Plugin']}

setup_kwargs = {
    'name': 'flake8-error-link',
    'version': '1.2.3',
    'description': 'A linter that ensures all raised Exceptions include an error with a link to more information',
    'long_description': '# flake8-error-link\n\nHave you ever encountered an error when using a package and then gone to Google\nto find out how to solve the error? Wouldn\'t your users prefer to go directly\nto your documentation that tells them exactly what went wrong and how to\nresolve that error? `flake8-error-link` checks the way exceptions are raised in\nyour code base to ensure that a link with more information is provided.\n\n## Getting Started\n\n```shell\npython -m venv venv\nsource ./venv/bin/activate\npip install flake8 flake8-error-link\nflake8 source.py\n```\n\nOn the following code:\n\n```Python\n# source.py\nraise Exception\n```\n\nThis will produce warnings such as:\n\n```shell\nsource.py:1:0: ELI001 builtin exceptions should be raised with a link to more information: https://github.com/jdkandersson/flake8-error-link#fix-eli001\n```\n\nThis can be resolved by changing the code to:\n\n```Python\n# source.py\nraise Exception("more information: https://github.com/jdkandersson/flake8-error-link#fix-eli001")\n```\n\n## Configuration\n\nThe plugin adds the following configurations to `flake8`:\n\n* `--error-link-regex`: The regular expression to use to verify the way\n  exceptions are reased, defaults to\n  `more information: (mailto\\:|(news|(ht|f)tp(s?))\\:\\/\\/){1}\\S+`\n\n\n## Rules\n\nA few rules have been defined to allow for selective suppression:\n\n* `ELI001`: checks that any builtin exceptions that are raised with constant\n   arguments include a link to more information.\n* `ELI002`: checks that any custom exceptions that are raised with constant\n   arguments include a link to more information.\n* `ELI003`: checks that any exceptions that are raised with variable arguments\n  include a constant argument with a link to more information.\n* `ELI004`: checks that any exceptions that are re-raised include a constant\n  argument with a link to more information.\n\n### Fix ELI001\n\nThis linting rule is trigger by raising an inbuilt exception without providing\na constant that includes a link to more information as one of the arguments to\nthe constructor. For example:\n\n```Python\nraise Exception\n\nraise ValueError\n\nraise Exception()\n\nraise Exception("oh no! :(")\n```\n\nThese examples can be fixed by using something like:\n\n```Python\nraise Exception(\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli001"\n)\n\nraise ValueError(\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli001"\n)\n\nraise Exception(\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli001"\n)\n\nraise Exception(\n    "oh no! :(",\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli001",\n)\n```\n\n### Fix ELI002\n\nThis linting rule is trigger by raising a custom exception without providing\na constant that include a link to more information as one of the arguments to\nthe constructor. For example:\n\n```Python\nclass CustomError(Exception):\n    pass\n\nraise CustomError\n\nraise CustomError()\n\nraise CustomError("bummer...")\n```\n\nThese examples can be fixed by using something like:\n\n```Python\nclass CustomError(Exception):\n    pass\n\nraise CustomError(\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli002"\n)\n\nraise CustomError(\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli002"\n)\n\nraise CustomError(\n    "bummer...",\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli002",\n)\n```\n\n### Fix ELI003\n\nThis linting rule is trigger by raising an exception and passing at least one\nargument without providing a constant that include a link to more information\nas one of the arguments to the constructor. For example:\n\n```Python\nmessage = "gotcha"\n\ndef get_message():\n    return message\n\nraise Exception(get_message())\n\nraise Exception(f"{message} quite badly")\n```\n\nThese examples can be fixed by using something like:\n\n```Python\nmessage = "gotcha"\n\ndef get_message():\n    return message\n\nraise Exception(\n    get_message(),\n    "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli003",\n)\n\nraise Exception(\n    f"{message} quite badly, more information: https://github.com/jdkandersson/flake8-error-link#fix-eli003"\n)\n```\n\n### Fix ELI004\n\nThis linting rule is trigger by re-raising an exception. For example:\n\n```Python\ntry:\n    raise Exception(\n        "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli004"\n    )\nexcept Exception:\n    raise\n```\n\nThis example can be fixed by using something like:\n\n```Python\ntry:\n    raise Exception(\n        "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli004"\n    )\nexcept Exception as exc:\n    raise Exception(\n        "more information: https://github.com/jdkandersson/flake8-error-link#fix-eli004"\n    ) from exc\n```\n\nThis rule can be spurious at times if an exception is re-raisesd that already\nhas a more information link. Regardless, it is usually a good idea to include a\nspecific link for a problem. The context is usually different when an exception\nis re-raised so it could be useful to include documentation for that context\nrather then relying on any link provided by the original exception.\n',
    'author': 'David Andersson',
    'author_email': 'david@jdkandersson.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8.1,<4.0.0',
}


setup(**setup_kwargs)
