# Copyright (c) 2021 Emanuele Bellocchia
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.


# Imports
import unittest
from bip_utils import (
    Bip49BitcoinCashMainNet, Bip49BitcoinCashTestNet, Bip49LitecoinMainNet, Bip49LitecoinTestNet,
    Bip49, Bip44Coins, Bip49Coins, Bip84Coins
)
from tests.bip.bip44_base.test_bip44_base import Bip44BaseTestHelper
from tests.bip.bip44.test_bip44 import TEST_SEED

# Results generated with: https://iancoleman.io/bip39
# For Litecoin, the standard extended keys were generated by this library after begin tested with the alternated ones,
# because the website only allows alternate keys for BIP49
# There are some differences from the website and the specs I found for Litecoin testnet (extended keys prefixes) so,
# in that case, the keys were generated by this library after begin tested for the correct addresses
TEST_VECT = [
    # Bitcoin
    {
        "coin": Bip49Coins.BITCOIN,
        "names": ("Bitcoin", "BTC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "ypub6Ww3ibxVfGzLrAH1PNcjyAWenMTbbAosGNB6VvmSEgytSER9azLDWCxoJwW7Ke7icmizBMXrzBx9979FfaHxHcrArf3zbeJJJUZPf663zsP",
            "ex_priv": "yprvAHwhK6RbpuS3dgCYHM5jc2ZvEKd7Bi61u9FVhYMpgMSuZS613T1xxQeKTffhrHY79hZ5PsskBjcc6C2V7DrnsMsNaGDaWev3GLRQRgV7hxF",
        },
        "chain_ext": {
            "ex_pub": "ypub6Ynvx7RLNYgWzFGM8aeU43hFNjTh7u5Grrup7Ryu2nKZ1Y8FWKaJZXiUrkJSnMmGVNBoVH1DNDtQ32tR4YFDRSpSUXjjvsiMnCvoPHVWXJP",
            "ex_priv": "yprvAKoaYbtSYB8DmmBt2Z7TgukWphdCiSMRVdzDK3aHUSna8jo6xnG41jQ11ToPk4SQnE5sau6CYK4od9fyz53mK7huW4JskyMMEmixACuyhhr",
        },
        "addresses": [
            "37VucYSaXLCAsxYyAPfbSi9eh4iEcbShgf",
            "3LtMnn87fqUeHBUG414p9CWwnoV6E2pNKS",
            "3B4cvWGR8X6Xs8nvTxVUoMJV77E4f7oaia",
            "38CahkVftQneLonbWtfWxiiaT2fdnzsEAN",
            "37mbeJptxfQC6SNNLJ9a8efCY4BwBh5Kak",
        ],
    },
    # Bitcoin Cash
    {
        "coin": Bip49Coins.BITCOIN_CASH,
        "names": ("Bitcoin Cash", "BCH"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "ypub6XD1Q33WmwYsBXZG237uvDEwq6TyKQ4qJgmsaXBYzmkEeiqqBjNfGbj4eryszYmuthLu7ZSTcXGFQnYachKy7wZbtkW6JQZc6BBzbyWhaeg",
            "ex_priv": "yprvAJDezXWcwZzZy3Unv1auZ5JDH4dUuwLywTrGn8mwSSDFmvWgeC4QioQaoZhU4DSe5FQe6oRKgscT2yifCEL2WMBP143ZaJPhq47uDT9qvRy",
        },
        "chain_ext": {
            "ex_pub": "ypub6YYwTNows1mBXUEKPiwjb59omko4q1L1xizLTHcf9jGnSpqEnMkqsCrvkfdMj7FiJM46V4Z8SXfMEXSddYM5GqUqaRZW2Vq9c1XHDqWh9sk",
            "ex_priv": "yprvAKZb3sH42eCtJz9rHhQjDwD5DixaRYcAbW4jeuD3bPjoa2W6EpSbKQYSuN5Jxs9Wr2swLny7koEo8FyteuemBvK85SG1ihwSo8MM3ZXaSU3",
        },
        "addresses_legacy": {
            "cls": Bip49BitcoinCashMainNet,
            "addresses": [
                "3Bwt6R3hEY2FoB3PMjGUbxKKeHM2tZGMkA",
                "3QAXpUmNCMLRBfBHqZ4X2QtbvKpscf8XF8",
                "3KPaWEXZ8UwhQ8CtxeHA4oUYfnWWHQaaMU",
                "32vT3n51KZ14osw4BMiJX4Zf36x8SAVNgE",
                "3QnzPDR9CLSoTotiDEWffkaGMfqawq8yAu",
            ],
        },
        "addresses": [
            "bitcoincash:ppcgpaskfreturznhcjdvmeunscff6ufu5kt8jx4qt",
            "bitcoincash:prmgdapr6x6dk8w3qhlmaxsavsq3jlvwgvfc2s2kqy",
            "bitcoincash:prpzt72nhvhsum6lf8u0udu7knqs003ewuldavgk6g",
            "bitcoincash:pqxc9qdqfjpdrs7mxvvxk55kxxkz328w2v56p07svx",
            "bitcoincash:pr7kcxn8jz75luswpdut06f2ll2am0dwzumrq7wyj4",
        ],
    },
    # BitcoinSV
    {
        "coin": Bip49Coins.BITCOIN_SV,
        "names": ("BitcoinSV", "BSV"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "ypub6XeYcPUzGwK9dPtY7Vkb6P1LgeaT6GEoqoPW1xAciZ6HKqBYJjL5pGABtEQizEw2jkGSHyYfc6DGCCR2ZnKDoSCD4qtp56uy4jtC6N1XyV3",
            "ex_priv": "yprvAJfCCsx6SZkrQup51UDajF4c8cjxgoWxUaTuDZm1ADZJT2rPmC1qGTqi2vXLRzFRmdGpKoosLeZZfJnQ4wF37o8prmUqNqSdyhjM63KXWwT",
        },
        "chain_ext": {
            "ex_pub": "ypub6Zc68W6DVfKx49xWgrb6HCCMHkzvSEhewiZDdBU97hzD5rWDaoHmBokhdN36a3uFDyxSTsQEiYxJfNQ3rromoeChXB4pPMBjamhQLwVxP7i",
            "ex_priv": "yprvALcjizZKfHmeqft3aq45v4FcjjAS2myoaVdcpo4XZNTED4B53FyWe1SDn69kJmPDajxdGyxsM2gSJh4yti5PRwkbVBokibu9EwUeGDXvEEo",
        },
        "addresses": [
            "3DBZLGqauQhKR7CBM8fhi9gZANUp9EcA2F",
            "3BdMLsRbEf6rFKpQzFaNH5pRfmYdEXLZUN",
            "37gow1C7tEGUGh2LUusj5ZQANynvr82BdY",
            "3Q1ZdpKhecKkbon3UPUDXeAW16WdtXp9Rj",
            "36tsqGBbuGkB7NLFXAQ6JPwu2bQjszFgYH",
        ],
    },
    # Dash
    {
        "coin": Bip49Coins.DASH,
        "names": ("Dash", "DASH"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
        "wif_master": "7qjXbkdi3WJ1SRu6pVnUcqQzk9RimTJB3cCUrw9V1HGFoqaB948",
        "account": {
            "ex_pub": "ypub6XfMrvptRS5DPon5Ncq6w9AKZRde7w4fPxMCmBioQFUZ8wJNY5gQy3KfG8wL6L111jDBVA4opQfq9J82uCGpaSrcmxKkwoZuRXpS1HdgjP8",
            "ex_priv": "yprvAJg1TRHzb4WvBKhcGbJ6a1Db1Po9iULp2jRbxoKBquwaG8yDzYNARF1BQpkNqZHNUc5b3tT1Cgy7pA7L9SPH4UZrdF4cKvmHGjzWWP9VjZ4",
        },
        "chain_ext": {
            "ex_pub": "ypub6ZQqzbH8PeLbg36yPKPeVquiSiZJZtRX2cab4HwvrUhwAC8FYJLznZoedn6jg71geRfVwEZvJ2agVgGjaMoSH5nMg83ZtnxHxcNAFJeg5MB",
            "ex_priv": "yprvALRVb5kEZGnJTZ2WHHre8hxytgipARhffPezFuYKJ9AxHPo6zm2kEmVAnWu76MeK7LXZ54wKP1cFaXB4XzdcE79QfHoGB1n7oixVVxpfn2w",
        },
        "addresses": [
            "7bG9yjGpzvKvffhHkCkDRsLLZ5iJxyDcX7",
            "7WeB1MquavUF2AdQwiu5kiHNtSMA8RWKka",
            "7cmSUivNKPsH6J6MmsMfYJrQS2p3GZzSm7",
            "7mG6G6PEiS5qFxFXCvGw8E56tVizUP6zyW",
            "7iWwAjC3qd3uBPkkAZsLjcjtQNP8maaBfB",
        ],
    },
    # Dogecoin
    {
        "coin": Bip49Coins.DOGECOIN,
        "names": ("Dogecoin", "DOGE"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "dgpv51eADS3spNJh98bWAfYnAW8K1gMy86HKmH1dpyT8kLsUKBqssT3jsLLFWyK4zbruL51UjejFDzrFzBcwjjA57rSv6D2978QigKG4xbCfJV6",
        "wif_master": "6JKHV5zEqwBbEhAf7qEWk5qNcu6gs4XtrCwXe3WFH8xR7BcSCbr",
        "account": {
            "ex_pub": "dgub8rgNmp9pEE4teMZS2ooaWnN8PC3XwAdtjmPnKiBER8ZdTfNKQZ58dVc11NrMVNWJvaZBAzqjZiyNpJ4rNpJuniwHdbVDgBBUAXDsGRmjDBo",
            "ex_priv": "dgpv57oMS8Dvg5QZMVuGzaBcPvpXvdfu73cBTAcaAbnFd1WcucFcznXMzCCMS8Tvjp3E5HzY3PKoXkwURyjUm6DhhychRQBXQpf3xFCEYMThwj6",
        },
        "chain_ext": {
            "ex_pub": "dgub8t8oNKJryeExnXMktuGVuYytUUekFvyLRefyodU6F9njRD8twMxy8raotyHhxpKRfSfJoKBg5EPB4e5yWHVnfGs85tdkTnBqrgSnPT7S5Gr",
            "ex_priv": "dgpv59Fn2dNyRVadVfhbrfeXnhSJ1vH7Rowd93tmeX57T2jisA2CXbRCVZBAKmrifgszhCPcyNmS16Azse2Tm5DmwEMTrvoH12n4eR8FYkeA4P9",
        },
        "addresses": [
            "9xzLEaWbi3eVzRzv6YdGcXbXQMHydntSq7",
            "ABxNotFnVaS85dcyXY76d2Fbr12qph6UTy",
            "9wrexodMKHBmtjnf2XU7eXSThsynMzGE6A",
            "AAqxBLjK4FdCehqRc12ZR4b7L2KsYVF8zc",
            "A4wfuH4RwkkrwpRvB8zqaq1PA2rRXhap9T",
        ],
    },
    # Litecoin
    {
        "coin": Bip49Coins.LITECOIN,
        "names": ("Litecoin", "LTC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
        "ex_master_cls": Bip49LitecoinMainNet,
        "ex_master_alt": "Mtpv7L6Q8tPadPv8jY9X5qf2qdkW39MvWo7VyJDVMGib3ZZjW9p4vJh2GghWrqCpo55Xkf27EqvAJRtVCAEU5EX5qaF5kwjWrd9KkoeRrSjowo7",
        "wif_master": "6uJgfG4pBbMffTdMSGQVurdK6xBcZjhf1iDU2jtPAw5PzRdhx9m",
        "account": {
            "ex_pub": "ypub6WZ2nNciqS7sCCFCH64AswfvBu4pLXTdDQcvTkrSFyEbashNb6vEJwXTCB7axKdR4TSbNYTqnU7S6sYPs9afBYqTytiTdjzmcDVRuYcrtso",
            "ex_priv": "yprvAHZgNs5q14ZZyiAjB4XAWojBdsEKw4jmrBhKfNSphdhci5NE3Zbym9CyLvKLFLNSSV99Pty82vKR6bQ5tAD9nWdqwDETVon86NdDftk4Qra",
            "ex_pub_alt": "Mtub2rz9F1pkisRsSZX8sa4Ajon9GhPP6JymLgpuHqbYdU5JKFLBF7Qy8b1tZ3dccj2fefrAxfrPdVkpCxuWn3g72UctH2bvJRkp6iFmp8aLeRZ",
            "ex_priv_alt": "Mtpv7RooeEQDUitupgpJcxZnfDwvq8hC24R7GAiscrqFhHHhit96vCNY7yudJgrM841dMbiRUQceC12566XAHHC8Rd1BtnBdokq9tmF7jLLvUdh",
        },
        "chain_ext": {
            "ex_pub": "ypub6ZaxXJpR4G9iauzhwLi2gV4JxjLqjEHk4XWqK76M8V1LhNZYGj7LiyT7VF8U9wtAvGjnfe5LfEkPKxE3i4M76vtoW38KRU4VKBoiABHGuWt",
            "ex_priv": "yprvALbc7oHXDtbRNRvEqKB2KM7aQhWMKmZthJbEWigja9UMpaEPjBo6BB8ddyCopnFeun5DbkMyd2QkNo6YAPNxRFBJvwbTi2gR3Qy4wiu5vaz",
            "ex_pub_alt": "Mtub2v24yx2SwhTiqHGeXpi2YMAY3XfQV1otBoip9BqTVyr3RkCLvjc5YcwYr7eVpMHRWV9NFmTtWGPmS3bAcxSYwrgDoB1n69pXoga44kARLke",
            "ex_priv_alt": "Mtpv7UqjPAbuhYvmDQZpHDDeTmLKbxyDQmFE7HcnUD5AZo4SqP1GbpZeY1qHbjjphVtqpteVgG1Vn77QNJDcZWMw4MYetWYe1yjSqoay1B25d5b",
        },
        "addresses_depr": {
            "cls": Bip49LitecoinMainNet,
            "addresses": [
                "31jkZShyVAMkVz5cQpu4qp73DrAXtFP3rZ",
                "32pqrhqZauSgZoFj9wK7pQKg5AHXYYvYCf",
                "3CT62pXvCKF23eRgXQHXxoywFHf2ZGFgLd",
                "36ELz7iJTw3wuMG9zrATo3Bk7CrXeMXBig",
                "3MDXS6KsscyBM3DKQxzUfSFXpQJtdJT8na",
            ],
        },
        "addresses": [
            "M7wtsL7wSHDBJVMWWhtQfTMSYYkyooAAXM",
            "M92zAbFXY2J7NJXdFpJTe3a5PrsyZQhKZK",
            "MJfELhwt9S6Sr9hadHGsnTELZzFUVjMrFc",
            "MCSVJ18GR3uNhrY46j9ocgS9RuSygeGvC5",
            "MTRfjyjqpjpc9YVDWqypV5Vw96uLZcUajY",
        ],
    },
    # Zcash
    {
        "coin": Bip49Coins.ZCASH,
        "names": ("Zcash", "ZEC"),
        "is_testnet": False,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
        "wif_master": "5HzxC8XHHAtoC5jVvScY8Tr99Ud9MwFdF2pJKYsMTUknJZEurYr",
        "account": {
            "ex_pub": "ypub6Xsh1czMqaFJxfbfwoHocPLcZQBa8N94UuGziygg76pgGmHVSknYmL5EHLfdQsFomh943UhHap83thb49Y64rWtV8yePRVRPzBEPbZHrCcF",
            "ex_priv": "yprvAJtLc7TU1Ch1kBXCqmkoFFPt1NM5iuRD7gMPvbH4YmHhPxxLuDUJDXkkS5ksuwpAaEKsbUZkcaYnXiLwQQ63M6LJ369EZBKKqa2n8EMydrA",
        },
        "chain_ext": {
            "ex_pub": "ypub6Z3hN8BQqSpFwUW9SfgiVttz9tJUqet4MUvFKM7d79ZaCQFDzR6iACFHJ6yKewdS6Ku1EVzQAVkQjodFc2eqv7X1anjDXZHDVPhMFxedcsw",
            "ex_priv": "yprvAL4LxceX15FxizRgLe9i8kxFbrTzSCACzFzeWxi1Yp2bKbv5SsnTcPvoSr4QNZ8X5YMQ8v2rqEkdedTVEHynsZMmMQuws6V8gLi3wErATkQ",
        },
        "addresses": [
            "t3WrebyH3US8WvEdS9yNEWJZGApy3beYdeL",
            "t3VTbz8F6Qrk63S21i2gLcHi7Hj2uTipcpb",
            "t3XYvamZQDF1QNjnvnqrnzm4uChnyS4dwNq",
            "t3gECjqstPbYHoVrircZqFwxpPk95rb4Ffc",
            "t3cYvaYasuzDan2FpZbY8k6bc7MQfv54zdZ",
        ],
    },

    # Bitcoin test net
    {
        "coin": Bip49Coins.BITCOIN_TESTNET,
        "names": ("Bitcoin TestNet", "BTC"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "uprv8tXDerPXZ1QsVNjUJWTurs9kA1KGfKUAts74GCkcXtU8GwnH33GDRbNJpEqTvipfCyycARtQJhmdfWf8oKt41X9LL1zeD2pLsWmxEk3VAwd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "upub5EFU65HtV5TeiSHmZZm7FUffBGy8UKeqp7vw43jYbvZPpoVsgU93oac7Wk3u6moKegAEWtGNF8DehrnHtv21XXEMYRUocHqguyjknFHYfgY",
            "ex_priv": "uprv91G7gZkzehuMVxDJTYE6tLivdF8e4rvzSu1LFfKw3b2Qx1Aj8vpoFnHdfUZ3hmi9jsvPifmZ24RTN2KhwB8BfMLTVqaBReibyaFFcTP1s9n",
        },
        "chain_ext": {
            "ex_pub": "upub5F7X3ZAt1HsUyFLTFU9vhKeGULy77aDoJFhscvGBV91tm2mzQ5egFGpeP4nGskwERwbU48g14qREqXJ388X8XBiaLm7PWwk3S45Fe3WAvdK",
            "ex_priv": "uprv928Ae3dzAvKBkmFz9ScvLBhXvK8ci7Vww2nGpXrZvoUutESqrYLRhUWAXpK5acXh517npKuCpJ7NXaoWnLs1dLB9w3MHe3KNUm7hPENqMzt",
        },
        "addresses": [
            "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
            "2N55m54k8vr95ggehfUcNkdbUuQvaqG2GxK",
            "2N9LKph9TKtv1WLDfaUJp4D8EKwsyASYnGX",
            "2MyVXDzGJgATSdkhKHWvStpBoGEZb1fwjha",
            "2MuKeQzUHhUQWUZgx5AuNWoQ7YWx6vsXxrv",
        ],
    },
    # Bitcoin Cash test net
    {
        "coin": Bip49Coins.BITCOIN_CASH_TESTNET,
        "names": ("Bitcoin Cash TestNet", "BCH"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "uprv8tXDerPXZ1QsVNjUJWTurs9kA1KGfKUAts74GCkcXtU8GwnH33GDRbNJpEqTvipfCyycARtQJhmdfWf8oKt41X9LL1zeD2pLsWmxEk3VAwd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "upub5EFU65HtV5TeiSHmZZm7FUffBGy8UKeqp7vw43jYbvZPpoVsgU93oac7Wk3u6moKegAEWtGNF8DehrnHtv21XXEMYRUocHqguyjknFHYfgY",
            "ex_priv": "uprv91G7gZkzehuMVxDJTYE6tLivdF8e4rvzSu1LFfKw3b2Qx1Aj8vpoFnHdfUZ3hmi9jsvPifmZ24RTN2KhwB8BfMLTVqaBReibyaFFcTP1s9n",
        },
        "chain_ext": {
            "ex_pub": "upub5F7X3ZAt1HsUyFLTFU9vhKeGULy77aDoJFhscvGBV91tm2mzQ5egFGpeP4nGskwERwbU48g14qREqXJ388X8XBiaLm7PWwk3S45Fe3WAvdK",
            "ex_priv": "uprv928Ae3dzAvKBkmFz9ScvLBhXvK8ci7Vww2nGpXrZvoUutESqrYLRhUWAXpK5acXh517npKuCpJ7NXaoWnLs1dLB9w3MHe3KNUm7hPENqMzt",
        },
        "addresses_legacy": {
            "cls": Bip49BitcoinCashTestNet,
            "addresses": [
                "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
                "2N55m54k8vr95ggehfUcNkdbUuQvaqG2GxK",
                "2N9LKph9TKtv1WLDfaUJp4D8EKwsyASYnGX",
                "2MyVXDzGJgATSdkhKHWvStpBoGEZb1fwjha",
                "2MuKeQzUHhUQWUZgx5AuNWoQ7YWx6vsXxrv",
            ],
        },
        "addresses": [
            "bchtest:pqeke2snuz9evzq2x26asxx4nd9t8vm8ggd9ngnmpx",
            "bchtest:pzqawj7d8qxqtau3686vsxph2ew7exerfg60w5xcq0",
            "bchtest:pzc83an7qwjcq6ex952nmkhha4wvruzg7un3fz27x0",
            "bchtest:ppzgfa67kf0sng9rw7nl84m5qrurjcm3lqv4kqu2pa",
            "bchtest:pqtvvn3z9v5gwza4q3qrrde6s6tsc3k5vy3wpqejv9",
        ],
    },
    # BitcoinSV test net
    {
        "coin": Bip49Coins.BITCOIN_SV_TESTNET,
        "names": ("BitcoinSV TestNet", "BSV"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "uprv8tXDerPXZ1QsVNjUJWTurs9kA1KGfKUAts74GCkcXtU8GwnH33GDRbNJpEqTvipfCyycARtQJhmdfWf8oKt41X9LL1zeD2pLsWmxEk3VAwd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "upub5EFU65HtV5TeiSHmZZm7FUffBGy8UKeqp7vw43jYbvZPpoVsgU93oac7Wk3u6moKegAEWtGNF8DehrnHtv21XXEMYRUocHqguyjknFHYfgY",
            "ex_priv": "uprv91G7gZkzehuMVxDJTYE6tLivdF8e4rvzSu1LFfKw3b2Qx1Aj8vpoFnHdfUZ3hmi9jsvPifmZ24RTN2KhwB8BfMLTVqaBReibyaFFcTP1s9n",
        },
        "chain_ext": {
            "ex_pub": "upub5F7X3ZAt1HsUyFLTFU9vhKeGULy77aDoJFhscvGBV91tm2mzQ5egFGpeP4nGskwERwbU48g14qREqXJ388X8XBiaLm7PWwk3S45Fe3WAvdK",
            "ex_priv": "uprv928Ae3dzAvKBkmFz9ScvLBhXvK8ci7Vww2nGpXrZvoUutESqrYLRhUWAXpK5acXh517npKuCpJ7NXaoWnLs1dLB9w3MHe3KNUm7hPENqMzt",
        },
        "addresses": [
            "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
            "2N55m54k8vr95ggehfUcNkdbUuQvaqG2GxK",
            "2N9LKph9TKtv1WLDfaUJp4D8EKwsyASYnGX",
            "2MyVXDzGJgATSdkhKHWvStpBoGEZb1fwjha",
            "2MuKeQzUHhUQWUZgx5AuNWoQ7YWx6vsXxrv",
        ],
    },
    # Dash test net
    {
        "coin": Bip49Coins.DASH_TESTNET,
        "names": ("Dash TestNet", "DASH"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "uprv8tXDerPXZ1QsVNjUJWTurs9kA1KGfKUAts74GCkcXtU8GwnH33GDRbNJpEqTvipfCyycARtQJhmdfWf8oKt41X9LL1zeD2pLsWmxEk3VAwd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "upub5EFU65HtV5TeiSHmZZm7FUffBGy8UKeqp7vw43jYbvZPpoVsgU93oac7Wk3u6moKegAEWtGNF8DehrnHtv21XXEMYRUocHqguyjknFHYfgY",
            "ex_priv": "uprv91G7gZkzehuMVxDJTYE6tLivdF8e4rvzSu1LFfKw3b2Qx1Aj8vpoFnHdfUZ3hmi9jsvPifmZ24RTN2KhwB8BfMLTVqaBReibyaFFcTP1s9n",
        },
        "chain_ext": {
            "ex_pub": "upub5F7X3ZAt1HsUyFLTFU9vhKeGULy77aDoJFhscvGBV91tm2mzQ5egFGpeP4nGskwERwbU48g14qREqXJ388X8XBiaLm7PWwk3S45Fe3WAvdK",
            "ex_priv": "uprv928Ae3dzAvKBkmFz9ScvLBhXvK8ci7Vww2nGpXrZvoUutESqrYLRhUWAXpK5acXh517npKuCpJ7NXaoWnLs1dLB9w3MHe3KNUm7hPENqMzt",
        },
        "addresses": [
            "8j7NLynPotJD3x4MHGemN36XPSLBKr6cYn",
            "8rFznqz9Fv7zvxyNLDexvSREW8KcuNUAmR",
            "8vWZYUPTextvkcYLFDMQE1wyvfH1FLALz4",
            "8kfkwmWK1ESMt31yxFy34d1Yrwxd8sUL6R",
            "8gVt8miJ2YPRir1cjuwxgcDs9EM98qVtNJ",
        ],
    },
    # Dogecoin test net
    {
        "coin": Bip49Coins.DOGECOIN_TESTNET,
        "names": ("Dogecoin TestNet", "DOGE"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "tgpv1aRS3XcGkbKXDbWwtRDZFYqUaCsEVXgkHt9m5mEjaWGGA18gaU1qZatwWCmjT66o2CmSNJmXkAvG29sYFXrz11WEaLwVrckr5LMkUrVeQmp",
        "wif_master": "95f58LEtaFKPRnQCMwkSuhbedNDHpWUaAFpTtUbenXWc4oSGq5N",
        "account": {
            "ex_pub": "tgub5S3MQvudQSULX2ew5gbiPsxFW14EjCBH8WqFELD2tKsZP7dq688C35DuvftJvxQSZTnVqrndkHZFQRBc8YchoDC2fGtwdnRPUmJzBnYBr3w",
            "ex_priv": "tgpv1hAL5EyjrHp1EAzn3SykH2Qf3Sgbu59Zqv435Dp46CpYq4X8gMaRPmpGMSVKE8zHZ6iDvYegTXa5ifY7PP77eqhMkAX35Ef7BPq3refCiHE",
        },
        "chain_ext": {
            "ex_pub": "tgub5SuQNQncvetAmqhcmazXqivro54DNSkEcecBoCjfmYL4KLuwojdpUmSSnzcghwYMLjDjP7CGZzkqY5hMMm7pnsgFTcXXYSKjzqeV3Y1ww2o",
            "ex_priv": "tgpv1i2P2irjNWDqUz3TjMNZisPGLWgaYKiXL3pye6LgyRH3mHoFPy63qU2oDnFM6yoptDud2CnLFmFztE1vEYqwcpY4BNJ9HdFsgahVdMq7Rbv",
        },
        "addresses": [
            "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
            "2N55m54k8vr95ggehfUcNkdbUuQvaqG2GxK",
            "2N9LKph9TKtv1WLDfaUJp4D8EKwsyASYnGX",
            "2MyVXDzGJgATSdkhKHWvStpBoGEZb1fwjha",
            "2MuKeQzUHhUQWUZgx5AuNWoQ7YWx6vsXxrv",
        ],
    },
    # Litecoin test net
    {
        "coin": Bip49Coins.LITECOIN_TESTNET,
        "names": ("Litecoin TestNet", "LTC"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "ttpv96BtqegdxXceQk8r9KuoG5yiMACLxANu9hh98NpMwpzcCa8XfrJ7uwnRBMzsE5n9y2exs7VQBBdHNiJ66BrDUWE28WoexgbFVRkRc2abBR9",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "ttub4e78U9UeJNe63CKWYyANN9P7FxhuRxQNnFhSnp9yPiLVTzi46eu8kjp1GyCKBpWt5pftjBcPoZ22XefHhZ6eCKj7gVE6igKcDzrfzcGAGJ8",
            "ex_priv": "ttpv9CvnsN474E78RKcgJMfzHZYtpQ1iMhqihjbR7qPgTXYtsdWymjrhk8hk2biT18feVvbkRMNYtYH75DxfE36M8LR9JLPCBJVWbVDiynpG7JX",
        },
        "chain_ext": {
            "ex_pub": "ttub4eyBRdMdpb3vJ1NCEsZBozMiZ2ht5CyLGPUPMggcGvnzQDzApGQmCS2Y9Hvgxoens678GS22dGDcfKB2vmbmBzDLUprgdLDxk5CArPc8iLo",
            "ex_priv": "ttpv9Dnqpqw6aSWxg8fMzG4ojQXW7U1gzxQgBsNMghvKLk1Poro6VMNLBpvGtwUUsyVBq3o9X1WCgmy2EnSU5CqB6KFqjYAJPh6H6g6AkX8gtNN",
        },
        "addresses_depr": {
            "cls": Bip49LitecoinTestNet,
            "addresses": [
                "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
                "2N55m54k8vr95ggehfUcNkdbUuQvaqG2GxK",
                "2N9LKph9TKtv1WLDfaUJp4D8EKwsyASYnGX",
                "2MyVXDzGJgATSdkhKHWvStpBoGEZb1fwjha",
                "2MuKeQzUHhUQWUZgx5AuNWoQ7YWx6vsXxrv",
            ],
        },
        "addresses": [
            "QRHtkDQdVvNNwrVjEdeCGviCw7Ny3SNNiA",
            "QYSXC5cNwxCApsQkHaePqL2v3oNQftuzPX",
            "Qch5wi1hLzy6eWyiCaLq8uZfULKnxc7HnY",
            "QSrHM18YhGWXmwTMucxTyWdEQd1QxHjMPy",
            "QNgQY1LXiaTbckSzhGwPbVqYguPvq7FAU5",
        ],
    },
    # Zcash test net
    {
        "coin": Bip49Coins.ZCASH_TESTNET,
        "names": ("Zcash TestNet", "ZEC"),
        "is_testnet": True,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "ex_master": "uprv8tXDerPXZ1QsVNjUJWTurs9kA1KGfKUAts74GCkcXtU8GwnH33GDRbNJpEqTvipfCyycARtQJhmdfWf8oKt41X9LL1zeD2pLsWmxEk3VAwd",
        "wif_master": "91mamsLpsPxwA9EnYnWT14Q6o8yrX6npaygFQBDroDVq5dZG3q3",
        "account": {
            "ex_pub": "upub5EFU65HtV5TeiSHmZZm7FUffBGy8UKeqp7vw43jYbvZPpoVsgU93oac7Wk3u6moKegAEWtGNF8DehrnHtv21XXEMYRUocHqguyjknFHYfgY",
            "ex_priv": "uprv91G7gZkzehuMVxDJTYE6tLivdF8e4rvzSu1LFfKw3b2Qx1Aj8vpoFnHdfUZ3hmi9jsvPifmZ24RTN2KhwB8BfMLTVqaBReibyaFFcTP1s9n",
        },
        "chain_ext": {
            "ex_pub": "upub5F7X3ZAt1HsUyFLTFU9vhKeGULy77aDoJFhscvGBV91tm2mzQ5egFGpeP4nGskwERwbU48g14qREqXJ388X8XBiaLm7PWwk3S45Fe3WAvdK",
            "ex_priv": "uprv928Ae3dzAvKBkmFz9ScvLBhXvK8ci7Vww2nGpXrZvoUutESqrYLRhUWAXpK5acXh517npKuCpJ7NXaoWnLs1dLB9w3MHe3KNUm7hPENqMzt",
        },
        "addresses": [
            "t2BEicU8di9CukDjmoZoUFia4wX9E2KsEEH",
            "t2JPM4LLPAB2hdEenrWofp7tn4D8fcF5okX",
            "t2NduoxjhZDodStDkmWW77hRXUk63xL54Km",
            "t2Co7DFrYuVM4aJhQUZ7jxJV6R2mfqBNZEj",
            "t28dEQG4XvoJ8R7h3GD6faHhQhKABoQVipY",
        ],
    },
]

# Tests for default path derivation
TEST_VECT_DEFAULT_PATH = [
    # Bitcoin
    {
        "coin": Bip49Coins.BITCOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "37VucYSaXLCAsxYyAPfbSi9eh4iEcbShgf",
    },
    # Bitcoin Cash
    {
        "coin": Bip49Coins.BITCOIN_CASH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "bitcoincash:ppcgpaskfreturznhcjdvmeunscff6ufu5kt8jx4qt",
    },
    # BitcoinSV
    {
        "coin": Bip49Coins.BITCOIN_SV,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "3DBZLGqauQhKR7CBM8fhi9gZANUp9EcA2F",
    },
    # Dash
    {
        "coin": Bip49Coins.DASH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "7bG9yjGpzvKvffhHkCkDRsLLZ5iJxyDcX7",
    },
    # Dogecoin
    {
        "coin": Bip49Coins.DOGECOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "9xzLEaWbi3eVzRzv6YdGcXbXQMHydntSq7",
    },
    # Litecoin
    {
        "coin": Bip49Coins.LITECOIN,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "M7wtsL7wSHDBJVMWWhtQfTMSYYkyooAAXM",
    },
    # Zcash
    {
        "coin": Bip49Coins.ZCASH,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "t3WrebyH3US8WvEdS9yNEWJZGApy3beYdeL",
    },

    # Bitcoin test net
    {
        "coin": Bip49Coins.BITCOIN_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
    },
    # Bitcoin Cash test net
    {
        "coin": Bip49Coins.BITCOIN_CASH_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "bchtest:pqeke2snuz9evzq2x26asxx4nd9t8vm8ggd9ngnmpx",
    },
    # BitcoinSV test net
    {
        "coin": Bip49Coins.BITCOIN_SV_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
    },
    # Dash test net
    {
        "coin": Bip49Coins.DASH_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "8j7NLynPotJD3x4MHGemN36XPSLBKr6cYn",
    },
    # Dogecoin test net
    {
        "coin": Bip49Coins.DOGECOIN_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "2Mww8dCYPUpKHofjgcXcBCEGmniw9CoaiD2",
    },
    # Litecoin test net
    {
        "coin": Bip49Coins.LITECOIN_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "QRHtkDQdVvNNwrVjEdeCGviCw7Ny3SNNiA",
    },
    # Zcash test net
    {
        "coin": Bip49Coins.ZCASH_TESTNET,
        "seed": b"5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
        "default_address": "t2BEicU8di9CukDjmoZoUFia4wX9E2KsEEH",
    },
]

# Tests for different key formats
TEST_VECT_KEY_FORMATS = {
    "coin": Bip49Coins.BITCOIN,
    "seed": "5eb00bbddcf069084889a8ab9155568165f5c453ccb85e70811aaed6f6da5fc19a5ac40b389cd370d086206dec8aa6c43daea6690f20ad3d8d48b2d2ce9e38e4",
    "ex_priv": "yprvABrGsX5C9jantZVwdwcQhDXkqsu4RoSAZKBwPnLA3uyeVM3C3fvTuqzru4fovMSLqYSqALGe9MBqCf7Pg7Y7CTsjoNnLYg6HxR2Xo44NX7E",
    "raw_priv": "1837c1be8e2995ec11cda2b066151be2cfb48adf9e47b151d46adab3a21cdf67",
    "ex_pub": "ypub6QqdH2c5z79673aQjy9R4MUVPujYqGA1vY7YCAjmcFWdN9NLbDEiTeKLkP7ECKzds8NrfRnX8zGyZtXA9QFT7mTs8vi1PVeq8YQpcNKUMvQ",
    "raw_compr_pub": "03d902f35f560e0470c63313c7369168d9d7df2d49bf295fd9fb7cb109ccee0494",
    "raw_uncompr_pub": "04d902f35f560e0470c63313c7369168d9d7df2d49bf295fd9fb7cb109ccee04947d000a1345d3845dd83b4c5814f876c918305b598f066c958fad972bf59f2ec7",
}

# Tests for extended keys with valid and invalid depths
TEST_VECT_EX_KEY_DEPTHS = {
    # Private key with depth 5 is fine
    "ex_priv_5": "yprvANXJmbMRf4r2NVPx9K5sDCnhND9fjZqWtBthvRJrN2XCU7H9KJmMLP2LnsgLbhdoaNcD89Fw7zktymVkW6eVcX9MKHpeAkEd94Hm9nWKWVw",
    # Private key with depth 6 shall raise an exception
    "ex_priv_6": "yprvAQyUkCuvz2wzk4dUqEyQFXe4sUeEkCSnmz13PAxrNQZrorMkkAw8yVaBcQU3MUp17y9NSYn1ugT6QdETtS85GCsxUi33EVk1EynV37De7V5",
    # Public key with depth 2 shall raise an exception
    "ex_pub_2": "ypub6VVTyCM8KkwKjEfeUaygfZCPb3Xxx7DfhCFRndbDREZfWkQg5Q5p3pHMDCoiTfPKwejRBnbhr4SocDthWYoJv4VU1Xfje7FE7sERi3CsDRt",
    # Public key with depth 3 is fine
    "ex_pub_3": "ypub6Ww3ibxVfGzLrAH1PNcjyAWenMTbbAosGNB6VvmSEgytSER9azLDWCxoJwW7Ke7icmizBMXrzBx9979FfaHxHcrArf3zbeJJJUZPf663zsP",
    # Public key with depth 5 is fine
    "ex_pub_5": "ypub6bWfB6tKVSQKayURFLcsaLjRvEzA92ZNFQpJioiTvN4BLucHrr5btBLpeBDjuV2mGb2wXWL1taoBNWf9xNgjHrPWkhSxxfrDGiciopL6N6E",
    # Public key with depth 6 shall raise an exception
    "ex_pub_6": "ypub6dxq9iSppQWHxYhwwGWQcfaoRWUj9fAe9CveBZNTvk6qgeguHiFPXHtfTi5oPFEiVe7vH7mUdUrjeQpAbxPxZ1nXhDDvx4xQw5kREC9vNE1",
}


#
# Tests
#
class Bip49Tests(unittest.TestCase):
    # Test specification name
    def test_spec_name(self):
        self.assertEqual(Bip49.SpecName(), "BIP-0049")

    # Run all tests in test vector using FromSeed for construction
    def test_from_seed(self):
        Bip44BaseTestHelper.test_from_seed(self, Bip49, TEST_VECT)

    # Run all tests in test vector using FromExtendedKey for construction
    def test_from_ex_key(self):
        Bip44BaseTestHelper.test_from_ex_key(self, Bip49, TEST_VECT)

    # Run all tests in test vector using FromPrivateKey for construction
    def test_from_priv_key(self):
        Bip44BaseTestHelper.test_from_priv_key(self, Bip49, TEST_VECT)

    # Test default path derivation
    def test_default_path_derivation(self):
        Bip44BaseTestHelper.test_default_path_derivation(self, Bip49, TEST_VECT_DEFAULT_PATH)

    # Test for IsLevel method
    def test_is_level(self):
        Bip44BaseTestHelper.test_is_level(self, Bip49, Bip49Coins, TEST_SEED)

    # Test different key formats
    def test_key_formats(self):
        Bip44BaseTestHelper.test_key_formats(self, Bip49, TEST_VECT_KEY_FORMATS)

    # Test construction from extended keys with valid and invalid depths
    def test_from_ex_key_depth(self):
        Bip44BaseTestHelper.test_from_ex_key_depth(self, Bip49, Bip49Coins, TEST_VECT_EX_KEY_DEPTHS)

    # Test type error during construction
    def test_type_error(self):
        Bip44BaseTestHelper.test_type_error(self, Bip49, [Bip44Coins, Bip84Coins])

    # Test invalid path derivations
    def test_invalid_derivations(self):
        Bip44BaseTestHelper.test_invalid_derivations(self, Bip49, Bip49Coins, TEST_SEED)
